﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/sagemaker/SageMakerRequest.h>
#include <aws/sagemaker/SageMaker_EXPORTS.h>
#include <aws/sagemaker/model/Device.h>

#include <utility>

namespace Aws {
namespace SageMaker {
namespace Model {

/**
 */
class UpdateDevicesRequest : public SageMakerRequest {
 public:
  AWS_SAGEMAKER_API UpdateDevicesRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateDevices"; }

  AWS_SAGEMAKER_API Aws::String SerializePayload() const override;

  AWS_SAGEMAKER_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the fleet the devices belong to.</p>
   */
  inline const Aws::String& GetDeviceFleetName() const { return m_deviceFleetName; }
  inline bool DeviceFleetNameHasBeenSet() const { return m_deviceFleetNameHasBeenSet; }
  template <typename DeviceFleetNameT = Aws::String>
  void SetDeviceFleetName(DeviceFleetNameT&& value) {
    m_deviceFleetNameHasBeenSet = true;
    m_deviceFleetName = std::forward<DeviceFleetNameT>(value);
  }
  template <typename DeviceFleetNameT = Aws::String>
  UpdateDevicesRequest& WithDeviceFleetName(DeviceFleetNameT&& value) {
    SetDeviceFleetName(std::forward<DeviceFleetNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>List of devices to register with Edge Manager agent.</p>
   */
  inline const Aws::Vector<Device>& GetDevices() const { return m_devices; }
  inline bool DevicesHasBeenSet() const { return m_devicesHasBeenSet; }
  template <typename DevicesT = Aws::Vector<Device>>
  void SetDevices(DevicesT&& value) {
    m_devicesHasBeenSet = true;
    m_devices = std::forward<DevicesT>(value);
  }
  template <typename DevicesT = Aws::Vector<Device>>
  UpdateDevicesRequest& WithDevices(DevicesT&& value) {
    SetDevices(std::forward<DevicesT>(value));
    return *this;
  }
  template <typename DevicesT = Device>
  UpdateDevicesRequest& AddDevices(DevicesT&& value) {
    m_devicesHasBeenSet = true;
    m_devices.emplace_back(std::forward<DevicesT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_deviceFleetName;

  Aws::Vector<Device> m_devices;
  bool m_deviceFleetNameHasBeenSet = false;
  bool m_devicesHasBeenSet = false;
};

}  // namespace Model
}  // namespace SageMaker
}  // namespace Aws
