﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/neptunedata/NeptunedataRequest.h>
#include <aws/neptunedata/Neptunedata_EXPORTS.h>

#include <utility>

namespace Aws {
namespace neptunedata {
namespace Model {

/**
 */
class StartMLDataProcessingJobRequest : public NeptunedataRequest {
 public:
  AWS_NEPTUNEDATA_API StartMLDataProcessingJobRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "StartMLDataProcessingJob"; }

  AWS_NEPTUNEDATA_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>A unique identifier for the new job. The default is an autogenerated
   * UUID.</p>
   */
  inline const Aws::String& GetId() const { return m_id; }
  inline bool IdHasBeenSet() const { return m_idHasBeenSet; }
  template <typename IdT = Aws::String>
  void SetId(IdT&& value) {
    m_idHasBeenSet = true;
    m_id = std::forward<IdT>(value);
  }
  template <typename IdT = Aws::String>
  StartMLDataProcessingJobRequest& WithId(IdT&& value) {
    SetId(std::forward<IdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The job ID of a completed data processing job run on an earlier version of
   * the data.</p>
   */
  inline const Aws::String& GetPreviousDataProcessingJobId() const { return m_previousDataProcessingJobId; }
  inline bool PreviousDataProcessingJobIdHasBeenSet() const { return m_previousDataProcessingJobIdHasBeenSet; }
  template <typename PreviousDataProcessingJobIdT = Aws::String>
  void SetPreviousDataProcessingJobId(PreviousDataProcessingJobIdT&& value) {
    m_previousDataProcessingJobIdHasBeenSet = true;
    m_previousDataProcessingJobId = std::forward<PreviousDataProcessingJobIdT>(value);
  }
  template <typename PreviousDataProcessingJobIdT = Aws::String>
  StartMLDataProcessingJobRequest& WithPreviousDataProcessingJobId(PreviousDataProcessingJobIdT&& value) {
    SetPreviousDataProcessingJobId(std::forward<PreviousDataProcessingJobIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The URI of the Amazon S3 location where you want SageMaker to download the
   * data needed to run the data processing job.</p>
   */
  inline const Aws::String& GetInputDataS3Location() const { return m_inputDataS3Location; }
  inline bool InputDataS3LocationHasBeenSet() const { return m_inputDataS3LocationHasBeenSet; }
  template <typename InputDataS3LocationT = Aws::String>
  void SetInputDataS3Location(InputDataS3LocationT&& value) {
    m_inputDataS3LocationHasBeenSet = true;
    m_inputDataS3Location = std::forward<InputDataS3LocationT>(value);
  }
  template <typename InputDataS3LocationT = Aws::String>
  StartMLDataProcessingJobRequest& WithInputDataS3Location(InputDataS3LocationT&& value) {
    SetInputDataS3Location(std::forward<InputDataS3LocationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The URI of the Amazon S3 location where you want SageMaker to save the
   * results of a data processing job.</p>
   */
  inline const Aws::String& GetProcessedDataS3Location() const { return m_processedDataS3Location; }
  inline bool ProcessedDataS3LocationHasBeenSet() const { return m_processedDataS3LocationHasBeenSet; }
  template <typename ProcessedDataS3LocationT = Aws::String>
  void SetProcessedDataS3Location(ProcessedDataS3LocationT&& value) {
    m_processedDataS3LocationHasBeenSet = true;
    m_processedDataS3Location = std::forward<ProcessedDataS3LocationT>(value);
  }
  template <typename ProcessedDataS3LocationT = Aws::String>
  StartMLDataProcessingJobRequest& WithProcessedDataS3Location(ProcessedDataS3LocationT&& value) {
    SetProcessedDataS3Location(std::forward<ProcessedDataS3LocationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of an IAM role for SageMaker execution. This must be listed in your
   * DB cluster parameter group or an error will occur.</p>
   */
  inline const Aws::String& GetSagemakerIamRoleArn() const { return m_sagemakerIamRoleArn; }
  inline bool SagemakerIamRoleArnHasBeenSet() const { return m_sagemakerIamRoleArnHasBeenSet; }
  template <typename SagemakerIamRoleArnT = Aws::String>
  void SetSagemakerIamRoleArn(SagemakerIamRoleArnT&& value) {
    m_sagemakerIamRoleArnHasBeenSet = true;
    m_sagemakerIamRoleArn = std::forward<SagemakerIamRoleArnT>(value);
  }
  template <typename SagemakerIamRoleArnT = Aws::String>
  StartMLDataProcessingJobRequest& WithSagemakerIamRoleArn(SagemakerIamRoleArnT&& value) {
    SetSagemakerIamRoleArn(std::forward<SagemakerIamRoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of an IAM role that SageMaker can assume to
   * perform tasks on your behalf. This must be listed in your DB cluster parameter
   * group or an error will occur.</p>
   */
  inline const Aws::String& GetNeptuneIamRoleArn() const { return m_neptuneIamRoleArn; }
  inline bool NeptuneIamRoleArnHasBeenSet() const { return m_neptuneIamRoleArnHasBeenSet; }
  template <typename NeptuneIamRoleArnT = Aws::String>
  void SetNeptuneIamRoleArn(NeptuneIamRoleArnT&& value) {
    m_neptuneIamRoleArnHasBeenSet = true;
    m_neptuneIamRoleArn = std::forward<NeptuneIamRoleArnT>(value);
  }
  template <typename NeptuneIamRoleArnT = Aws::String>
  StartMLDataProcessingJobRequest& WithNeptuneIamRoleArn(NeptuneIamRoleArnT&& value) {
    SetNeptuneIamRoleArn(std::forward<NeptuneIamRoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of ML instance used during data processing. Its memory should be
   * large enough to hold the processed dataset. The default is the smallest ml.r5
   * type whose memory is ten times larger than the size of the exported graph data
   * on disk.</p>
   */
  inline const Aws::String& GetProcessingInstanceType() const { return m_processingInstanceType; }
  inline bool ProcessingInstanceTypeHasBeenSet() const { return m_processingInstanceTypeHasBeenSet; }
  template <typename ProcessingInstanceTypeT = Aws::String>
  void SetProcessingInstanceType(ProcessingInstanceTypeT&& value) {
    m_processingInstanceTypeHasBeenSet = true;
    m_processingInstanceType = std::forward<ProcessingInstanceTypeT>(value);
  }
  template <typename ProcessingInstanceTypeT = Aws::String>
  StartMLDataProcessingJobRequest& WithProcessingInstanceType(ProcessingInstanceTypeT&& value) {
    SetProcessingInstanceType(std::forward<ProcessingInstanceTypeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The disk volume size of the processing instance. Both input data and
   * processed data are stored on disk, so the volume size must be large enough to
   * hold both data sets. The default is 0. If not specified or 0, Neptune ML chooses
   * the volume size automatically based on the data size.</p>
   */
  inline int GetProcessingInstanceVolumeSizeInGB() const { return m_processingInstanceVolumeSizeInGB; }
  inline bool ProcessingInstanceVolumeSizeInGBHasBeenSet() const { return m_processingInstanceVolumeSizeInGBHasBeenSet; }
  inline void SetProcessingInstanceVolumeSizeInGB(int value) {
    m_processingInstanceVolumeSizeInGBHasBeenSet = true;
    m_processingInstanceVolumeSizeInGB = value;
  }
  inline StartMLDataProcessingJobRequest& WithProcessingInstanceVolumeSizeInGB(int value) {
    SetProcessingInstanceVolumeSizeInGB(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Timeout in seconds for the data processing job. The default is 86,400 (1
   * day).</p>
   */
  inline int GetProcessingTimeOutInSeconds() const { return m_processingTimeOutInSeconds; }
  inline bool ProcessingTimeOutInSecondsHasBeenSet() const { return m_processingTimeOutInSecondsHasBeenSet; }
  inline void SetProcessingTimeOutInSeconds(int value) {
    m_processingTimeOutInSecondsHasBeenSet = true;
    m_processingTimeOutInSeconds = value;
  }
  inline StartMLDataProcessingJobRequest& WithProcessingTimeOutInSeconds(int value) {
    SetProcessingTimeOutInSeconds(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>One of the two model types that Neptune ML currently supports: heterogeneous
   * graph models (<code>heterogeneous</code>), and knowledge graph
   * (<code>kge</code>). The default is none. If not specified, Neptune ML chooses
   * the model type automatically based on the data.</p>
   */
  inline const Aws::String& GetModelType() const { return m_modelType; }
  inline bool ModelTypeHasBeenSet() const { return m_modelTypeHasBeenSet; }
  template <typename ModelTypeT = Aws::String>
  void SetModelType(ModelTypeT&& value) {
    m_modelTypeHasBeenSet = true;
    m_modelType = std::forward<ModelTypeT>(value);
  }
  template <typename ModelTypeT = Aws::String>
  StartMLDataProcessingJobRequest& WithModelType(ModelTypeT&& value) {
    SetModelType(std::forward<ModelTypeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A data specification file that describes how to load the exported graph data
   * for training. The file is automatically generated by the Neptune export toolkit.
   * The default is <code>training-data-configuration.json</code>.</p>
   */
  inline const Aws::String& GetConfigFileName() const { return m_configFileName; }
  inline bool ConfigFileNameHasBeenSet() const { return m_configFileNameHasBeenSet; }
  template <typename ConfigFileNameT = Aws::String>
  void SetConfigFileName(ConfigFileNameT&& value) {
    m_configFileNameHasBeenSet = true;
    m_configFileName = std::forward<ConfigFileNameT>(value);
  }
  template <typename ConfigFileNameT = Aws::String>
  StartMLDataProcessingJobRequest& WithConfigFileName(ConfigFileNameT&& value) {
    SetConfigFileName(std::forward<ConfigFileNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The IDs of the subnets in the Neptune VPC. The default is None.</p>
   */
  inline const Aws::Vector<Aws::String>& GetSubnets() const { return m_subnets; }
  inline bool SubnetsHasBeenSet() const { return m_subnetsHasBeenSet; }
  template <typename SubnetsT = Aws::Vector<Aws::String>>
  void SetSubnets(SubnetsT&& value) {
    m_subnetsHasBeenSet = true;
    m_subnets = std::forward<SubnetsT>(value);
  }
  template <typename SubnetsT = Aws::Vector<Aws::String>>
  StartMLDataProcessingJobRequest& WithSubnets(SubnetsT&& value) {
    SetSubnets(std::forward<SubnetsT>(value));
    return *this;
  }
  template <typename SubnetsT = Aws::String>
  StartMLDataProcessingJobRequest& AddSubnets(SubnetsT&& value) {
    m_subnetsHasBeenSet = true;
    m_subnets.emplace_back(std::forward<SubnetsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The VPC security group IDs. The default is None.</p>
   */
  inline const Aws::Vector<Aws::String>& GetSecurityGroupIds() const { return m_securityGroupIds; }
  inline bool SecurityGroupIdsHasBeenSet() const { return m_securityGroupIdsHasBeenSet; }
  template <typename SecurityGroupIdsT = Aws::Vector<Aws::String>>
  void SetSecurityGroupIds(SecurityGroupIdsT&& value) {
    m_securityGroupIdsHasBeenSet = true;
    m_securityGroupIds = std::forward<SecurityGroupIdsT>(value);
  }
  template <typename SecurityGroupIdsT = Aws::Vector<Aws::String>>
  StartMLDataProcessingJobRequest& WithSecurityGroupIds(SecurityGroupIdsT&& value) {
    SetSecurityGroupIds(std::forward<SecurityGroupIdsT>(value));
    return *this;
  }
  template <typename SecurityGroupIdsT = Aws::String>
  StartMLDataProcessingJobRequest& AddSecurityGroupIds(SecurityGroupIdsT&& value) {
    m_securityGroupIdsHasBeenSet = true;
    m_securityGroupIds.emplace_back(std::forward<SecurityGroupIdsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Key Management Service (Amazon KMS) key that SageMaker uses to
   * encrypt data on the storage volume attached to the ML compute instances that run
   * the training job. The default is None.</p>
   */
  inline const Aws::String& GetVolumeEncryptionKMSKey() const { return m_volumeEncryptionKMSKey; }
  inline bool VolumeEncryptionKMSKeyHasBeenSet() const { return m_volumeEncryptionKMSKeyHasBeenSet; }
  template <typename VolumeEncryptionKMSKeyT = Aws::String>
  void SetVolumeEncryptionKMSKey(VolumeEncryptionKMSKeyT&& value) {
    m_volumeEncryptionKMSKeyHasBeenSet = true;
    m_volumeEncryptionKMSKey = std::forward<VolumeEncryptionKMSKeyT>(value);
  }
  template <typename VolumeEncryptionKMSKeyT = Aws::String>
  StartMLDataProcessingJobRequest& WithVolumeEncryptionKMSKey(VolumeEncryptionKMSKeyT&& value) {
    SetVolumeEncryptionKMSKey(std::forward<VolumeEncryptionKMSKeyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Key Management Service (Amazon KMS) key that SageMaker uses to
   * encrypt the output of the processing job. The default is none.</p>
   */
  inline const Aws::String& GetS3OutputEncryptionKMSKey() const { return m_s3OutputEncryptionKMSKey; }
  inline bool S3OutputEncryptionKMSKeyHasBeenSet() const { return m_s3OutputEncryptionKMSKeyHasBeenSet; }
  template <typename S3OutputEncryptionKMSKeyT = Aws::String>
  void SetS3OutputEncryptionKMSKey(S3OutputEncryptionKMSKeyT&& value) {
    m_s3OutputEncryptionKMSKeyHasBeenSet = true;
    m_s3OutputEncryptionKMSKey = std::forward<S3OutputEncryptionKMSKeyT>(value);
  }
  template <typename S3OutputEncryptionKMSKeyT = Aws::String>
  StartMLDataProcessingJobRequest& WithS3OutputEncryptionKMSKey(S3OutputEncryptionKMSKeyT&& value) {
    SetS3OutputEncryptionKMSKey(std::forward<S3OutputEncryptionKMSKeyT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_id;

  Aws::String m_previousDataProcessingJobId;

  Aws::String m_inputDataS3Location;

  Aws::String m_processedDataS3Location;

  Aws::String m_sagemakerIamRoleArn;

  Aws::String m_neptuneIamRoleArn;

  Aws::String m_processingInstanceType;

  int m_processingInstanceVolumeSizeInGB{0};

  int m_processingTimeOutInSeconds{0};

  Aws::String m_modelType;

  Aws::String m_configFileName;

  Aws::Vector<Aws::String> m_subnets;

  Aws::Vector<Aws::String> m_securityGroupIds;

  Aws::String m_volumeEncryptionKMSKey;

  Aws::String m_s3OutputEncryptionKMSKey;
  bool m_idHasBeenSet = false;
  bool m_previousDataProcessingJobIdHasBeenSet = false;
  bool m_inputDataS3LocationHasBeenSet = false;
  bool m_processedDataS3LocationHasBeenSet = false;
  bool m_sagemakerIamRoleArnHasBeenSet = false;
  bool m_neptuneIamRoleArnHasBeenSet = false;
  bool m_processingInstanceTypeHasBeenSet = false;
  bool m_processingInstanceVolumeSizeInGBHasBeenSet = false;
  bool m_processingTimeOutInSecondsHasBeenSet = false;
  bool m_modelTypeHasBeenSet = false;
  bool m_configFileNameHasBeenSet = false;
  bool m_subnetsHasBeenSet = false;
  bool m_securityGroupIdsHasBeenSet = false;
  bool m_volumeEncryptionKMSKeyHasBeenSet = false;
  bool m_s3OutputEncryptionKMSKeyHasBeenSet = false;
};

}  // namespace Model
}  // namespace neptunedata
}  // namespace Aws
