﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/quicksight/QuickSightRequest.h>
#include <aws/quicksight/QuickSight_EXPORTS.h>
#include <aws/quicksight/model/AssignmentStatus.h>

#include <utility>

namespace Aws {
namespace QuickSight {
namespace Model {

/**
 */
class UpdateIAMPolicyAssignmentRequest : public QuickSightRequest {
 public:
  AWS_QUICKSIGHT_API UpdateIAMPolicyAssignmentRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateIAMPolicyAssignment"; }

  AWS_QUICKSIGHT_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The ID of the Amazon Web Services account that contains the IAM policy
   * assignment. </p>
   */
  inline const Aws::String& GetAwsAccountId() const { return m_awsAccountId; }
  inline bool AwsAccountIdHasBeenSet() const { return m_awsAccountIdHasBeenSet; }
  template <typename AwsAccountIdT = Aws::String>
  void SetAwsAccountId(AwsAccountIdT&& value) {
    m_awsAccountIdHasBeenSet = true;
    m_awsAccountId = std::forward<AwsAccountIdT>(value);
  }
  template <typename AwsAccountIdT = Aws::String>
  UpdateIAMPolicyAssignmentRequest& WithAwsAccountId(AwsAccountIdT&& value) {
    SetAwsAccountId(std::forward<AwsAccountIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the assignment, also called a rule. The name must be unique
   * within the Amazon Web Services account.</p>
   */
  inline const Aws::String& GetAssignmentName() const { return m_assignmentName; }
  inline bool AssignmentNameHasBeenSet() const { return m_assignmentNameHasBeenSet; }
  template <typename AssignmentNameT = Aws::String>
  void SetAssignmentName(AssignmentNameT&& value) {
    m_assignmentNameHasBeenSet = true;
    m_assignmentName = std::forward<AssignmentNameT>(value);
  }
  template <typename AssignmentNameT = Aws::String>
  UpdateIAMPolicyAssignmentRequest& WithAssignmentName(AssignmentNameT&& value) {
    SetAssignmentName(std::forward<AssignmentNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The namespace of the assignment.</p>
   */
  inline const Aws::String& GetNamespace() const { return m_namespace; }
  inline bool NamespaceHasBeenSet() const { return m_namespaceHasBeenSet; }
  template <typename NamespaceT = Aws::String>
  void SetNamespace(NamespaceT&& value) {
    m_namespaceHasBeenSet = true;
    m_namespace = std::forward<NamespaceT>(value);
  }
  template <typename NamespaceT = Aws::String>
  UpdateIAMPolicyAssignmentRequest& WithNamespace(NamespaceT&& value) {
    SetNamespace(std::forward<NamespaceT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The status of the assignment. Possible values are as follows:</p> <ul> <li>
   * <p> <code>ENABLED</code> - Anything specified in this assignment is used when
   * creating the data source.</p> </li> <li> <p> <code>DISABLED</code> - This
   * assignment isn't used when creating the data source.</p> </li> <li> <p>
   * <code>DRAFT</code> - This assignment is an unfinished draft and isn't used when
   * creating the data source.</p> </li> </ul>
   */
  inline AssignmentStatus GetAssignmentStatus() const { return m_assignmentStatus; }
  inline bool AssignmentStatusHasBeenSet() const { return m_assignmentStatusHasBeenSet; }
  inline void SetAssignmentStatus(AssignmentStatus value) {
    m_assignmentStatusHasBeenSet = true;
    m_assignmentStatus = value;
  }
  inline UpdateIAMPolicyAssignmentRequest& WithAssignmentStatus(AssignmentStatus value) {
    SetAssignmentStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN for the IAM policy to apply to the Amazon Quick Sight users and
   * groups specified in this assignment.</p>
   */
  inline const Aws::String& GetPolicyArn() const { return m_policyArn; }
  inline bool PolicyArnHasBeenSet() const { return m_policyArnHasBeenSet; }
  template <typename PolicyArnT = Aws::String>
  void SetPolicyArn(PolicyArnT&& value) {
    m_policyArnHasBeenSet = true;
    m_policyArn = std::forward<PolicyArnT>(value);
  }
  template <typename PolicyArnT = Aws::String>
  UpdateIAMPolicyAssignmentRequest& WithPolicyArn(PolicyArnT&& value) {
    SetPolicyArn(std::forward<PolicyArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Quick Sight users, groups, or both that you want to assign the
   * policy to.</p>
   */
  inline const Aws::Map<Aws::String, Aws::Vector<Aws::String>>& GetIdentities() const { return m_identities; }
  inline bool IdentitiesHasBeenSet() const { return m_identitiesHasBeenSet; }
  template <typename IdentitiesT = Aws::Map<Aws::String, Aws::Vector<Aws::String>>>
  void SetIdentities(IdentitiesT&& value) {
    m_identitiesHasBeenSet = true;
    m_identities = std::forward<IdentitiesT>(value);
  }
  template <typename IdentitiesT = Aws::Map<Aws::String, Aws::Vector<Aws::String>>>
  UpdateIAMPolicyAssignmentRequest& WithIdentities(IdentitiesT&& value) {
    SetIdentities(std::forward<IdentitiesT>(value));
    return *this;
  }
  template <typename IdentitiesKeyT = Aws::String, typename IdentitiesValueT = Aws::Vector<Aws::String>>
  UpdateIAMPolicyAssignmentRequest& AddIdentities(IdentitiesKeyT&& key, IdentitiesValueT&& value) {
    m_identitiesHasBeenSet = true;
    m_identities.emplace(std::forward<IdentitiesKeyT>(key), std::forward<IdentitiesValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_awsAccountId;

  Aws::String m_assignmentName;

  Aws::String m_namespace;

  AssignmentStatus m_assignmentStatus{AssignmentStatus::NOT_SET};

  Aws::String m_policyArn;

  Aws::Map<Aws::String, Aws::Vector<Aws::String>> m_identities;
  bool m_awsAccountIdHasBeenSet = false;
  bool m_assignmentNameHasBeenSet = false;
  bool m_namespaceHasBeenSet = false;
  bool m_assignmentStatusHasBeenSet = false;
  bool m_policyArnHasBeenSet = false;
  bool m_identitiesHasBeenSet = false;
};

}  // namespace Model
}  // namespace QuickSight
}  // namespace Aws
