## ----setup, echo = FALSE, eval=TRUE, message=FALSE----------------------------
library(BiocStyle)
knitr::opts_knit$set(
  upload.fun = NULL,
  base.url = NULL) # use local files for images
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#"
)
# Override BiocStyle plot hook to avoid css_align issues
knitr::knit_hooks$set(plot = function(x, options) {
  paste0('![', basename(x), '](', x, ')')
})
runchunks = if (Sys.getenv("FORCE_VIGNETTE_REBUILD", "FALSE") == "TRUE") TRUE else FALSE

cache_file <- '../vignette_cache/cross-domain-interactions.RData'
if (!runchunks && file.exists(cache_file)) {
  load(cache_file)
  # If we loaded trimmed objects, reassign them to original names
  if (exists("se.hmp2.trimmed")) {
    se.hmp2 <- se.hmp2.trimmed
    rm(se.hmp2.trimmed)
  }
  if (exists("se.cross.trimmed")) {
    se.cross <- se.cross.trimmed
    rm(se.cross.trimmed)
  }
} else {
  if (!runchunks) {
    message("Cache file cross-domain-interactions.RData not found - building from scratch")
  }
  runchunks <- TRUE
}
saveout   = runchunks


## ----eval=TRUE----------------------------------------------------------------
library(SpiecEasi)
library(phyloseq)
data(hmp2)

## ----eval=runchunks, echo=TRUE, message=FALSE---------------------------------
# se.hmp2 <- spiec.easi(list(hmp216S, hmp2prot), method='mb', nlambda=40,
#               lambda.min.ratio=1e-2, pulsar.params = list(thresh = 0.05))

## ----eval=TRUE----------------------------------------------------------------
dtype <- c(rep(1,ntaxa(hmp216S)), rep(2,ntaxa(hmp2prot)))
plot(adj2igraph(getRefit(se.hmp2)), vertex.color=dtype+1, vertex.size=9)

## ----eval=runchunks-----------------------------------------------------------
# # Example with custom data
# # Assuming you have two data matrices with the same samples
# data1 <- matrix(rpois(100*50, 10), 100, 50)  # First data type
# data2 <- matrix(rpois(100*30, 5), 100, 30)   # Second data type
# 
# # Run cross-domain SPIEC-EASI
# se.cross <- spiec.easi(list(data1, data2), method='mb',
#                        lambda.min.ratio=1e-2, nlambda=20,
#                        pulsar.params=list(rep.num=50))
# 

## ----eval=TRUE----------------------------------------------------------------
# Create visualization with different colors for each domain
dtype <- c(rep(1, ncol(data1)), rep(2, ncol(data2)))
ig.cross <- adj2igraph(getRefit(se.cross))
plot(ig.cross, vertex.color=dtype+1, vertex.size=6)

## ----eval=TRUE----------------------------------------------------------------
sessionInfo()

## ----echo = FALSE, eval=TRUE, message=FALSE-----------------------------------
# Save objects if they exist
if (exists("se.hmp2") && exists("dtype")) {
  cache_file <- '../vignette_cache/cross-domain-interactions.RData'
  tryCatch({
    # Load trimming function and trim objects to reduce size
    source('../vignette_cache/trim_spiec_easi.R')
    se.hmp2.trimmed <- trim_spiec_easi(se.hmp2)
    se.cross.trimmed <- trim_spiec_easi(se.cross)
    
    # Save trimmed objects and other essential data
    save(se.hmp2.trimmed, se.cross.trimmed, dtype, data1, data2, ig.cross, file=cache_file)
    message("Saved trimmed cache file: ", cache_file, " in directory: ", getwd())
  }, error = function(e) {
    message("Failed to save cache file: ", e$message)
  })
}

