## ----setup, echo=FALSE, results="hide"----------------------------------------
knitr::opts_chunk$set(
    tidy = FALSE,
    cache = FALSE,
    dev = "CairoPNG",
    message = FALSE, error = FALSE, warning = TRUE
)

## ----install, eval=FALSE------------------------------------------------------
# if (!require("BiocManager", quietly = TRUE)) {
#     install.packages("BiocManager")
# }
# 
# BiocManager::install("LACHESIS")

## ----structure, echo=FALSE, fig.show='hold', fig.wide = TRUE------------------
knitr::include_graphics(system.file("extdata", "LACHESIS_package_structure.png", package = "LACHESIS"))

## ----readCNV_function, eval = FALSE-------------------------------------------
# readCNV(cn.info = NULL, chr.col = NULL, start.col = NULL, end.col = NULL, A.col = NULL, B.col = NULL, tcn.col = NULL, merge.tolerance = 10^5, ignore.XY = TRUE, max.cn = 4, tumor.id = NULL)

## ----readCNV_output, echo=FALSE-----------------------------------------------
library(knitr)

readcnv_output <- data.frame(
    Column1 = c("*Chr*", "*Start*", "*End*", "*A*", "*B*", "*TCN*"),
    Column2 = c(
        "the chromosome number",
        "start position of the segment",
        "end position of the segment",
        "copy number of the major allele",
        "copy number of the minor allele",
        "the total copy number"
    )
)

kable(readcnv_output, col.names = c("Output", "Definition"), escape = FALSE)

## ----readCNV_example1, message=FALSE, warning=FALSE---------------------------
aceseq_cn <- system.file("extdata", "ACESeq/NBE11_comb_pro_extra2.59_0.83.txt", package = "LACHESIS")
cn_data <- LACHESIS::readCNV(aceseq_cn)
head(cn_data, n = 1)

## ----readCNV_example2, message=FALSE, warning=FALSE---------------------------
ascat_cn <- system.file("extdata", "ASCAT/S98.segments.txt", package = "LACHESIS")
cn_data <- LACHESIS::readCNV(ascat_cn)
head(cn_data, n = 1)

## ----readCNV_example3, message=FALSE, warning=FALSE---------------------------
purple_cn <- system.file("extdata", "PURPLE/NB-S-599-T.purple.cnv.somatic.tsv", package = "LACHESIS")
cn_data <- LACHESIS::readCNV(purple_cn)
head(cn_data, n = 1)

## ----readVCF_function, eval=FALSE---------------------------------------------
# readVCF(vcf = NULL, ignore.XY = TRUE, vcf.source = "strelka", min.vaf = 0.01, min.depth = 30, t.sample = NULL, info.af = "AF", info.dp = "DP", filter.value = "PASS")

## ----readVCF_output, echo=FALSE-----------------------------------------------
library(knitr)

readvcf_output <- data.frame(
    Column1 = c("*chrom*", "*pos*", "*ref*", "*alt*", "*t_ref_count*", "*t_alt_count*", "*t_depth*", "*t_vaf*"),
    Column2 = c(
        "the chromosome",
        "the chromosomal coordinate of the SNV",
        "the reference nucleotide at this position",
        "the alternate nucleotide at this position",
        "the number of reads reporting the reference nucleotide",
        "the number of reads reporting the alternate nucleotide",
        "the read coverage at this position",
        "the variant allele frequency of the SSNV"
    )
)

knitr::kable(readvcf_output, col.names = c("Output", "Definition"), escape = FALSE)

## ----readVCF_example1, message=FALSE, warning=FALSE---------------------------
mutect_vcf <- system.file("extdata", "mutect.somatic.vcf.gz", package = "LACHESIS")
m_data <- LACHESIS::readVCF(vcf = mutect_vcf, vcf.source = "mutect", filter.value = ".")
head(m_data, n = 1)

## ----readVCF_example2, message=FALSE, warning=FALSE---------------------------
strelka_vcf <- system.file("extdata", "strelka2.somatic.snvs.vcf.gz", package = "LACHESIS")
s_data <- LACHESIS::readVCF(vcf = strelka_vcf, vcf.source = "strelka", filter.value = "PASS")
head(s_data, n = 1)

## ----readVCF_example3, message=FALSE, warning=FALSE---------------------------
dkfz_vcf <- system.file("extdata", "NBE15", "snvs_NBE15_somatic_snvs_conf_8_to_10.vcf", package = "LACHESIS")
d_data <- LACHESIS::readVCF(vcf = dkfz_vcf, vcf.source = "dkfz")
head(d_data, n = 1)

## ----plotVAF_function, eval = FALSE-------------------------------------------
# plotVAFdistr(vaf = NULL, vaf.interval = 0.05, t_sample = NULL, vaf.show.counts = FALSE, vaf.show.density = TRUE, vaf.col = "#34495e", vaf.border = "#bdc3c7", srtcounts = 45, output.file = NULL)

## ----plot1, message=FALSE, warning=FALSE, fig.cap="VAF distribution of SSNVs"----
strelka_vcf <- system.file("extdata", "strelka2.somatic.snvs.vcf.gz", package = "LACHESIS")
s_data <- LACHESIS::readVCF(vcf = strelka_vcf, vcf.source = "strelka")
head(s_data, n = 1)
LACHESIS::plotVAFdistr(s_data)

## ----nbImport_function, eval = FALSE------------------------------------------
# nbImport(cnv = NULL, snv = NULL, purity = NULL, ploidy = NULL)

## ----nbImport_output, echo=FALSE----------------------------------------------
library(knitr)

nb_import_output <- data.frame(
    Column1 = c("*chrom*", "*cn_start*", "*cn_end*", "*A*", "*B*", "*TCN*", "*snv_start*", "*ref*", "*alt*", "*t_ref_count*", "*t_alt_count*", "*t_depth*", "*t_vaf*", "*snv_end*"),
    Column2 = c(
        "the chromosome",
        "start of the segment",
        "end of the segment",
        "copy number of the major allele",
        "copy number of the minor allele",
        "the total copy number",
        "the chromosomal start coordinate of the SNV",
        "the reference nucleotide at this position",
        "the alternate nucleotide at this position",
        "the number of reads reporting the reference nucleotide",
        "the number of reads reporting the alternate nucleotide",
        "the read coverage at this position",
        "the variant allele frequency of the SSNV",
        "the chromosomal end coordinate of the SNV"
    )
)

knitr::kable(nb_import_output, col.names = c("Output", "Definition"), escape = FALSE)

## ----nbImport_example1, message=FALSE, warning=FALSE--------------------------
snvs <- system.file("extdata", "NBE15", "snvs_NBE15_somatic_snvs_conf_8_to_10.vcf", package = "LACHESIS")
s_data <- LACHESIS::readVCF(vcf = snvs, vcf.source = "dkfz")
aceseq_cn <- system.file("extdata", "NBE15", "NBE15_comb_pro_extra2.51_1.txt", package = "LACHESIS")
c_data <- LACHESIS::readCNV(aceseq_cn)
nb <- LACHESIS::nbImport(cnv = c_data, snv = s_data, purity = 1, ploidy = 2.51)

## ----nbImport_example2, message=FALSE, warning=FALSE--------------------------
snvs <- system.file("extdata", "NBE15", "snvs_NBE15_somatic_snvs_conf_8_to_10.vcf", package = "LACHESIS")
s_data <- LACHESIS::readVCF(vcf = snvs, vcf.source = "dkfz")
aceseq_cn <- system.file("extdata", "NBE15", "NBE15_comb_pro_extra2.51_1.txt", package = "LACHESIS")
c_data <- LACHESIS::readCNV(aceseq_cn)
sig.filepath <- system.file("extdata", "NBE15_Decomposed_MutationType_Probabilities.txt", package = "LACHESIS")
head(sig.filepath, n = 1)
nb <- LACHESIS::nbImport(cnv = c_data, snv = s_data, purity = 1, ploidy = 2.51, sig.assign = TRUE, ID = "NBE15", sig.file = sig.filepath, sig.select = c("SBS1", "SBS5", "SBS40a", "SBS18"))
head(nb, n = 1)
nb.2 <- LACHESIS::nbImport(cnv = c_data, snv = s_data, purity = 1, ploidy = 2.51, sig.assign = TRUE, ID = "NBE15", sig.select = c("SBS1", "SBS5", "SBS40", "SBS18"))
head(nb.2, n = 1)

## ----plotNB_function, eval = FALSE--------------------------------------------
# plotNB(nb = NULL, snvClonality = NULL, ref.build = "hg19", min.cn = 2, max.cn = 4, nb.col.abline = "gray70", nb.col.cn.2 = "#7f8c8d", nb.col.cn = "#16a085", nb.col.hist = "#34495e", nb.border = NA, nb.breaks = 100, samp.name = NULL, output.file = NULL, sig.show = FALSE)

## ----plot2, message=FALSE, warning=FALSE, fig.cap="*top plot* - copy number profile along the genome, *bottom plots* - VAF distribution of SSNVs stratified by copy number and minor/major alleles"----
snvs <- system.file("extdata", "NBE15", "snvs_NBE15_somatic_snvs_conf_8_to_10.vcf", package = "LACHESIS")
s_data <- LACHESIS::readVCF(vcf = snvs, vcf.source = "dkfz")
aceseq_cn <- system.file("extdata", "NBE15", "NBE15_comb_pro_extra2.51_1.txt", package = "LACHESIS")
c_data <- LACHESIS::readCNV(aceseq_cn)
nb <- snvs <- system.file("extdata", "NBE15", "snvs_NBE15_somatic_snvs_conf_8_to_10.vcf", package = "LACHESIS")
nb <- LACHESIS::nbImport(cnv = c_data, snv = s_data, purity = 1, ploidy = 2.51)
cl_muts <- LACHESIS::clonalMutationCounter(nb)
norm_muts <- LACHESIS::normalizeCounts(cl_muts)
mrca <- LACHESIS::MRCA(norm_muts)
snvClonality <- LACHESIS::estimateClonality(nbObj = nb, mrcaObj = mrca, ID = "NBE15", purity = 1)
LACHESIS::plotNB(nb = nb, snvClonality = snvClonality)

## ----plot3, message=FALSE, warning=FALSE, fig.cap="*top plot* - copy number profile along the genome, *bottom plots* - VAF distribution of SSNVs stratified by copy number and minor/major alleles"----
set.seed(123)
snvs <- system.file("extdata", "NBE15", "snvs_NBE15_somatic_snvs_conf_8_to_10.vcf", package = "LACHESIS")
s_data <- LACHESIS::readVCF(vcf = snvs, vcf.source = "dkfz")
aceseq_cn <- system.file("extdata", "NBE15", "NBE15_comb_pro_extra2.51_1.txt", package = "LACHESIS")
c_data <- LACHESIS::readCNV(aceseq_cn)
sig.filepath <- system.file("extdata", "NBE15_Decomposed_MutationType_Probabilities.txt", package = "LACHESIS")
nb <- LACHESIS::nbImport(cnv = c_data, snv = s_data, purity = 1, ploidy = 2.51, sig.assign = TRUE, ID = "NBE15", sig.file = sig.filepath, sig.select = c("SBS1", "SBS5", "SBS40a", "SBS18"))
cl_muts <- LACHESIS::clonalMutationCounter(nb)
norm_muts <- LACHESIS::normalizeCounts(cl_muts)
mrca <- LACHESIS::MRCA(norm_muts)
snvClonality <- LACHESIS::estimateClonality(nbObj = nb, mrcaObj = mrca, ID = "NBE15", purity = 1)
LACHESIS::plotNB(nb = nb, snvClonality = snvClonality, sig.show = TRUE)

## ----clonalmutcount_function, eval = FALSE------------------------------------
# clonalMutationCounter(nbObj = NULL, min.cn = 1, max.cn = 4, chromosomes = c(1:22))

## ----clonalmutcount_output, echo=FALSE----------------------------------------
library(knitr)

process_clonal_output <- data.frame(
    Column1 = c("*chrom*", "*TCN*", "*A*", "*B*", "*Seglength*", "*n_mut_A*", "*n_mut_B*", "*n_mut_total_clonal*"),
    Column2 = c(
        "the chromosome",
        "the total copy number",
        "copy number of the major allele",
        "copy number of the minor allele",
        "the segment length",
        "the number of mutations per copy with a multiplicity of A",
        "the number of mutations per copy with a multiplicity of B",
        "the total number of clonal mutations"
    )
)

knitr::kable(process_clonal_output, col.names = c("Column name", "Definition"), escape = FALSE)

## ----clonalmutcount_example1, message=FALSE, warning=FALSE--------------------
snvs <- system.file("extdata", "NBE15", "snvs_NBE15_somatic_snvs_conf_8_to_10.vcf", package = "LACHESIS")
s_data <- LACHESIS::readVCF(vcf = snvs, vcf.source = "dkfz")
aceseq_cn <- system.file("extdata", "NBE15", "NBE15_comb_pro_extra2.51_1.txt", package = "LACHESIS")
c_data <- LACHESIS::readCNV(aceseq_cn)
nb <- LACHESIS::nbImport(cnv = c_data, snv = s_data, purity = 1, ploidy = 2.51)
cl_muts <- LACHESIS::clonalMutationCounter(nb)
head(cl_muts, n = 1)

## ----normalizeCounts_function, eval=FALSE-------------------------------------
# normalizeCounts(countObj)

## ----normalizeCounts_output, echo=FALSE---------------------------------------
library(knitr)

normalize_counts_output <- data.frame(
    Column1 = c("*chrom*", "*TCN*", "*A*", "*B*", "*Seglength*", "*n_mut_A*", "*n_mut_B*", "*n_mut_total_clonal*", "*density_total_mean*", "*density_total_lower*", "*density_total_upper*", "*density_A_mean*", "*density_A_lower*", "*density_A_upper*", "*density_B_mean*", "*density_B_lower*", "*density_B_upper*"),
    Column2 = c(
        "the chromosome",
        "the total copy number",
        "copy number of the major allele",
        "copy number of the minor allele",
        "the segment length",
        "the number of mutations per copy with a multiplicity of A",
        "the number of mutations per copy with a multiplicity of B",
        "the total number of mutations",
        "mean mutation densities (1/Mb) per copy",
        "lower limit of mutation densities (1/Mb) per copy according to a 95% confidence interval",
        "upper limit of mutation densities (1/Mb) per copy ccording to a 95% confidence interval",
        "mean mutation densities (1/Mb) per copy with a multiplicity of A",
        "lower limit of mutation densities (1/Mb) per copy with a multiplicity of A according to a 95% confidence interval",
        "upper limit of mutation densities (1/Mb) per copy with a multiplicity of B according to a 95% confidence interval",
        "mean mutation densities (1/Mb) per copy with a multiplicity of B",
        "lower limit of mutation densities (1/Mb) per copy with a multiplicity of B",
        "upper limit of mutation densities (1/Mb) per copy with a multiplicity of B"
    )
)

knitr::kable(normalize_counts_output, col.names = c("Column name", "Definition"), escape = FALSE)

## ----normalizeCounts_example1, message=FALSE, warning=FALSE-------------------
snvs <- system.file("extdata", "NBE15", "snvs_NBE15_somatic_snvs_conf_8_to_10.vcf", package = "LACHESIS")
s_data <- LACHESIS::readVCF(vcf = snvs, vcf.source = "dkfz")
aceseq_cn <- system.file("extdata", "NBE15", "NBE15_comb_pro_extra2.51_1.txt", package = "LACHESIS")
c_data <- LACHESIS::readCNV(aceseq_cn)
nb <- LACHESIS::nbImport(cnv = c_data, snv = s_data, purity = 1, ploidy = 2.51)
cl_muts <- LACHESIS::clonalMutationCounter(nb)
norm_muts <- LACHESIS::normalizeCounts(cl_muts)
head(norm_muts, n = 1)

## ----MRCA_function, eval = FALSE----------------------------------------------
# MRCA(normObj = NULL, min.seg.size = 10^7, fp.mean = 0, fp.sd = 0, excl.chr = NULL)

## ----MRCA_output, echo=FALSE--------------------------------------------------
library(knitr)

mrca_dens_df <- data.frame(
    Column1 = c(
        "chrom", "TCN", "A", "B", "Seglength", "n_mut_A", "n_mut_B", "n_mut_total_clonal",
        "density_total_mean", "density_total_lower", "density_total_upper",
        "density_A_mean", "density_A_lower", "density_A_upper", "density_B_mean",
        "density_B_lower", "density_B_upper", "p_total_to_mrca", "p_A_to_mrca",
        "p_B_to_mrca", "p_adj_total_to_mrca", "p_adj_A_to_mrca", "p_adj_B_to_mrca",
        "MRCA_qual", "p_A_to_eca", "p_B_to_eca", "p_adj_A_to_eca", "p_adj_B_to_eca",
        "A_time", "B_time"
    ),
    Column2 = c(
        "the chromosome",
        "the total copy number",
        "copy number of the major allele",
        "copy number of the minor allele",
        "the segment length",
        "the normalized number of mutations per copy with a multiplicity of A",
        "the normalized number of mutations per copy with a multiplicity of B",
        "the normalized number of mutations per copy",
        "mean mutation densities (1/Mb) per copy",
        "lower limit of mutation densities (1/Mb) per copy according to a 95% confidence interval",
        "upper limit of mutation densities (1/Mb) per copy according to a 95% confidence interval",
        "mean mutation densities (1/Mb) per copy with a multiplicity of A",
        "lower limit of mutation densities (1/Mb) per copy with a multiplicity of A according to a 95% confidence interval",
        "upper limit of mutation densities (1/Mb) per copy with a multiplicity of B according to a 95% confidence interval",
        "mean mutation densities (1/Mb) per copy with a multiplicity of B",
        "lower limit of mutation densities (1/Mb) per copy with a multiplicity of B according to a 95% confidence interval",
        "upper limit of mutation densities (1/Mb) per copy with a multiplicity of B according to a 95% confidence interval",
        "probability that the mutation density per copy agrees with the mean mutation density at the MRCA",
        "probability that the mutation density per copy with a multiplicity of A agrees with the mean mutation density at the MRCA",
        "probability that the mutation density per copy with a multiplicity of B agrees with the mean mutation density at the MRCA",
        "Holm-corrected probability that the mutation density per copy agrees with the mean mutation density at the MRCA",
        "Holm-corrected probability that the mutation density per copy with a multiplicity of A agrees with the mean mutation density at the MRCA",
        "Holm-corrected probability that the mutation density per copy with a multiplicity of B agrees with the mean mutation density at the MRCA",
        "Quality control - PASS if `p_adj_to_mrca >= 0.01`",
        "probability that the mutation density per copy with a multiplicity of A agrees with the mean mutation density at the ECA",
        "probability that the mutation density per copy with a multiplicity of B agrees with the mean mutation density at the ECA",
        "Holm-corrected probability that the mutation density per copy with a multiplicity of A agrees with the mean mutation density at the ECA",
        "Holm-corrected probability that the mutation density per copy with a multiplicity of B agrees with the mean mutation density at the ECA",
        "Time point at which the copy number change of the major allele occurred (can be `ECA`, `MRCA`, or, if `A=1`, NA)",
        "Time point at which the copy number change of the major allele occurred (can be `ECA`, `MRCA`, or, if `B=1`, NA)"
    ),
    stringsAsFactors = FALSE
)

knitr::kable(mrca_dens_df, col.names = c("Column Name", "Definition"), align = "l")

## ----MRCA_example1, message=FALSE, warning=FALSE------------------------------
snvs <- system.file("extdata", "NBE15", "snvs_NBE15_somatic_snvs_conf_8_to_10.vcf", package = "LACHESIS")
s_data <- LACHESIS::readVCF(vcf = snvs, vcf.source = "dkfz")
aceseq_cn <- system.file("extdata", "NBE15", "NBE15_comb_pro_extra2.51_1.txt", package = "LACHESIS")
c_data <- LACHESIS::readCNV(aceseq_cn)
nb <- LACHESIS::nbImport(cnv = c_data, snv = s_data, purity = 1, ploidy = 2.51)
cl_muts <- LACHESIS::clonalMutationCounter(nb)
norm_muts <- LACHESIS::normalizeCounts(cl_muts)
mrca <- LACHESIS::MRCA(norm_muts)
head(mrca, n = 1)

## ----plotMutDens_function, eval = FALSE---------------------------------------
# plotMutationDensities(mrcaObj = NULL, samp.name = NULL, min.seg.size = 10^7, mut.col.zero = "#4FB12B", mut.col.multi = "#176A02", mut.border = NULL, mut.show.density = TRUE, mut.breaks = NULL, mut.xaxis = NULL, mut.show.realtime = FALSE, mut.snv.rate = 3.2, output.file = NULL)

## ----plot4, message=FALSE, warning=FALSE, fig.cap="*top plots* - Distribution of non-amplified and amplified clonal mutation densities across segments, *bottom plot* - Evolutionary timeline showing mutation densities at chromosomal gains (mean and 95% CI) and at the tumor's ECA and MRCA (mean and 95% CI of)"----
snvs <- system.file("extdata", "NBE15", "snvs_NBE15_somatic_snvs_conf_8_to_10.vcf", package = "LACHESIS")
s_data <- LACHESIS::readVCF(vcf = snvs, vcf.source = "dkfz")
aceseq_cn <- system.file("extdata", "NBE15", "NBE15_comb_pro_extra2.51_1.txt", package = "LACHESIS")
c_data <- LACHESIS::readCNV(aceseq_cn)
nb <- LACHESIS::nbImport(cnv = c_data, snv = s_data, purity = 1, ploidy = 2.51)
cl_muts <- LACHESIS::clonalMutationCounter(nb)
norm_muts <- LACHESIS::normalizeCounts(cl_muts)
mrca <- LACHESIS::MRCA(norm_muts)
LACHESIS::plotMutationDensities(mrca)

## ----estimateClonality_function, eval = FALSE---------------------------------
# estimateClonality(nbObj = NULL, mrcaObj = NULL, ID = NULL, purity = NULL)

## ----estimateClonality_example1, message=FALSE, warning=FALSE-----------------
snvs <- system.file("extdata", "NBE15", "snvs_NBE15_somatic_snvs_conf_8_to_10.vcf", package = "LACHESIS")
s_data <- LACHESIS::readVCF(vcf = snvs, vcf.source = "dkfz")
aceseq_cn <- system.file("extdata", "NBE15", "NBE15_comb_pro_extra2.51_1.txt", package = "LACHESIS")
c_data <- LACHESIS::readCNV(aceseq_cn)
nb <- LACHESIS::nbImport(cnv = c_data, snv = s_data, purity = 1, ploidy = 2.51)
cl_muts <- LACHESIS::clonalMutationCounter(nb)
norm_muts <- LACHESIS::normalizeCounts(cl_muts)
mrca <- LACHESIS::MRCA(norm_muts)
LACHESIS::estimateClonality(nbObj = nb, mrcaObj = mrca, ID = "NBE15", purity = 1)

## ----plotClonality_function, eval = FALSE-------------------------------------
# plotClonality(snvClonality = snvClonality, nb = NULL, sig.assign = FALSE, output.file = NULL)

## ----plotClonality_example1, message=FALSE, warning=FALSE---------------------
snvs <- system.file("extdata", "NBE15", "snvs_NBE15_somatic_snvs_conf_8_to_10.vcf", package = "LACHESIS")
s_data <- LACHESIS::readVCF(vcf = snvs, vcf.source = "dkfz")
aceseq_cn <- system.file("extdata", "NBE15", "NBE15_comb_pro_extra2.51_1.txt", package = "LACHESIS")
c_data <- LACHESIS::readCNV(aceseq_cn)
nb <- LACHESIS::nbImport(cnv = c_data, snv = s_data, purity = 1, ploidy = 2.51, ID = "NBE15")
cl_muts <- LACHESIS::clonalMutationCounter(nb)
norm_muts <- LACHESIS::normalizeCounts(cl_muts)
mrca <- LACHESIS::MRCA(norm_muts)
snvClonality <- LACHESIS::estimateClonality(nbObj = nb, mrcaObj = mrca, ID = "NBE15", purity = 1)
LACHESIS::plotClonality(snvClonality = snvClonality, nb = nb, sig.assign = FALSE)

## ----LACHESIS_function, eval = FALSE------------------------------------------
# LACHESIS(
#     input.files = NULL, ids = NULL, vcf.tumor.ids = NULL, cnv.files = NULL, snv.files = NULL,
#     vcf.source = NULL, purity = NULL, ploidy = NULL, cnv.chr.col = NULL, cnv.start.col = NULL,
#     cnv.end.col = NULL, cnv.A.col = NULL, cnv.B.col = NULL, cnv.tcn.col = NULL, age = NULL,
#     OS.time = NULL, OS = NULL, EFS.time = NULL, EFS = NULL, output.dir = NULL,
#     ignore.XY = TRUE, min.cn = 1, max.cn = 4, merge.tolerance = 10^5, min.vaf = 0.01,
#     min.depth = 30, vcf.info.af = "AF", vcf.info.dp = "DP", min.seg.size = 10^7, fp.mean = 0,
#     fp.sd = 0, excl.chr = NULL, ref_build = "hg19", ...
# )

## ----LACHESIS_output, echo=FALSE----------------------------------------------
library(knitr)

lachesis_output <- data.frame(
    Column1 = c(
        "Sample_ID", "MRCA_time_mean", "MRCA_time_lower/ MRCA_time_upper", "ECA_time_mean", "ECA_time_lower/ ECA_time_upper", "Ploidy", "Purity", "Age", "OS.time", "OS", "EFS.time", "EFS"
    ),
    Column2 = c(
        "the tumor sample ID",
        "lower and upper bounds of mutation density at MRCA as calculated in [MRCA function](#estimating-mutation-densities-at-ecamrca-mrca)",
        "mean mutation density at MRCA as calculated in [MRCA function](#estimating-mutation-densities-at-ecamrca-mrca)",
        "mean mutation density at ECA as calculated in [MRCA function](#estimating-mutation-densities-at-ecamrca-mrca)",
        "lower and upper bounds of mutation density at ECA as calculated in [MRCA function](#estimating-mutation-densities-at-ecamrca-mrca)",
        "user-specified average copy number in the tumor sample",
        "user-specified tumor cell content",
        "user-specified age at diagnosis",
        "user-specified overall survival time",
        "user-specified overall survival indicator variable",
        "user-specified eventfree survival time",
        "user-specified eventfree survival indicator variable"
    ),
    stringsAsFactors = FALSE
)

knitr::kable(lachesis_output, col.names = c("Column Name", "Definition"), align = "l")

## ----LACHESIS_example1, message=FALSE, warning=FALSE--------------------------
strelka_vcf <- system.file("extdata", "strelka2.somatic.snvs.vcf.gz", package = "LACHESIS")
aceseq_cn <- system.file("extdata", "ACESeq/NBE11_comb_pro_extra2.59_0.83.txt", package = "LACHESIS")
lachesis <- LACHESIS::LACHESIS(ids = "NBE11", cnv.files = aceseq_cn, snv.files = strelka_vcf, vcf.source = "strelka", purity = 0.83, ploidy = 2.59)
head(lachesis, n = 3)

## ----LACHESIS_example2, message=FALSE, warning=FALSE--------------------------
lachesis_input <- system.file("extdata", "Sample_template.txt", package = "LACHESIS")

## ----LACHESIS_example2_1, echo=FALSE------------------------------------------
input.files <- system.file("extdata", "Sample_template.txt", package = "LACHESIS")
input.files <- data.table::fread(input.files)

# cnv and snv files for example tumors
nbe11 <- list.files(system.file("extdata/NBE11/", package = "LACHESIS"), full.names = TRUE)
nbe15 <- list.files(system.file("extdata/NBE15/", package = "LACHESIS"), full.names = TRUE)
nbe26 <- list.files(system.file("extdata/NBE26/", package = "LACHESIS"), full.names = TRUE)

cnv.file <- c(nbe11[1], nbe15[1], nbe26[1])
snv.file <- c(nbe11[2], nbe15[2], nbe26[2])

input.files$cnv.file <- cnv.file
input.files$snv.file <- snv.file

# Make an example input file with paths to cnv and snv file along with other meta data
lachesis_input <- tempfile(pattern = "lachesis", tmpdir = tempdir(), fileext = ".tsv")
data.table::fwrite(x = input.files, file = lachesis_input, sep = "\t")

## ----LACHESIS_example2_4, message=FALSE, warning=FALSE------------------------
lachesis <- LACHESIS::LACHESIS(input.files = lachesis_input)
head(lachesis, n = 3)

## ----LACHESIS_example3, message=FALSE, warning=FALSE--------------------------
nbe11_vcf <- system.file("extdata", "NBE11/snvs_NBE11_somatic_snvs_conf_8_to_10.vcf", package = "LACHESIS")
nbe11_cn <- read.delim(system.file("extdata", "NBE11/NBE11_comb_pro_extra2.59_0.83.txt", package = "LACHESIS"), sep = "\t", header = TRUE)
nbe15_vcf <- system.file("extdata", "NBE15/snvs_NBE15_somatic_snvs_conf_8_to_10.vcf", package = "LACHESIS")
nbe15_cn <- read.delim(system.file("extdata", "NBE15/NBE15_comb_pro_extra2.51_1.txt", package = "LACHESIS"), sep = "\t", header = TRUE)

lachesis <- LACHESIS::LACHESIS(ids = c("NBE11", "NBE15"), cnv.files = list(nbe11_cn, nbe15_cn), snv.files = c(nbe11_vcf, nbe15_vcf), vcf.source = c("dkfz", "dkfz"), purity = c(0.83, 1), ploidy = c(2.59, 2.51), cnv.chr.col = c(1, 1), cnv.start.col = c(2, 2), cnv.end.col = c(3, 3), cnv.A.col = c(34, 34), cnv.B.col = c(35, 35), cnv.tcn.col = c(37, 37))
head(lachesis, n = 3)

## ----plotLACHESIS_function, eval = FALSE--------------------------------------
# plotLachesis(lachesis = NULL, lach.suppress.outliers = FALSE, lach.log.densities = FALSE, lach.col.multi = "#176A02", lach.border = NULL, binwidth = NULL, lach.col.zero = "#4FB12B", output.file = NULL)

## ----plotLACHESIS_example1, message=FALSE, warning=FALSE----------------------
lachesis_input <- system.file("extdata", "Sample_template.txt", package = "LACHESIS")

## ----plotLACHESIS_example1_1, echo=FALSE--------------------------------------
input.files <- system.file("extdata", "Sample_template.txt", package = "LACHESIS")
input.files <- data.table::fread(input.files)

# cnv and snv files for example tumors
nbe11 <- list.files(system.file("extdata/NBE11/", package = "LACHESIS"), full.names = TRUE)
nbe15 <- list.files(system.file("extdata/NBE15/", package = "LACHESIS"), full.names = TRUE)
nbe26 <- list.files(system.file("extdata/NBE26/", package = "LACHESIS"), full.names = TRUE)

cnv.file <- c(nbe11[1], nbe15[1], nbe26[1])
snv.file <- c(nbe11[2], nbe15[2], nbe26[2])

input.files$cnv.file <- cnv.file
input.files$snv.file <- snv.file

# Make an example input file with paths to cnv and snv file along with other meta data
lachesis_input <- tempfile(pattern = "lachesis", tmpdir = tempdir(), fileext = ".tsv")
data.table::fwrite(x = input.files, file = lachesis_input, sep = "\t")

## ----plot5, message=FALSE, warning=FALSE, fig.cap="Cumulative distribution of SSNV densities in ECA (dark green) and MRCA (light green)"----
lachesis <- LACHESIS::LACHESIS(input.files = lachesis_input)
LACHESIS::plotLachesis(lachesis)

## ----plotClinCor_function, eval = FALSE---------------------------------------
# plotClinicalCorrelations(lachesis = NULL, clin.par = "Age", clin.suppress.outliers = FALSE, clin.log.densities = FALSE, output.file = NULL)

## ----plot6, message=FALSE, warning=FALSE, fig.cap="Correlation between age at diagnosis and SNV densities at ECA and MRCA"----
input.files <- system.file("extdata", "Sample_template.txt", package = "LACHESIS")
lachesis <- LACHESIS::LACHESIS(input.files = lachesis_input)

LACHESIS::plotClinicalCorrelations(lachesis)

## ----plotSurvival_function, eval = FALSE--------------------------------------
# plotSurvival(lachesis = NULL, mrca.cutpoint = NULL, output.dir = NULL, surv.time = "OS.time", surv.event = "OS", surv.palette = c("dodgerblue", "dodgerblue4"), surv.time.breaks = NULL, surv.time.scale = 1, surv.title = "Survival probability", surv.ylab = "Survival")

## ----plot7, message=FALSE, warning=FALSE, fig.cap="Correlation between Survival and SNV densities at ECA and MRCA", echo = FALSE----
input.files <- system.file("extdata", "Sample_template.txt", package = "LACHESIS")
lachesis <- LACHESIS::LACHESIS(input.files = lachesis_input)

LACHESIS::plotSurvival(lachesis, surv.time = "EFS.time", surv.event = "EFS", mrca.cutpoint = 0.05)

## ----classifyLACHESIS_function, eval = FALSE----------------------------------
# classifyLACHESIS(lachesis, mrca.cutpoint = NULL, output.dir = NULL, entity = "neuroblastoma", infer.cutpoint = FALSE, surv.time = "OS.time", surv.event = "OS")

## ----classifyLACHESIS_example1, message=FALSE, warning=FALSE------------------
input.files <- system.file("extdata", "Sample_template.txt", package = "LACHESIS")
lachesis <- LACHESIS::LACHESIS(input.files = lachesis_input)

LACHESIS::classifyLACHESIS(lachesis, entity = "neuroblastoma")

## ----vaf-plots-positive, echo=FALSE, fig.show='hold', out.width="70%", fig.cap = "Clonal peaks (i.e. ->) correlate with expected VAFs (----)"----
knitr::include_graphics(system.file("extdata", "VAF_positive_example.png", package = "LACHESIS"))

## ----vaf-plots-negative, echo=FALSE, fig.show='hold', out.width="70%", fig.cap = "Clonal peaks cannot be clearly identified, a load of noise (i.e. ->) between the expected VAFs (----)"----
knitr::include_graphics(system.file("extdata", "VAF_negative_example.png", package = "LACHESIS"))

## ----warning_1, eval = FALSE--------------------------------------------------
# warning("Removing x segments with start > end")

## ----warning_2, eval = FALSE--------------------------------------------------
# warning("Less than 10% of the genome with valid copy number information.")

## ----warning_3, eval = FALSE--------------------------------------------------
# warning("No *x* identifier provided, assuming *y*")

## ----warning_4, eval = FALSE--------------------------------------------------
# warning("Allele information is not provided and will be assumed 1:1 in disomic regions, 2:1 in trisomic regions, 2:2 in tetrasomic regions, ... .")

## ----warning_5, eval = FALSE--------------------------------------------------
# warning("Removed x variants with no copy number overlaps")

## ----warning_6, eval = FALSE--------------------------------------------------
# warning(sum(workObj$MRCA_qual == "FAIL"), " segments did not conform to the mutation density at MRCA.")

## ----warning_7, eval = FALSE--------------------------------------------------
# warning("No sample name provided for samples x; sample name was set to 1 - y")

## ----warning_8, eval = FALSE--------------------------------------------------
# warning("No CNV/ SNV file provided for sample(s); sample(s) will be excluded")

## ----warning_9, eval = FALSE--------------------------------------------------
# warning("No output directory specified. Per-sample output will be discarded.")

## ----warning_10, eval = FALSE-------------------------------------------------
# warning("Insufficient data for sample ", x$ID)

## ----warning_11, eval = FALSE-------------------------------------------------
# warning("No column identifier  provided for samples x; column name will be inferred")

## ----warning_12, eval = FALSE-------------------------------------------------
# warning("Too few segments to estimate MRCA density for sample x.")

## ----warning_13, eval = FALSE-------------------------------------------------
# warning("Cannot produce summary statistics for a single case. Returning null.")

## ----warning_14, eval = FALSE-------------------------------------------------
# warning("Removing x samples with missing MRCA density estimate.")

## ----warning_15, eval = FALSE-------------------------------------------------
# warning("No sample with MRCA density estimate provided. Returning zero.")

## ----warning_16, eval = FALSE-------------------------------------------------
# warning("Removing x samples with missing survival time.")

## ----warning_17, eval = FALSE-------------------------------------------------
# warning("No survival events in cohort Returning zero.")

## ----sessionInfo--------------------------------------------------------------
sessionInfo()

