## ----setup--------------------------------------------------------------------
library(iscream)
data_dir <- system.file("extdata", package = "iscream")
bedfiles <- list.files(data_dir, pattern = "[a|b|c|d].bed.gz$", full.names = TRUE)
regions <- c(A = "chr1:1-6", B = "chr1:7-10", C = "chr1:11-14")

## ----gr, message=FALSE--------------------------------------------------------
if (!require("GenomicRanges", quietly = TRUE)) {
  stop("The 'GenomicRanges' package must be installed for this functionality")
}

## ----tabix_gr-----------------------------------------------------------------
gr <- GRanges(regions)
values(gr) <- DataFrame(
    gene = c("gene1", "gene2", "gene3"),
    some_metadata = c("s1", "s2", "s3")
)
gr
tabix_gr(bedfiles[1], gr)

## ----tabix--------------------------------------------------------------------
tabix(bedfiles[1], gr) |>
  makeGRangesFromDataFrame(
    starts.in.df.are.0based = TRUE,
    keep.extra.columns = TRUE
  )

## ----summarize_meth_regions---------------------------------------------------
(summary <- summarize_meth_regions(
  bedfiles,
  unname(regions),
  fun = c("sum", "mean"))
)
GRanges(summary$feature, summary = summary[, -1])

## ----summarize_regions--------------------------------------------------------
(summary <- summarize_regions(
  bedfiles,
  regions,
  column = 4,
  set_region_rownames = TRUE,
  fun = c("sum", "mean"))
)
GRanges(rownames(summary), summary = summary)

## ----make_mat-----------------------------------------------------------------
make_mat_gr(bedfiles, regions, column = 4, mat_name = "beta")

## ----mat_se, message=FALSE----------------------------------------------------
if (!require("SummarizedExperiment", quietly = TRUE)) {
  stop("The 'SummarizedExperiment' package must be installed for this functionality")
}
make_mat_se(bedfiles, regions, column = 4, mat_name = "beta", sparse = TRUE)

## ----message=FALSE------------------------------------------------------------
if (!require("bsseq", quietly = TRUE)) {
  stop("The 'bsseq' package must be installed for this functionality")
}
mats <- make_mat_bsseq(bedfiles, regions, sparse = FALSE)
do.call(BSseq, mats)

## ----si-----------------------------------------------------------------------
sessionInfo()

