# frozen_string_literal: true

module Gitlab
  module BackgroundMigration
    # Backfills owasp_top_10 column for vulnerability_reads table.
    class BackfillOwaspTopTenOfVulnerabilityReads < BatchedMigrationJob
      operation_name :set_owasp_top_10
      feature_category :vulnerability_management

      OWASP_TOP_10 = {
        "A1:2017-Injection" => 1,
        "A1:2017" => 1,
        "A2:2017-Broken Authentication" => 2,
        "A2:2017" => 2,
        "A3:2017-Sensitive Data Exposure" => 3,
        "A3:2017" => 3,
        "A4:2017-XML External Entities (XXE)" => 4,
        "A4:2017" => 4,
        "A5:2017-Broken Access Control" => 5,
        "A5:2017" => 5,
        "A6:2017-Security Misconfiguration" => 6,
        "A6:2017" => 6,
        "A7:2017-Cross-Site Scripting (XSS)" => 7,
        "A7:2017" => 7,
        "A8:2017-Insecure Deserialization" => 8,
        "A8:2017" => 8,
        "A9:2017-Using Components with Known Vulnerabilities" => 9,
        "A9:2017" => 9,
        "A10:2017-Insufficient Logging & Monitoring" => 10,
        "A10:2017" => 10,

        "A1:2021-Broken Access Control" => 11,
        "A1:2021" => 11,
        "A2:2021-Cryptographic Failures" => 12,
        "A2:2021" => 12,
        "A3:2021-Injection" => 13,
        "A3:2021" => 13,
        "A4:2021-Insecure Design" => 14,
        "A4:2021" => 14,
        "A5:2021-Security Misconfiguration" => 15,
        "A5:2021" => 15,
        "A6:2021-Vulnerable and Outdated Components" => 16,
        "A6:2021" => 16,
        "A7:2021-Identification and Authentication Failures" => 17,
        "A7:2021" => 17,
        "A8:2021-Software and Data Integrity Failures" => 18,
        "A8:2021" => 18,
        "A9:2021-Security Logging and Monitoring Failures" => 19,
        "A9:2021" => 19,
        "A10:2021-Server-Side Request Forgery" => 20,
        "A10:2021" => 20
      }.with_indifferent_access.freeze

      UPDATE_SQL = <<-SQL.squish
        UPDATE vulnerability_reads AS vr
        SET owasp_top_10 =
            CASE selected_ids.external_id
                #{OWASP_TOP_10.map { |external_id, value| "WHEN '#{external_id}' THEN #{value}" }.join(' ')}
                ELSE vr.owasp_top_10
            END
        FROM (
            SELECT vr.id, vi.external_id
            FROM vulnerability_reads vr
            INNER JOIN vulnerability_occurrences vo ON vr.vulnerability_id = vo.vulnerability_id
            INNER JOIN vulnerability_occurrence_identifiers voi ON vo.id = voi.occurrence_id
            INNER JOIN vulnerability_identifiers vi ON voi.identifier_id = vi.id
            WHERE LOWER(vi.external_type) = 'owasp'
            AND vi.external_id IN (?)
            AND vr.id IN (?)
        ) AS selected_ids
        WHERE vr.id = selected_ids.id
      SQL

      class VulnerabilitiesRead < ::ApplicationRecord
        self.table_name = 'vulnerability_reads'
      end

      def perform
        each_sub_batch do |sub_batch|
          update_query = VulnerabilitiesRead.sanitize_sql([UPDATE_SQL, OWASP_TOP_10.keys, sub_batch.select(:id)])
          connection.execute(update_query)
        end
      end
    end
  end
end
