# frozen_string_literal: true

require 'spec_helper'

RSpec.describe Issues::ResolveDiscussions, feature_category: :team_planning do
  let(:project) { create(:project, :repository) }
  let(:user) { create(:user) }

  before do
    stub_const('DummyService', Class.new(Issues::BaseService))
    DummyService.class_eval do
      include ::Issues::ResolveDiscussions

      def initialize(container:, current_user: nil, params: {})
        super
        filter_resolve_discussion_params
      end
    end

    project.add_developer(user)
  end

  describe "for resolving discussions" do
    let(:discussion) { create(:diff_note_on_merge_request, project: project, note: "Almost done").to_discussion }
    let(:merge_request) { discussion.noteable }
    let(:other_merge_request) { create(:merge_request, source_project: project, source_branch: "fix") }

    describe "#merge_request_for_resolving_discussion" do
      let(:service) { DummyService.new(container: project, current_user: user, params: { merge_request_to_resolve_discussions_of: merge_request.iid }) }

      it "finds the merge request" do
        expect(service.merge_request_to_resolve_discussions_of).to eq(merge_request)
      end

      it "only queries for the merge request once" do
        fake_finder = double

        expect(fake_finder).to receive(:find_by).exactly(1)
        expect(MergeRequestsFinder).to receive(:new).and_return(fake_finder).exactly(1)

        2.times { service.merge_request_to_resolve_discussions_of }
      end
    end

    describe "#discussions_to_resolve" do
      it "contains a single discussion when matching merge request and discussion are passed" do
        service = DummyService.new(
          container: project,
          current_user: user,
          params: {
            discussion_to_resolve: discussion.id,
            merge_request_to_resolve_discussions_of: merge_request.iid
          }
        )
        # We need to compare discussion id's because the Discussion-objects are rebuilt
        # which causes the object-id's not to be different.
        discussion_ids = service.discussions_to_resolve.map(&:id)

        expect(discussion_ids).to contain_exactly(discussion.id)
      end

      it "contains all discussions when only a merge request is passed" do
        second_discussion = Discussion.new([
          create(
            :diff_note_on_merge_request,
            noteable: merge_request,
            project: merge_request.target_project,
            line_number: 15
          )
        ])
        service = DummyService.new(
          container: project,
          current_user: user,
          params: { merge_request_to_resolve_discussions_of: merge_request.iid }
        )
        # We need to compare discussion id's because the Discussion-objects are rebuilt
        # which causes the object-id's not to be different.
        discussion_ids = service.discussions_to_resolve.map(&:id)

        expect(discussion_ids).to contain_exactly(discussion.id, second_discussion.id)
      end

      it "contains only unresolved discussions" do
        _second_discussion = Discussion.new([
          create(
            :diff_note_on_merge_request,
            :resolved,
            noteable: merge_request,
            project: merge_request.target_project,
            line_number: 15
          )
        ])
        service = DummyService.new(
          container: project,
          current_user: user,
          params: { merge_request_to_resolve_discussions_of: merge_request.iid }
        )
        # We need to compare discussion id's because the Discussion-objects are rebuilt
        # which causes the object-id's not to be different.
        discussion_ids = service.discussions_to_resolve.map(&:id)

        expect(discussion_ids).to contain_exactly(discussion.id)
      end

      it "is empty when a discussion and another merge request are passed" do
        service = DummyService.new(
          container: project,
          current_user: user,
          params: {
            discussion_to_resolve: discussion.id,
            merge_request_to_resolve_discussions_of: other_merge_request.iid
          }
        )

        expect(service.discussions_to_resolve).to be_empty
      end
    end
  end
end
