/**
 * Copyright (c) 2021, 2026 Contributors to the Eclipse Foundation
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.lsat.mpt.transformation;

import activity.Activity;
import activity.ActivitySet;
import activity.Claim;
import activity.util.ActivityParametersUtil;
import com.google.common.collect.Iterables;
import common.CommonFactory;
import common.HasName;
import common.Import;
import common.Parameter;
import common.ParameterReference;
import common.TypeDefinition;
import dispatching.ActivityDispatching;
import dispatching.Dispatch;
import dispatching.DispatchGroup;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Objects;
import java.util.function.Consumer;
import machine.IResource;
import machine.Resource;
import machine.ResourceItem;
import org.eclipse.emf.common.util.EList;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * Add a named Activity for all combinations of ResourceItems.
 * The original (unusable) activity is removed.
 */
@SuppressWarnings("all")
public final class ReduceActivityDispatching {
  private ReduceActivityDispatching() {
  }

  /**
   * Reduces expanded activities back to their original format stored on disk.
   * Activities refer to the supplied activitySet
   */
  public static void reduce(final ActivityDispatching activityDispatching, final ActivitySet activitySet) {
    final HashSet<Activity> allActivities = CollectionLiterals.<Activity>newHashSet();
    EList<Activity> _activities = activitySet.getActivities();
    Iterables.<Activity>addAll(allActivities, _activities);
    final Function1<ActivitySet, EList<Activity>> _function = (ActivitySet it) -> {
      return it.getActivities();
    };
    Iterable<Activity> _flatMap = IterableExtensions.<ActivitySet, Activity>flatMap(Iterables.<ActivitySet>filter(activitySet.loadAll(), ActivitySet.class), _function);
    Iterables.<Activity>addAll(allActivities, _flatMap);
    final LinkedHashMap<String, Activity> expandMap = CollectionLiterals.<String, Activity>newLinkedHashMap();
    final Consumer<Activity> _function_1 = (Activity a) -> {
      final Consumer<String> _function_2 = (String it) -> {
        expandMap.put(it, a);
      };
      ActivityParametersUtil.getAllExpandedNames(a).forEach(_function_2);
    };
    allActivities.forEach(_function_1);
    final Import importActivitySet = CommonFactory.eINSTANCE.createImport();
    importActivitySet.setImportURI(activitySet.eResource().getURI().toString());
    EList<Import> _imports = activityDispatching.getImports();
    _imports.add(importActivitySet);
    final Function1<DispatchGroup, EList<Dispatch>> _function_2 = (DispatchGroup it) -> {
      return it.getDispatches();
    };
    final Consumer<Dispatch> _function_3 = (Dispatch dis) -> {
      final Activity activity = expandMap.get(dis.getActivity().getName());
      final Collection<List<String>> parNames = ActivityParametersUtil.getParameterNames(activity.getName(), dis.getActivity().getName());
      final Function1<Claim, IResource> _function_4 = (Claim it) -> {
        return it.getResource();
      };
      final Function1<IResource, Boolean> _function_5 = (IResource it) -> {
        return Boolean.valueOf(((it instanceof ResourceItem) || (it instanceof Resource)));
      };
      List<IResource> resourceItemsFromActivity = IterableExtensions.<IResource>toList(IterableExtensions.<IResource>filter(IterableExtensions.<Claim, IResource>map(Iterables.<Claim>filter(dis.getActivity().getNodes(), Claim.class), _function_4), _function_5));
      final List<TypeDefinition> declarations = ActivityParametersUtil.getDeclarations(activity);
      int _size = parNames.size();
      int _size_1 = declarations.size();
      boolean _notEquals = (_size != _size_1);
      if (_notEquals) {
        throw new RuntimeException("Cannot convert expanded activity");
      }
      final Iterator<TypeDefinition> iter = declarations.iterator();
      final List<IResource> resources = resourceItemsFromActivity;
      for (final List<String> parName : parNames) {
        {
          final Parameter parameter = CommonFactory.eINSTANCE.createParameter();
          final TypeDefinition declaration = iter.next();
          for (int i = 0; (i < parName.size()); i++) {
            {
              final Function1<ParameterReference, HasName> _function_6 = (ParameterReference it) -> {
                return it.getReference();
              };
              final HasName namedReference = ReduceActivityDispatching.findItem(activity, resources, declaration, parName.get(i), ListExtensions.<ParameterReference, HasName>map(parameter.getReferences(), _function_6));
              if ((namedReference instanceof ResourceItem)) {
                resources.add(((IResource)namedReference));
              }
              if ((namedReference == null)) {
                String _get = parName.get(i);
                String _plus = ("Cannot find item with name " + _get);
                throw new RuntimeException(_plus);
              }
              ParameterReference ref = CommonFactory.eINSTANCE.createParameterReference();
              ref.setReference(namedReference);
              EList<ParameterReference> _references = parameter.getReferences();
              _references.add(ref);
            }
          }
          EList<Parameter> _parameters = dis.getParameters();
          _parameters.add(parameter);
        }
      }
      dis.setActivity(activity);
    };
    IterableExtensions.<DispatchGroup, Dispatch>flatMap(activityDispatching.getDispatchGroups(), _function_2).forEach(_function_3);
    EList<Import> _imports_1 = activityDispatching.getImports();
    _imports_1.remove(importActivitySet);
    final Function1<Import, Boolean> _function_4 = (Import it) -> {
      return Boolean.valueOf(it.getImportURI().contains(".activity"));
    };
    Import _findFirst = IterableExtensions.<Import>findFirst(activityDispatching.getImports(), _function_4);
    _findFirst.setImportURI(activitySet.eResource().getURI().toString());
  }

  /**
   * returns the model item for itemName base on the type definition of parameter
   */
  private static HasName findItem(final Activity activity, final List<? extends IResource> resources, final TypeDefinition declaration, final String itemName, final List<? extends HasName> current) {
    final Collection<? extends HasName> candidates = ActivityParametersUtil.findCandidate(activity, resources, declaration, current);
    final Function1<HasName, Boolean> _function = (HasName it) -> {
      String _name = it.getName();
      return Boolean.valueOf(Objects.equals(_name, itemName));
    };
    return IterableExtensions.findFirst(candidates, _function);
  }

  /**
   * returns a map with key all possible activity names and activity as values
   */
  private static LinkedHashMap<String, Activity> expandNames(final List<Activity> activities) {
    return ExpandActivities.expandNames(activities);
  }
}
