/*
 * Decompiled with CFR 0.152.
 */
package org.mapdb;

import java.io.DataInput;
import java.io.IOError;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Iterator;
import java.util.List;
import java.util.Queue;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.logging.Logger;
import java.util.zip.CRC32;
import org.mapdb.CompressLZF;
import org.mapdb.DB;
import org.mapdb.DataInput2;
import org.mapdb.DataOutput2;
import org.mapdb.EncryptionXTEA;
import org.mapdb.Engine;
import org.mapdb.EngineWrapper;
import org.mapdb.Serializer;
import org.mapdb.SerializerPojo;
import org.mapdb.TxEngine;

public abstract class Store
implements Engine {
    protected static final Logger LOG = Logger.getLogger(Store.class.getName());
    protected final boolean checksum;
    protected final boolean compress;
    protected final boolean encrypt;
    protected final byte[] password;
    protected final EncryptionXTEA encryptionXTEA;
    protected static final int CHECKSUM_FLAG_MASK = 1;
    protected static final int COMPRESS_FLAG_MASK = 4;
    protected static final int ENCRYPT_FLAG_MASK = 8;
    protected static final int CHUNK_SIZE = 0x100000;
    protected static final int CHUNK_SIZE_MOD_MASK = 1048575;
    protected SerializerPojo serializerPojo;
    protected final ThreadLocal<CompressLZF> LZF;
    protected Lock serializerPojoInitLock = new ReentrantLock(false);
    protected final ReentrantLock structuralLock = new ReentrantLock(false);
    protected final ReentrantReadWriteLock newRecidLock = new ReentrantReadWriteLock(false);
    protected final ReentrantReadWriteLock[] locks = new ReentrantReadWriteLock[128];
    protected final Queue<DataOutput2> recycledDataOuts;
    private static final int LOCK_MASK = 127;
    List<Runnable> closeListeners;

    protected Store(boolean checksum, boolean compress, byte[] password, boolean disableLocks) {
        for (int i = 0; i < this.locks.length; ++i) {
            this.locks[i] = new ReentrantReadWriteLock(false);
        }
        this.recycledDataOuts = new ArrayBlockingQueue<DataOutput2>(128);
        this.closeListeners = new CopyOnWriteArrayList<Runnable>();
        this.checksum = checksum;
        this.compress = compress;
        this.encrypt = password != null;
        this.password = password;
        this.encryptionXTEA = !this.encrypt ? null : new EncryptionXTEA(password);
        this.LZF = !compress ? null : new ThreadLocal<CompressLZF>(){

            @Override
            protected CompressLZF initialValue() {
                return new CompressLZF();
            }
        };
    }

    public abstract long getMaxRecid();

    public abstract ByteBuffer getRaw(long var1);

    public abstract Iterator<Long> getFreeRecids();

    public abstract void updateRaw(long var1, ByteBuffer var3);

    public abstract long getSizeLimit();

    public abstract long getCurrSize();

    public abstract long getFreeSize();

    public abstract String calculateStatistics();

    public void printStatistics() {
        System.out.println(this.calculateStatistics());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public SerializerPojo getSerializerPojo() {
        Lock pojoLock = this.serializerPojoInitLock;
        if (pojoLock != null) {
            pojoLock.lock();
            try {
                if (this.serializerPojo == null) {
                    CopyOnWriteArrayList<SerializerPojo.ClassInfo> classInfos = this.get(2L, SerializerPojo.serializer);
                    this.serializerPojo = new SerializerPojo(classInfos);
                    this.serializerPojoInitLock = null;
                }
            }
            finally {
                pojoLock.unlock();
            }
        }
        return this.serializerPojo;
    }

    protected void lockAllWrite() {
        this.newRecidLock.writeLock().lock();
        for (ReentrantReadWriteLock l : this.locks) {
            l.writeLock().lock();
        }
        this.structuralLock.lock();
    }

    protected void unlockAllWrite() {
        this.structuralLock.unlock();
        for (ReentrantReadWriteLock l : this.locks) {
            l.writeLock().unlock();
        }
        this.newRecidLock.writeLock().unlock();
    }

    protected <A> DataOutput2 serialize(A value, Serializer<A> serializer) {
        try {
            DataOutput2 out = this.newDataOut2();
            serializer.serialize(out, value);
            if (out.pos > 0) {
                if (this.compress) {
                    int newLen;
                    DataOutput2 tmp = this.newDataOut2();
                    tmp.ensureAvail(out.pos + 40);
                    CompressLZF lzf = this.LZF.get();
                    try {
                        newLen = lzf.compress(out.buf, out.pos, tmp.buf, 0);
                    }
                    catch (IndexOutOfBoundsException e) {
                        newLen = 0;
                    }
                    if (newLen >= out.pos) {
                        newLen = 0;
                    }
                    if (newLen == 0) {
                        this.recycledDataOuts.offer(tmp);
                        out.ensureAvail(out.pos + 1);
                        System.arraycopy(out.buf, 0, out.buf, 1, out.pos);
                        ++out.pos;
                        out.buf[0] = 0;
                    } else {
                        int decompSize = out.pos;
                        out.pos = 0;
                        DataOutput2.packInt(out, decompSize);
                        out.write(tmp.buf, 0, newLen);
                        this.recycledDataOuts.offer(tmp);
                    }
                }
                if (this.encrypt) {
                    int size = out.pos;
                    if (size % 16 != 0) {
                        size += 16 - size % 16;
                    }
                    int sizeDif = size - out.pos;
                    out.ensureAvail(sizeDif + 1);
                    this.encryptionXTEA.encrypt(out.buf, 0, size);
                    out.pos = size;
                    out.writeByte(sizeDif);
                }
                if (this.checksum) {
                    CRC32 crc = new CRC32();
                    crc.update(out.buf, 0, out.pos);
                    out.writeInt((int)crc.getValue());
                }
            }
            return out;
        }
        catch (IOException e) {
            throw new IOError(e);
        }
    }

    protected DataOutput2 newDataOut2() {
        DataOutput2 tmp = this.recycledDataOuts.poll();
        if (tmp == null) {
            tmp = new DataOutput2();
        } else {
            tmp.pos = 0;
        }
        return tmp;
    }

    protected <A> A deserialize(Serializer<A> serializer, int size, DataInput input) throws IOException {
        DataInput2 di = (DataInput2)input;
        if (size > 0) {
            DataOutput2 tmp;
            if (this.checksum) {
                tmp = this.newDataOut2();
                tmp.ensureAvail(size -= 4);
                int oldPos = di.pos;
                di.read(tmp.buf, 0, size);
                di.pos = oldPos;
                CRC32 crc = new CRC32();
                crc.update(tmp.buf, 0, size);
                this.recycledDataOuts.offer(tmp);
                int check = (int)crc.getValue();
                int checkExpected = di.buf.getInt(di.pos + size);
                if (check != checkExpected) {
                    throw new IOException("Checksum does not match, data broken");
                }
            }
            if (this.encrypt) {
                tmp = this.newDataOut2();
                tmp.ensureAvail(--size);
                di.read(tmp.buf, 0, size);
                this.encryptionXTEA.decrypt(tmp.buf, 0, size);
                int cut = di.readUnsignedByte();
                di = new DataInput2(tmp.buf);
                size -= cut;
            }
            if (this.compress) {
                int decompSize = DataInput2.unpackInt(di);
                if (decompSize == 0) {
                    --size;
                } else {
                    DataOutput2 out = this.newDataOut2();
                    out.ensureAvail(decompSize);
                    CompressLZF lzf = this.LZF.get();
                    lzf.expand(di.buf, di.pos, out.buf, 0, decompSize);
                    di = new DataInput2(out.buf);
                    size = decompSize;
                }
            }
        }
        int start = di.pos;
        A ret = serializer.deserialize(di, size);
        if (size + start > di.pos) {
            throw new AssertionError((Object)"data were not fully read, check your serializer ");
        }
        if (size + start < di.pos) {
            throw new AssertionError((Object)"data were read beyond record size, check your serializer");
        }
        return ret;
    }

    public static Store forDB(DB db) {
        return Store.forEngine(db.engine);
    }

    public static Store forEngine(Engine e) {
        if (e instanceof EngineWrapper) {
            return Store.forEngine(((EngineWrapper)e).getWrappedEngine());
        }
        if (e instanceof TxEngine.Tx) {
            return Store.forEngine(((TxEngine.Tx)e).getWrappedEngine());
        }
        return (Store)e;
    }

    protected int expectedMasks() {
        return (this.encrypt ? 8 : 0) | (this.checksum ? 1 : 0) | (this.compress ? 4 : 0);
    }

    protected static int lockPos(long key) {
        int h = (int)(key ^ key >>> 32);
        h ^= h >>> 20 ^ h >>> 12;
        h ^= h >>> 7 ^ h >>> 4;
        return h & 0x7F;
    }

    @Override
    public boolean canSnapshot() {
        return false;
    }

    @Override
    public Engine snapshot() throws UnsupportedOperationException {
        throw new UnsupportedOperationException("Snapshots are not supported");
    }

    @Override
    public void closeListenerRegister(Runnable closeListener) {
        this.closeListeners.add(closeListener);
    }

    @Override
    public void closeListenerUnregister(Runnable closeListener) {
        this.closeListeners.remove(closeListener);
    }
}

