/*******************************************************************************
 * Copyright (c) 2002, 2017 IBM Corporation and others.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     xored software, Inc. - initial API and Implementation
 *******************************************************************************/
package org.eclipse.dltk.ast.declarations;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.dltk.ast.ASTNode;
import org.eclipse.dltk.ast.ASTVisitor;
import org.eclipse.dltk.ast.parser.IModuleDeclaration;
import org.eclipse.dltk.ast.statements.Block;
import org.eclipse.dltk.ast.statements.Statement;
import org.eclipse.dltk.ast.utils.ASTUtil;
import org.eclipse.dltk.internal.compiler.lookup.SourceModuleScope;
import org.eclipse.dltk.utils.CorePrinter;

public class ModuleDeclaration extends ASTNode implements IModuleDeclaration {
	private List types;
	private List functions;
	protected List variables;

	private Block body;

	private boolean rebuildEnabled;

	// TODO: make private
	public SourceModuleScope scope;

	public List getTypeList() {
		return this.types;
	}

	public List getFunctionList() {
		return this.functions;
	}

	public List getVariablesList() {
		return this.variables;
	}

	public ModuleDeclaration(int sourceLength) {
		this(sourceLength, false);
	}

	public ModuleDeclaration(int sourceLength, boolean rebuildEnabled) {
		super(0, sourceLength);

		this.body = new Block();
		this.body.setEnd(sourceLength);
		this.types = new ArrayList();
		this.functions = new ArrayList();
		this.variables = new ArrayList();

		this.rebuildEnabled = rebuildEnabled;
	}

	@Override
	public final void traverse(ASTVisitor visitor) throws Exception {
		if (visitor.visit(this)) {
			this.body.traverse(visitor);
			visitor.endvisit(this);
		}
	}

	public void setStatements(List<ASTNode> statements) {
		this.body = new Block(this.sourceStart(), this.sourceEnd(), statements);
	}

	public void addStatement(ASTNode statement) {
		this.body.addStatement(statement);
	}

	public List<ASTNode> getStatements() {
		return this.body.getStatements();
	}

	@Override
	public void printNode(CorePrinter output) {
		output.formatPrintLn("Module" + this.getSourceRange().toString() + ":"); //$NON-NLS-1$ //$NON-NLS-2$
		this.body.printNode(output);
	}

	public boolean isEmpty() {
		return this.body.getStatements().isEmpty();
	}

	protected void doRebuild() {
		// Iterator i = this.getStatements().iterator();
		// while( i.hasNext()) {
		// ASTNode node = (ASTNode)i.next();
		// if( node instanceof MethodDeclaration ) {
		// this.functions.add( node );
		// }
		// else if( node instanceof TypeDeclaration ) {
		// this.types.add(node);
		// }
		// }
	}

	public final void rebuild() {
		if (this.rebuildEnabled) {
			this.doRebuild();
		}
	}

	protected boolean isRebuildEnabled() {
		return this.rebuildEnabled;
	}

	public void disableRebuild() {
		this.rebuildEnabled = false;
	}

	public TypeDeclaration[] getTypes() {
		return ASTUtil.getTypes(this.getStatements(), this.types);
	}

	public MethodDeclaration[] getFunctions() {
		return ASTUtil.getMethods(this.getStatements(), this.functions);
	}

	public FieldDeclaration[] getVariables() {
		return ASTUtil.getVariables(this.getStatements(), this.variables );
	}

	public ASTNode[] getNonTypeOrMethodNode() {
		List<ASTNode> statements = this.getStatements();
		List<ASTNode> results = new ArrayList<>();
		if (statements != null) {
			Iterator<ASTNode> it = statements.iterator();
			while (it.hasNext()) {
				ASTNode node = it.next();
				if (!(node instanceof TypeDeclaration)
						&& !(node instanceof MethodDeclaration)) {
					results.add(node);
				}
			}
		}
		return results.toArray(new ASTNode[results.size()]);
	}

	@Override
	public void setEnd(int end) {
		super.setEnd(end);
		this.body.setEnd(end);
	}

	@Override
	public void setStart(int start) {
		super.setStart(start);
		this.body.setStart(start);
	}
	public void rebuildMethods() {
	}

	public void removeStatement(Statement node) {
		this.body.removeStatement(node);
	}
}
