"use strict";
// *****************************************************************************
// Copyright (C) 2017 TypeFox and others.
//
// This program and the accompanying materials are made available under the
// terms of the Eclipse Public License v. 2.0 which is available at
// http://www.eclipse.org/legal/epl-2.0.
//
// This Source Code may also be made available under the following Secondary
// Licenses when the conditions for such availability set forth in the Eclipse
// Public License v. 2.0 are satisfied: GNU General Public License, version 2
// with the GNU Classpath Exception which is available at
// https://www.gnu.org/software/classpath/license.html.
//
// SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
// *****************************************************************************
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var __param = (this && this.__param) || function (paramIndex, decorator) {
    return function (target, key) { decorator(target, key, paramIndex); }
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.CommonFrontendContribution = exports.RECENT_COMMANDS_STORAGE_KEY = exports.supportPaste = exports.supportCopy = exports.supportCut = exports.CommonCommands = exports.CommonMenus = void 0;
/* eslint-disable max-len, @typescript-eslint/indent */
const debounce = require("lodash.debounce");
const inversify_1 = require("inversify");
const menu_1 = require("../common/menu");
const command_1 = require("../common/command");
const uri_command_handler_1 = require("../common/uri-command-handler");
const selection_service_1 = require("../common/selection-service");
const message_service_1 = require("../common/message-service");
const opener_service_1 = require("../browser/opener-service");
const application_shell_1 = require("./shell/application-shell");
const tab_bars_1 = require("./shell/tab-bars");
const about_dialog_1 = require("./about-dialog");
const browser = require("./browser");
const uri_1 = require("../common/uri");
const context_key_service_1 = require("./context-key-service");
const os_1 = require("../common/os");
const resource_context_key_1 = require("./resource-context-key");
const selection_1 = require("../common/selection");
const storage_service_1 = require("./storage-service");
const navigatable_1 = require("./navigatable");
const quick_view_service_1 = require("./quick-input/quick-view-service");
const environment_1 = require("@theia/application-package/lib/environment");
const icon_theme_service_1 = require("./icon-theme-service");
const color_registry_1 = require("./color-registry");
const core_preferences_1 = require("./core-preferences");
const theming_1 = require("./theming");
const preferences_1 = require("./preferences");
const clipboard_service_1 = require("./clipboard-service");
const encoding_registry_1 = require("./encoding-registry");
const encodings_1 = require("../common/encodings");
const env_variables_1 = require("../common/env-variables");
const authentication_service_1 = require("./authentication-service");
const saveable_1 = require("./saveable");
const quick_input_1 = require("./quick-input");
const localization_1 = require("../common/i18n/localization");
const nls_1 = require("../common/nls");
const current_widget_command_adapter_1 = require("./shell/current-widget-command-adapter");
const dialogs_1 = require("./dialogs");
const window_service_1 = require("./window/window-service");
const frontend_application_config_provider_1 = require("./frontend-application-config-provider");
const decoration_style_1 = require("./decoration-style");
const widgets_1 = require("./widgets");
const save_resource_service_1 = require("./save-resource-service");
var CommonMenus;
(function (CommonMenus) {
    CommonMenus.FILE = [...menu_1.MAIN_MENU_BAR, '1_file'];
    CommonMenus.FILE_NEW = [...CommonMenus.FILE, '1_new'];
    CommonMenus.FILE_OPEN = [...CommonMenus.FILE, '2_open'];
    CommonMenus.FILE_SAVE = [...CommonMenus.FILE, '3_save'];
    CommonMenus.FILE_AUTOSAVE = [...CommonMenus.FILE, '4_autosave'];
    CommonMenus.FILE_SETTINGS = [...CommonMenus.FILE, '5_settings'];
    CommonMenus.FILE_SETTINGS_SUBMENU = [...CommonMenus.FILE_SETTINGS, '1_settings_submenu'];
    CommonMenus.FILE_SETTINGS_SUBMENU_OPEN = [...CommonMenus.FILE_SETTINGS_SUBMENU, '1_settings_submenu_open'];
    CommonMenus.FILE_SETTINGS_SUBMENU_THEME = [...CommonMenus.FILE_SETTINGS_SUBMENU, '2_settings_submenu_theme'];
    CommonMenus.FILE_CLOSE = [...CommonMenus.FILE, '6_close'];
    CommonMenus.EDIT = [...menu_1.MAIN_MENU_BAR, '2_edit'];
    CommonMenus.EDIT_UNDO = [...CommonMenus.EDIT, '1_undo'];
    CommonMenus.EDIT_CLIPBOARD = [...CommonMenus.EDIT, '2_clipboard'];
    CommonMenus.EDIT_FIND = [...CommonMenus.EDIT, '3_find'];
    CommonMenus.VIEW = [...menu_1.MAIN_MENU_BAR, '4_view'];
    CommonMenus.VIEW_PRIMARY = [...CommonMenus.VIEW, '0_primary'];
    CommonMenus.VIEW_APPEARANCE = [...CommonMenus.VIEW, '1_appearance'];
    CommonMenus.VIEW_APPEARANCE_SUBMENU = [...CommonMenus.VIEW_APPEARANCE, '1_appearance_submenu'];
    CommonMenus.VIEW_APPEARANCE_SUBMENU_SCREEN = [...CommonMenus.VIEW_APPEARANCE_SUBMENU, '2_appearance_submenu_screen'];
    CommonMenus.VIEW_APPEARANCE_SUBMENU_BAR = [...CommonMenus.VIEW_APPEARANCE_SUBMENU, '3_appearance_submenu_bar'];
    CommonMenus.VIEW_EDITOR_SUBMENU = [...CommonMenus.VIEW_APPEARANCE, '2_editor_submenu'];
    CommonMenus.VIEW_EDITOR_SUBMENU_SPLIT = [...CommonMenus.VIEW_EDITOR_SUBMENU, '1_editor_submenu_split'];
    CommonMenus.VIEW_EDITOR_SUBMENU_ORTHO = [...CommonMenus.VIEW_EDITOR_SUBMENU, '2_editor_submenu_ortho'];
    CommonMenus.VIEW_VIEWS = [...CommonMenus.VIEW, '2_views'];
    CommonMenus.VIEW_LAYOUT = [...CommonMenus.VIEW, '3_layout'];
    CommonMenus.VIEW_TOGGLE = [...CommonMenus.VIEW, '4_toggle'];
    CommonMenus.SETTINGS_OPEN = [...menu_1.SETTINGS_MENU, '1_settings_open'];
    CommonMenus.SETTINGS__THEME = [...menu_1.SETTINGS_MENU, '2_settings_theme'];
    // last menu item
    CommonMenus.HELP = [...menu_1.MAIN_MENU_BAR, '9_help'];
})(CommonMenus = exports.CommonMenus || (exports.CommonMenus = {}));
var CommonCommands;
(function (CommonCommands) {
    CommonCommands.FILE_CATEGORY = 'File';
    CommonCommands.VIEW_CATEGORY = 'View';
    CommonCommands.PREFERENCES_CATEGORY = 'Preferences';
    CommonCommands.FILE_CATEGORY_KEY = nls_1.nls.getDefaultKey(CommonCommands.FILE_CATEGORY);
    CommonCommands.VIEW_CATEGORY_KEY = nls_1.nls.getDefaultKey(CommonCommands.VIEW_CATEGORY);
    CommonCommands.PREFERENCES_CATEGORY_KEY = nls_1.nls.getDefaultKey(CommonCommands.PREFERENCES_CATEGORY);
    CommonCommands.OPEN = {
        id: 'core.open',
    };
    CommonCommands.CUT = command_1.Command.toDefaultLocalizedCommand({
        id: 'core.cut',
        label: 'Cut'
    });
    CommonCommands.COPY = command_1.Command.toDefaultLocalizedCommand({
        id: 'core.copy',
        label: 'Copy'
    });
    CommonCommands.PASTE = command_1.Command.toDefaultLocalizedCommand({
        id: 'core.paste',
        label: 'Paste'
    });
    CommonCommands.COPY_PATH = command_1.Command.toDefaultLocalizedCommand({
        id: 'core.copy.path',
        label: 'Copy Path'
    });
    CommonCommands.UNDO = command_1.Command.toDefaultLocalizedCommand({
        id: 'core.undo',
        label: 'Undo'
    });
    CommonCommands.REDO = command_1.Command.toDefaultLocalizedCommand({
        id: 'core.redo',
        label: 'Redo'
    });
    CommonCommands.SELECT_ALL = command_1.Command.toDefaultLocalizedCommand({
        id: 'core.selectAll',
        label: 'Select All'
    });
    CommonCommands.FIND = command_1.Command.toDefaultLocalizedCommand({
        id: 'core.find',
        label: 'Find'
    });
    CommonCommands.REPLACE = command_1.Command.toDefaultLocalizedCommand({
        id: 'core.replace',
        label: 'Replace'
    });
    CommonCommands.NEXT_TAB = command_1.Command.toDefaultLocalizedCommand({
        id: 'core.nextTab',
        category: CommonCommands.VIEW_CATEGORY,
        label: 'Show Next Tab'
    });
    CommonCommands.PREVIOUS_TAB = command_1.Command.toDefaultLocalizedCommand({
        id: 'core.previousTab',
        category: CommonCommands.VIEW_CATEGORY,
        label: 'Show Previous Tab'
    });
    CommonCommands.NEXT_TAB_IN_GROUP = command_1.Command.toLocalizedCommand({
        id: 'core.nextTabInGroup',
        category: CommonCommands.VIEW_CATEGORY,
        label: 'Switch to Next Tab in Group'
    }, 'theia/core/common/showNextTabInGroup', CommonCommands.VIEW_CATEGORY_KEY);
    CommonCommands.PREVIOUS_TAB_IN_GROUP = command_1.Command.toLocalizedCommand({
        id: 'core.previousTabInGroup',
        category: CommonCommands.VIEW_CATEGORY,
        label: 'Switch to Previous Tab in Group'
    }, 'theia/core/common/showPreviousTabInGroup', CommonCommands.VIEW_CATEGORY_KEY);
    CommonCommands.NEXT_TAB_GROUP = command_1.Command.toLocalizedCommand({
        id: 'core.nextTabGroup',
        category: CommonCommands.VIEW_CATEGORY,
        label: 'Switch to Next Tab Group'
    }, 'theia/core/common/showNextTabGroup', CommonCommands.VIEW_CATEGORY_KEY);
    CommonCommands.PREVIOUS_TAB_GROUP = command_1.Command.toLocalizedCommand({
        id: 'core.previousTabBar',
        category: CommonCommands.VIEW_CATEGORY,
        label: 'Switch to Previous Tab Group'
    }, 'theia/core/common/showPreviousTabGroup', CommonCommands.VIEW_CATEGORY_KEY);
    CommonCommands.CLOSE_TAB = command_1.Command.toLocalizedCommand({
        id: 'core.close.tab',
        category: CommonCommands.VIEW_CATEGORY,
        label: 'Close Tab'
    }, 'theia/core/common/closeTab', CommonCommands.VIEW_CATEGORY_KEY);
    CommonCommands.CLOSE_OTHER_TABS = command_1.Command.toLocalizedCommand({
        id: 'core.close.other.tabs',
        category: CommonCommands.VIEW_CATEGORY,
        label: 'Close Other Tabs'
    }, 'theia/core/common/closeOthers', CommonCommands.VIEW_CATEGORY_KEY);
    CommonCommands.CLOSE_SAVED_TABS = command_1.Command.toDefaultLocalizedCommand({
        id: 'workbench.action.closeUnmodifiedEditors',
        category: CommonCommands.VIEW_CATEGORY,
        label: 'Close Saved Editors in Group',
    });
    CommonCommands.CLOSE_RIGHT_TABS = command_1.Command.toLocalizedCommand({
        id: 'core.close.right.tabs',
        category: CommonCommands.VIEW_CATEGORY,
        label: 'Close Tabs to the Right'
    }, 'theia/core/common/closeRight', CommonCommands.VIEW_CATEGORY_KEY);
    CommonCommands.CLOSE_ALL_TABS = command_1.Command.toLocalizedCommand({
        id: 'core.close.all.tabs',
        category: CommonCommands.VIEW_CATEGORY,
        label: 'Close All Tabs'
    }, 'theia/core/common/closeAll', CommonCommands.VIEW_CATEGORY_KEY);
    CommonCommands.CLOSE_MAIN_TAB = command_1.Command.toLocalizedCommand({
        id: 'core.close.main.tab',
        category: CommonCommands.VIEW_CATEGORY,
        label: 'Close Tab in Main Area'
    }, 'theia/core/common/closeTabMain', CommonCommands.VIEW_CATEGORY_KEY);
    CommonCommands.CLOSE_OTHER_MAIN_TABS = command_1.Command.toLocalizedCommand({
        id: 'core.close.other.main.tabs',
        category: CommonCommands.VIEW_CATEGORY,
        label: 'Close Other Tabs in Main Area'
    }, 'theia/core/common/closeOtherTabMain', CommonCommands.VIEW_CATEGORY_KEY);
    CommonCommands.CLOSE_ALL_MAIN_TABS = command_1.Command.toLocalizedCommand({
        id: 'core.close.all.main.tabs',
        category: CommonCommands.VIEW_CATEGORY,
        label: 'Close All Tabs in Main Area'
    }, 'theia/core/common/closeAllTabMain', CommonCommands.VIEW_CATEGORY_KEY);
    CommonCommands.COLLAPSE_PANEL = command_1.Command.toLocalizedCommand({
        id: 'core.collapse.tab',
        category: CommonCommands.VIEW_CATEGORY,
        label: 'Collapse Side Panel'
    }, 'theia/core/common/collapseTab', CommonCommands.VIEW_CATEGORY_KEY);
    CommonCommands.COLLAPSE_ALL_PANELS = command_1.Command.toLocalizedCommand({
        id: 'core.collapse.all.tabs',
        category: CommonCommands.VIEW_CATEGORY,
        label: 'Collapse All Side Panels'
    }, 'theia/core/common/collapseAllTabs', CommonCommands.VIEW_CATEGORY_KEY);
    CommonCommands.TOGGLE_BOTTOM_PANEL = command_1.Command.toLocalizedCommand({
        id: 'core.toggle.bottom.panel',
        category: CommonCommands.VIEW_CATEGORY,
        label: 'Toggle Bottom Panel'
    }, 'theia/core/common/collapseBottomPanel', CommonCommands.VIEW_CATEGORY_KEY);
    CommonCommands.TOGGLE_STATUS_BAR = command_1.Command.toDefaultLocalizedCommand({
        id: 'workbench.action.toggleStatusbarVisibility',
        category: CommonCommands.VIEW_CATEGORY,
        label: 'Toggle Status Bar Visibility'
    });
    CommonCommands.PIN_TAB = command_1.Command.toDefaultLocalizedCommand({
        id: 'workbench.action.pinEditor',
        category: CommonCommands.VIEW_CATEGORY,
        label: 'Pin Editor'
    });
    CommonCommands.UNPIN_TAB = command_1.Command.toDefaultLocalizedCommand({
        id: 'workbench.action.unpinEditor',
        category: CommonCommands.VIEW_CATEGORY,
        label: 'Unpin Editor'
    });
    CommonCommands.TOGGLE_MAXIMIZED = command_1.Command.toLocalizedCommand({
        id: 'core.toggleMaximized',
        category: CommonCommands.VIEW_CATEGORY,
        label: 'Toggle Maximized'
    }, 'theia/core/common/toggleMaximized', CommonCommands.VIEW_CATEGORY_KEY);
    CommonCommands.OPEN_VIEW = command_1.Command.toDefaultLocalizedCommand({
        id: 'core.openView',
        category: CommonCommands.VIEW_CATEGORY,
        label: 'Open View...'
    });
    CommonCommands.SHOW_MENU_BAR = command_1.Command.toDefaultLocalizedCommand({
        id: 'window.menuBarVisibility',
        category: CommonCommands.VIEW_CATEGORY,
        label: 'Show Menu Bar'
    });
    CommonCommands.SAVE = command_1.Command.toDefaultLocalizedCommand({
        id: 'core.save',
        category: CommonCommands.FILE_CATEGORY,
        label: 'Save',
    });
    CommonCommands.SAVE_WITHOUT_FORMATTING = command_1.Command.toDefaultLocalizedCommand({
        id: 'core.saveWithoutFormatting',
        category: CommonCommands.FILE_CATEGORY,
        label: 'Save without Formatting',
    });
    CommonCommands.SAVE_ALL = command_1.Command.toDefaultLocalizedCommand({
        id: 'core.saveAll',
        category: CommonCommands.FILE_CATEGORY,
        label: 'Save All',
    });
    CommonCommands.AUTO_SAVE = command_1.Command.toDefaultLocalizedCommand({
        id: 'textEditor.commands.autosave',
        category: CommonCommands.FILE_CATEGORY,
        label: 'Auto Save',
    });
    CommonCommands.ABOUT_COMMAND = command_1.Command.toDefaultLocalizedCommand({
        id: 'core.about',
        label: 'About'
    });
    CommonCommands.OPEN_PREFERENCES = command_1.Command.toDefaultLocalizedCommand({
        id: 'preferences:open',
        category: CommonCommands.PREFERENCES_CATEGORY,
        label: 'Open Settings (UI)',
    });
    CommonCommands.SELECT_COLOR_THEME = command_1.Command.toDefaultLocalizedCommand({
        id: 'workbench.action.selectTheme',
        label: 'Color Theme',
        category: CommonCommands.PREFERENCES_CATEGORY
    });
    CommonCommands.SELECT_ICON_THEME = command_1.Command.toDefaultLocalizedCommand({
        id: 'workbench.action.selectIconTheme',
        label: 'File Icon Theme',
        category: CommonCommands.PREFERENCES_CATEGORY
    });
    CommonCommands.CONFIGURE_DISPLAY_LANGUAGE = command_1.Command.toDefaultLocalizedCommand({
        id: 'workbench.action.configureLanguage',
        label: 'Configure Display Language'
    });
})(CommonCommands = exports.CommonCommands || (exports.CommonCommands = {}));
exports.supportCut = browser.isNative || document.queryCommandSupported('cut');
exports.supportCopy = browser.isNative || document.queryCommandSupported('copy');
// Chrome incorrectly returns true for document.queryCommandSupported('paste')
// when the paste feature is available but the calling script has insufficient
// privileges to actually perform the action
exports.supportPaste = browser.isNative || (!browser.isChrome && document.queryCommandSupported('paste'));
exports.RECENT_COMMANDS_STORAGE_KEY = 'commands';
let CommonFrontendContribution = class CommonFrontendContribution {
    constructor(shell, selectionService, messageService, openerService, aboutDialog, localizationProvider, saveResourceService) {
        this.shell = shell;
        this.selectionService = selectionService;
        this.messageService = messageService;
        this.openerService = openerService;
        this.aboutDialog = aboutDialog;
        this.localizationProvider = localizationProvider;
        this.saveResourceService = saveResourceService;
        this.commonDecorationsStyleSheet = decoration_style_1.DecorationStyle.createStyleSheet('coreCommonDecorationsStyle');
        this.shouldPreventClose = false;
    }
    async configure(app) {
        const configDirUri = await this.environments.getConfigDirUri();
        // Global settings
        this.encodingRegistry.registerOverride({
            encoding: encodings_1.UTF8,
            parent: new uri_1.default(configDirUri)
        });
        this.contextKeyService.createKey('isLinux', os_1.OS.type() === os_1.OS.Type.Linux);
        this.contextKeyService.createKey('isMac', os_1.OS.type() === os_1.OS.Type.OSX);
        this.contextKeyService.createKey('isWindows', os_1.OS.type() === os_1.OS.Type.Windows);
        this.contextKeyService.createKey('isWeb', !this.isElectron());
        this.pinnedKey = this.contextKeyService.createKey('activeEditorIsPinned', false);
        this.updatePinnedKey();
        this.shell.onDidChangeActiveWidget(() => this.updatePinnedKey());
        this.initResourceContextKeys();
        this.registerCtrlWHandling();
        this.updateStyles();
        this.updateThemeFromPreference('workbench.colorTheme');
        this.updateThemeFromPreference('workbench.iconTheme');
        this.preferences.ready.then(() => this.setSashProperties());
        this.preferences.onPreferenceChanged(e => this.handlePreferenceChange(e, app));
        this.themeService.onDidColorThemeChange(() => this.updateThemePreference('workbench.colorTheme'));
        this.iconThemes.onDidChangeCurrent(() => this.updateThemePreference('workbench.iconTheme'));
        app.shell.leftPanelHandler.addBottomMenu({
            id: 'settings-menu',
            iconClass: 'codicon codicon-settings-gear',
            title: nls_1.nls.localizeByDefault(CommonCommands.PREFERENCES_CATEGORY),
            menuPath: menu_1.SETTINGS_MENU,
            order: 0,
        });
        const accountsMenu = {
            id: 'accounts-menu',
            iconClass: 'codicon codicon-person',
            title: nls_1.nls.localizeByDefault('Accounts'),
            menuPath: menu_1.ACCOUNTS_MENU,
            order: 1,
        };
        this.authenticationService.onDidRegisterAuthenticationProvider(() => {
            app.shell.leftPanelHandler.addBottomMenu(accountsMenu);
        });
        this.authenticationService.onDidUnregisterAuthenticationProvider(() => {
            if (this.authenticationService.getProviderIds().length === 0) {
                app.shell.leftPanelHandler.removeBottomMenu(accountsMenu.id);
            }
        });
    }
    updateStyles() {
        document.body.classList.remove('theia-editor-highlightModifiedTabs');
        if (this.preferences['workbench.editor.highlightModifiedTabs']) {
            document.body.classList.add('theia-editor-highlightModifiedTabs');
        }
    }
    updatePinnedKey() {
        const activeTab = this.shell.findTabBar();
        const pinningTarget = activeTab && this.shell.findTitle(activeTab);
        const value = pinningTarget && (0, widgets_1.isPinned)(pinningTarget);
        this.pinnedKey.set(value);
    }
    updateThemePreference(preferenceName) {
        const inspect = this.preferenceService.inspect(preferenceName);
        const workspaceValue = inspect && inspect.workspaceValue;
        const userValue = inspect && inspect.globalValue;
        const value = workspaceValue || userValue;
        const newValue = preferenceName === 'workbench.colorTheme' ? this.themeService.getCurrentTheme().id : this.iconThemes.current;
        if (newValue !== value) {
            const scope = workspaceValue !== undefined ? preferences_1.PreferenceScope.Workspace : preferences_1.PreferenceScope.User;
            this.preferenceService.set(preferenceName, newValue, scope);
        }
    }
    updateThemeFromPreference(preferenceName) {
        const inspect = this.preferenceService.inspect(preferenceName);
        const workspaceValue = inspect && inspect.workspaceValue;
        const userValue = inspect && inspect.globalValue;
        const value = workspaceValue || userValue;
        if (value !== undefined) {
            if (preferenceName === 'workbench.colorTheme') {
                this.themeService.setCurrentTheme(value || this.themeService.defaultTheme.id);
            }
            else {
                this.iconThemes.current = value || this.iconThemes.default.id;
            }
        }
    }
    handlePreferenceChange(e, app) {
        switch (e.preferenceName) {
            case 'workbench.editor.highlightModifiedTabs': {
                this.updateStyles();
                break;
            }
            case 'workbench.colorTheme':
            case 'workbench.iconTheme': {
                this.updateThemeFromPreference(e.preferenceName);
                break;
            }
            case 'window.menuBarVisibility': {
                const { newValue } = e;
                const mainMenuId = 'main-menu';
                if (newValue === 'compact') {
                    this.shell.leftPanelHandler.addTopMenu({
                        id: mainMenuId,
                        iconClass: 'codicon codicon-menu',
                        title: nls_1.nls.localizeByDefault('Application Menu'),
                        menuPath: menu_1.MAIN_MENU_BAR,
                        order: 0,
                    });
                }
                else {
                    app.shell.leftPanelHandler.removeTopMenu(mainMenuId);
                }
                break;
            }
            case 'workbench.sash.hoverDelay':
            case 'workbench.sash.size': {
                this.setSashProperties();
                break;
            }
        }
    }
    setSashProperties() {
        const sashRule = `:root {
            --theia-sash-hoverDelay: ${this.preferences['workbench.sash.hoverDelay']}ms;
            --theia-sash-width: ${this.preferences['workbench.sash.size']}px;
        }`;
        decoration_style_1.DecorationStyle.deleteStyleRule(':root', this.commonDecorationsStyleSheet);
        this.commonDecorationsStyleSheet.insertRule(sashRule);
    }
    onStart() {
        this.storageService.getData(exports.RECENT_COMMANDS_STORAGE_KEY, { recent: [] })
            .then(tasks => this.commandRegistry.recent = tasks.recent);
    }
    onStop() {
        const recent = this.commandRegistry.recent;
        this.storageService.setData(exports.RECENT_COMMANDS_STORAGE_KEY, { recent });
    }
    initResourceContextKeys() {
        const updateContextKeys = () => {
            const selection = this.selectionService.selection;
            const resourceUri = navigatable_1.Navigatable.is(selection) && selection.getResourceUri() || selection_1.UriSelection.getUri(selection);
            this.resourceContextKey.set(resourceUri);
        };
        updateContextKeys();
        this.selectionService.onSelectionChanged(updateContextKeys);
    }
    registerMenus(registry) {
        registry.registerSubmenu(CommonMenus.FILE, nls_1.nls.localizeByDefault('File'));
        registry.registerSubmenu(CommonMenus.EDIT, nls_1.nls.localizeByDefault('Edit'));
        registry.registerSubmenu(CommonMenus.VIEW, nls_1.nls.localizeByDefault('View'));
        registry.registerSubmenu(CommonMenus.HELP, nls_1.nls.localizeByDefault('Help'));
        registry.registerMenuAction(CommonMenus.FILE_SAVE, {
            commandId: CommonCommands.SAVE.id
        });
        registry.registerMenuAction(CommonMenus.FILE_SAVE, {
            commandId: CommonCommands.SAVE_ALL.id
        });
        registry.registerMenuAction(CommonMenus.FILE_AUTOSAVE, {
            commandId: CommonCommands.AUTO_SAVE.id
        });
        registry.registerSubmenu(CommonMenus.FILE_SETTINGS_SUBMENU, nls_1.nls.localizeByDefault(CommonCommands.PREFERENCES_CATEGORY));
        registry.registerMenuAction(CommonMenus.EDIT_UNDO, {
            commandId: CommonCommands.UNDO.id,
            order: '0'
        });
        registry.registerMenuAction(CommonMenus.EDIT_UNDO, {
            commandId: CommonCommands.REDO.id,
            order: '1'
        });
        registry.registerMenuAction(CommonMenus.EDIT_FIND, {
            commandId: CommonCommands.FIND.id,
            order: '0'
        });
        registry.registerMenuAction(CommonMenus.EDIT_FIND, {
            commandId: CommonCommands.REPLACE.id,
            order: '1'
        });
        registry.registerMenuAction(CommonMenus.EDIT_CLIPBOARD, {
            commandId: CommonCommands.CUT.id,
            order: '0'
        });
        registry.registerMenuAction(CommonMenus.EDIT_CLIPBOARD, {
            commandId: CommonCommands.COPY.id,
            order: '1'
        });
        registry.registerMenuAction(CommonMenus.EDIT_CLIPBOARD, {
            commandId: CommonCommands.PASTE.id,
            order: '2'
        });
        registry.registerMenuAction(CommonMenus.EDIT_CLIPBOARD, {
            commandId: CommonCommands.COPY_PATH.id,
            order: '3'
        });
        registry.registerMenuAction(CommonMenus.VIEW_APPEARANCE_SUBMENU_BAR, {
            commandId: CommonCommands.TOGGLE_BOTTOM_PANEL.id,
            order: '1'
        });
        registry.registerMenuAction(CommonMenus.VIEW_APPEARANCE_SUBMENU_BAR, {
            commandId: CommonCommands.TOGGLE_STATUS_BAR.id,
            order: '2',
            label: nls_1.nls.localizeByDefault('Toggle Status Bar Visibility')
        });
        registry.registerMenuAction(CommonMenus.VIEW_APPEARANCE_SUBMENU_BAR, {
            commandId: CommonCommands.COLLAPSE_ALL_PANELS.id,
            order: '3'
        });
        registry.registerMenuAction(tab_bars_1.SHELL_TABBAR_CONTEXT_CLOSE, {
            commandId: CommonCommands.CLOSE_TAB.id,
            label: nls_1.nls.localizeByDefault('Close'),
            order: '0'
        });
        registry.registerMenuAction(tab_bars_1.SHELL_TABBAR_CONTEXT_CLOSE, {
            commandId: CommonCommands.CLOSE_OTHER_TABS.id,
            label: nls_1.nls.localizeByDefault('Close Others'),
            order: '1'
        });
        registry.registerMenuAction(tab_bars_1.SHELL_TABBAR_CONTEXT_CLOSE, {
            commandId: CommonCommands.CLOSE_RIGHT_TABS.id,
            label: nls_1.nls.localizeByDefault('Close to the Right'),
            order: '2'
        });
        registry.registerMenuAction(tab_bars_1.SHELL_TABBAR_CONTEXT_CLOSE, {
            commandId: CommonCommands.CLOSE_SAVED_TABS.id,
            label: nls_1.nls.localizeByDefault('Close Saved'),
            order: '3',
        });
        registry.registerMenuAction(tab_bars_1.SHELL_TABBAR_CONTEXT_CLOSE, {
            commandId: CommonCommands.CLOSE_ALL_TABS.id,
            label: nls_1.nls.localizeByDefault('Close All'),
            order: '4'
        });
        registry.registerMenuAction(tab_bars_1.SHELL_TABBAR_CONTEXT_SPLIT, {
            commandId: CommonCommands.COLLAPSE_PANEL.id,
            label: CommonCommands.COLLAPSE_PANEL.label,
            order: '5'
        });
        registry.registerMenuAction(tab_bars_1.SHELL_TABBAR_CONTEXT_SPLIT, {
            commandId: CommonCommands.TOGGLE_MAXIMIZED.id,
            label: CommonCommands.TOGGLE_MAXIMIZED.label,
            order: '6'
        });
        registry.registerMenuAction(CommonMenus.VIEW_APPEARANCE_SUBMENU_SCREEN, {
            commandId: CommonCommands.TOGGLE_MAXIMIZED.id,
            label: CommonCommands.TOGGLE_MAXIMIZED.label,
            order: '6'
        });
        registry.registerMenuAction(tab_bars_1.SHELL_TABBAR_CONTEXT_COPY, {
            commandId: CommonCommands.COPY_PATH.id,
            label: CommonCommands.COPY_PATH.label,
            order: '1',
        });
        registry.registerMenuAction(CommonMenus.VIEW_APPEARANCE_SUBMENU_BAR, {
            commandId: CommonCommands.SHOW_MENU_BAR.id,
            label: nls_1.nls.localizeByDefault('Toggle Menu Bar'),
            order: '0'
        });
        registry.registerMenuAction(tab_bars_1.SHELL_TABBAR_CONTEXT_PIN, {
            commandId: CommonCommands.PIN_TAB.id,
            label: nls_1.nls.localizeByDefault('Pin'),
            order: '7'
        });
        registry.registerMenuAction(tab_bars_1.SHELL_TABBAR_CONTEXT_PIN, {
            commandId: CommonCommands.UNPIN_TAB.id,
            label: nls_1.nls.localizeByDefault('Unpin'),
            order: '8'
        });
        registry.registerMenuAction(CommonMenus.HELP, {
            commandId: CommonCommands.ABOUT_COMMAND.id,
            label: CommonCommands.ABOUT_COMMAND.label,
            order: '9'
        });
        registry.registerMenuAction(CommonMenus.VIEW_PRIMARY, {
            commandId: CommonCommands.OPEN_VIEW.id
        });
        registry.registerMenuAction(CommonMenus.FILE_SETTINGS_SUBMENU_THEME, {
            commandId: CommonCommands.SELECT_COLOR_THEME.id
        });
        registry.registerMenuAction(CommonMenus.FILE_SETTINGS_SUBMENU_THEME, {
            commandId: CommonCommands.SELECT_ICON_THEME.id
        });
        registry.registerMenuAction(CommonMenus.SETTINGS__THEME, {
            commandId: CommonCommands.SELECT_COLOR_THEME.id
        });
        registry.registerMenuAction(CommonMenus.SETTINGS__THEME, {
            commandId: CommonCommands.SELECT_ICON_THEME.id
        });
        registry.registerSubmenu(CommonMenus.VIEW_APPEARANCE_SUBMENU, nls_1.nls.localizeByDefault('Appearance'));
    }
    registerCommands(commandRegistry) {
        commandRegistry.registerCommand(CommonCommands.OPEN, uri_command_handler_1.UriAwareCommandHandler.MultiSelect(this.selectionService, {
            execute: uris => uris.map(uri => (0, opener_service_1.open)(this.openerService, uri)),
        }));
        commandRegistry.registerCommand(CommonCommands.CUT, {
            execute: () => {
                if (exports.supportCut) {
                    document.execCommand('cut');
                }
                else {
                    this.messageService.warn("Please use the browser's cut command or shortcut.");
                }
            }
        });
        commandRegistry.registerCommand(CommonCommands.COPY, {
            execute: () => {
                if (exports.supportCopy) {
                    document.execCommand('copy');
                }
                else {
                    this.messageService.warn("Please use the browser's copy command or shortcut.");
                }
            }
        });
        commandRegistry.registerCommand(CommonCommands.PASTE, {
            execute: () => {
                if (exports.supportPaste) {
                    document.execCommand('paste');
                }
                else {
                    this.messageService.warn("Please use the browser's paste command or shortcut.");
                }
            }
        });
        commandRegistry.registerCommand(CommonCommands.COPY_PATH, uri_command_handler_1.UriAwareCommandHandler.MultiSelect(this.selectionService, {
            isVisible: uris => Array.isArray(uris) && uris.some(uri => uri instanceof uri_1.default),
            isEnabled: uris => Array.isArray(uris) && uris.some(uri => uri instanceof uri_1.default),
            execute: async (uris) => {
                if (uris.length) {
                    const lineDelimiter = os_1.isWindows ? '\r\n' : '\n';
                    const text = uris.map(resource => resource.path).join(lineDelimiter);
                    await this.clipboardService.writeText(text);
                }
                else {
                    await this.messageService.info('Open a file first to copy its path');
                }
            }
        }));
        commandRegistry.registerCommand(CommonCommands.UNDO, {
            execute: () => document.execCommand('undo')
        });
        commandRegistry.registerCommand(CommonCommands.REDO, {
            execute: () => document.execCommand('redo')
        });
        commandRegistry.registerCommand(CommonCommands.SELECT_ALL, {
            execute: () => document.execCommand('selectAll')
        });
        commandRegistry.registerCommand(CommonCommands.FIND, {
            execute: () => { }
        });
        commandRegistry.registerCommand(CommonCommands.REPLACE, {
            execute: () => { }
        });
        commandRegistry.registerCommand(CommonCommands.NEXT_TAB, {
            isEnabled: () => this.shell.currentTabBar !== undefined,
            execute: () => this.shell.activateNextTab()
        });
        commandRegistry.registerCommand(CommonCommands.PREVIOUS_TAB, {
            isEnabled: () => this.shell.currentTabBar !== undefined,
            execute: () => this.shell.activatePreviousTab()
        });
        commandRegistry.registerCommand(CommonCommands.NEXT_TAB_IN_GROUP, {
            isEnabled: () => this.shell.nextTabIndexInTabBar() !== -1,
            execute: () => this.shell.activateNextTabInTabBar()
        });
        commandRegistry.registerCommand(CommonCommands.PREVIOUS_TAB_IN_GROUP, {
            isEnabled: () => this.shell.previousTabIndexInTabBar() !== -1,
            execute: () => this.shell.activatePreviousTabInTabBar()
        });
        commandRegistry.registerCommand(CommonCommands.NEXT_TAB_GROUP, {
            isEnabled: () => this.shell.nextTabBar() !== undefined,
            execute: () => this.shell.activateNextTabBar()
        });
        commandRegistry.registerCommand(CommonCommands.PREVIOUS_TAB_GROUP, {
            isEnabled: () => this.shell.previousTabBar() !== undefined,
            execute: () => this.shell.activatePreviousTabBar()
        });
        commandRegistry.registerCommand(CommonCommands.CLOSE_TAB, new current_widget_command_adapter_1.CurrentWidgetCommandAdapter(this.shell, {
            isEnabled: title => Boolean(title === null || title === void 0 ? void 0 : title.closable),
            execute: (title, tabBar) => tabBar && this.shell.closeTabs(tabBar, candidate => candidate === title),
        }));
        commandRegistry.registerCommand(CommonCommands.CLOSE_OTHER_TABS, new current_widget_command_adapter_1.CurrentWidgetCommandAdapter(this.shell, {
            isEnabled: (title, tabbar) => Boolean(tabbar === null || tabbar === void 0 ? void 0 : tabbar.titles.some(candidate => candidate !== title && candidate.closable)),
            execute: (title, tabbar) => tabbar && this.shell.closeTabs(tabbar, candidate => candidate !== title && candidate.closable),
        }));
        commandRegistry.registerCommand(CommonCommands.CLOSE_SAVED_TABS, new current_widget_command_adapter_1.CurrentWidgetCommandAdapter(this.shell, {
            isEnabled: (_title, tabbar) => Boolean(tabbar === null || tabbar === void 0 ? void 0 : tabbar.titles.some(candidate => candidate.closable && !saveable_1.Saveable.isDirty(candidate.owner))),
            execute: (_title, tabbar) => tabbar && this.shell.closeTabs(tabbar, candidate => candidate.closable && !saveable_1.Saveable.isDirty(candidate.owner)),
        }));
        commandRegistry.registerCommand(CommonCommands.CLOSE_RIGHT_TABS, new current_widget_command_adapter_1.CurrentWidgetCommandAdapter(this.shell, {
            isEnabled: (title, tabbar) => {
                let targetSeen = false;
                return Boolean(tabbar === null || tabbar === void 0 ? void 0 : tabbar.titles.some(candidate => {
                    if (targetSeen && candidate.closable) {
                        return true;
                    }
                    ;
                    if (candidate === title) {
                        targetSeen = true;
                    }
                    ;
                }));
            },
            isVisible: (_title, tabbar) => {
                var _a;
                const area = (_a = (tabbar && this.shell.getAreaFor(tabbar))) !== null && _a !== void 0 ? _a : this.shell.currentTabArea;
                return area !== undefined && area !== 'left' && area !== 'right';
            },
            execute: (title, tabbar) => {
                if (tabbar) {
                    let targetSeen = false;
                    this.shell.closeTabs(tabbar, candidate => {
                        if (targetSeen && candidate.closable) {
                            return true;
                        }
                        ;
                        if (candidate === title) {
                            targetSeen = true;
                        }
                        ;
                        return false;
                    });
                }
            }
        }));
        commandRegistry.registerCommand(CommonCommands.CLOSE_ALL_TABS, new current_widget_command_adapter_1.CurrentWidgetCommandAdapter(this.shell, {
            isEnabled: (_title, tabbar) => Boolean(tabbar === null || tabbar === void 0 ? void 0 : tabbar.titles.some(title => title.closable)),
            execute: (_title, tabbar) => tabbar && this.shell.closeTabs(tabbar, candidate => candidate.closable),
        }));
        commandRegistry.registerCommand(CommonCommands.CLOSE_MAIN_TAB, {
            isEnabled: () => {
                const currentWidget = this.shell.getCurrentWidget('main');
                return currentWidget !== undefined && currentWidget.title.closable;
            },
            execute: () => this.shell.getCurrentWidget('main').close()
        });
        commandRegistry.registerCommand(CommonCommands.CLOSE_OTHER_MAIN_TABS, {
            isEnabled: () => {
                const currentWidget = this.shell.getCurrentWidget('main');
                return currentWidget !== undefined &&
                    this.shell.mainAreaTabBars.some(tb => tb.titles.some(title => title.owner !== currentWidget && title.closable));
            },
            execute: () => {
                const currentWidget = this.shell.getCurrentWidget('main');
                this.shell.closeTabs('main', title => title.owner !== currentWidget && title.closable);
            }
        });
        commandRegistry.registerCommand(CommonCommands.CLOSE_ALL_MAIN_TABS, {
            isEnabled: () => this.shell.mainAreaTabBars.some(tb => tb.titles.some(title => title.closable)),
            execute: () => this.shell.closeTabs('main', title => title.closable)
        });
        commandRegistry.registerCommand(CommonCommands.COLLAPSE_PANEL, new current_widget_command_adapter_1.CurrentWidgetCommandAdapter(this.shell, {
            isEnabled: (_title, tabbar) => Boolean(tabbar && application_shell_1.ApplicationShell.isSideArea(this.shell.getAreaFor(tabbar))),
            isVisible: (_title, tabbar) => Boolean(tabbar && application_shell_1.ApplicationShell.isSideArea(this.shell.getAreaFor(tabbar))),
            execute: (_title, tabbar) => tabbar && this.shell.collapsePanel(this.shell.getAreaFor(tabbar))
        }));
        commandRegistry.registerCommand(CommonCommands.COLLAPSE_ALL_PANELS, {
            execute: () => {
                this.shell.collapsePanel('left');
                this.shell.collapsePanel('right');
                this.shell.collapsePanel('bottom');
            }
        });
        commandRegistry.registerCommand(CommonCommands.TOGGLE_BOTTOM_PANEL, {
            isEnabled: () => this.shell.getWidgets('bottom').length > 0,
            execute: () => {
                if (this.shell.isExpanded('bottom')) {
                    this.shell.collapsePanel('bottom');
                }
                else {
                    this.shell.expandPanel('bottom');
                }
            }
        });
        commandRegistry.registerCommand(CommonCommands.TOGGLE_STATUS_BAR, {
            execute: () => this.preferenceService.updateValue('workbench.statusBar.visible', !this.preferences['workbench.statusBar.visible'])
        });
        commandRegistry.registerCommand(CommonCommands.TOGGLE_MAXIMIZED, new current_widget_command_adapter_1.CurrentWidgetCommandAdapter(this.shell, {
            isEnabled: title => Boolean((title === null || title === void 0 ? void 0 : title.owner) && this.shell.canToggleMaximized(title === null || title === void 0 ? void 0 : title.owner)),
            isVisible: title => Boolean((title === null || title === void 0 ? void 0 : title.owner) && this.shell.canToggleMaximized(title === null || title === void 0 ? void 0 : title.owner)),
            execute: title => (title === null || title === void 0 ? void 0 : title.owner) && this.shell.toggleMaximized(title === null || title === void 0 ? void 0 : title.owner),
        }));
        commandRegistry.registerCommand(CommonCommands.SHOW_MENU_BAR, {
            isEnabled: () => !os_1.isOSX,
            isVisible: () => !os_1.isOSX,
            execute: () => {
                const menuBarVisibility = 'window.menuBarVisibility';
                const visibility = this.preferences[menuBarVisibility];
                if (visibility !== 'compact') {
                    this.preferenceService.updateValue(menuBarVisibility, 'compact');
                }
                else {
                    this.preferenceService.updateValue(menuBarVisibility, 'classic');
                }
            }
        });
        commandRegistry.registerCommand(CommonCommands.SAVE, {
            execute: () => this.save({ formatType: 1 /* ON */ })
        });
        commandRegistry.registerCommand(CommonCommands.SAVE_WITHOUT_FORMATTING, {
            execute: () => this.save({ formatType: 2 /* OFF */ })
        });
        commandRegistry.registerCommand(CommonCommands.SAVE_ALL, {
            execute: () => this.shell.saveAll({ formatType: 3 /* DIRTY */ })
        });
        commandRegistry.registerCommand(CommonCommands.ABOUT_COMMAND, {
            execute: () => this.openAbout()
        });
        commandRegistry.registerCommand(CommonCommands.OPEN_VIEW, {
            execute: () => { var _a; return (_a = this.quickInputService) === null || _a === void 0 ? void 0 : _a.open(quick_view_service_1.QuickViewService.PREFIX); }
        });
        commandRegistry.registerCommand(CommonCommands.SELECT_COLOR_THEME, {
            execute: () => this.selectColorTheme()
        });
        commandRegistry.registerCommand(CommonCommands.SELECT_ICON_THEME, {
            execute: () => this.selectIconTheme()
        });
        commandRegistry.registerCommand(CommonCommands.PIN_TAB, new current_widget_command_adapter_1.CurrentWidgetCommandAdapter(this.shell, {
            isEnabled: title => Boolean(title && !(0, widgets_1.isPinned)(title)),
            execute: title => this.togglePinned(title),
        }));
        commandRegistry.registerCommand(CommonCommands.UNPIN_TAB, new current_widget_command_adapter_1.CurrentWidgetCommandAdapter(this.shell, {
            isEnabled: title => Boolean(title && (0, widgets_1.isPinned)(title)),
            execute: title => this.togglePinned(title),
        }));
        commandRegistry.registerCommand(CommonCommands.CONFIGURE_DISPLAY_LANGUAGE, {
            execute: () => this.configureDisplayLanguage()
        });
    }
    isElectron() {
        return environment_1.environment.electron.is();
    }
    togglePinned(title) {
        if (title) {
            (0, widgets_1.togglePinned)(title);
            this.updatePinnedKey();
        }
    }
    registerKeybindings(registry) {
        if (exports.supportCut) {
            registry.registerKeybinding({
                command: CommonCommands.CUT.id,
                keybinding: 'ctrlcmd+x'
            });
        }
        if (exports.supportCopy) {
            registry.registerKeybinding({
                command: CommonCommands.COPY.id,
                keybinding: 'ctrlcmd+c'
            });
        }
        if (exports.supportPaste) {
            registry.registerKeybinding({
                command: CommonCommands.PASTE.id,
                keybinding: 'ctrlcmd+v'
            });
        }
        registry.registerKeybinding({
            command: CommonCommands.COPY_PATH.id,
            keybinding: os_1.isWindows ? 'shift+alt+c' : 'ctrlcmd+alt+c',
            when: '!editorFocus'
        });
        registry.registerKeybindings(
        // Edition
        {
            command: CommonCommands.UNDO.id,
            keybinding: 'ctrlcmd+z'
        }, {
            command: CommonCommands.REDO.id,
            keybinding: 'ctrlcmd+shift+z'
        }, {
            command: CommonCommands.SELECT_ALL.id,
            keybinding: 'ctrlcmd+a'
        }, {
            command: CommonCommands.FIND.id,
            keybinding: 'ctrlcmd+f'
        }, {
            command: CommonCommands.REPLACE.id,
            keybinding: 'ctrlcmd+alt+f'
        }, 
        // Tabs
        {
            command: CommonCommands.NEXT_TAB.id,
            keybinding: 'ctrlcmd+tab'
        }, {
            command: CommonCommands.NEXT_TAB.id,
            keybinding: 'ctrlcmd+alt+d'
        }, {
            command: CommonCommands.PREVIOUS_TAB.id,
            keybinding: 'ctrlcmd+shift+tab'
        }, {
            command: CommonCommands.PREVIOUS_TAB.id,
            keybinding: 'ctrlcmd+alt+a'
        }, {
            command: CommonCommands.CLOSE_MAIN_TAB.id,
            keybinding: this.isElectron() ? (os_1.isWindows ? 'ctrl+f4' : 'ctrlcmd+w') : 'alt+w'
        }, {
            command: CommonCommands.CLOSE_OTHER_MAIN_TABS.id,
            keybinding: 'ctrlcmd+alt+t'
        }, {
            command: CommonCommands.CLOSE_ALL_MAIN_TABS.id,
            keybinding: this.isElectron() ? 'ctrlCmd+k ctrlCmd+w' : 'alt+shift+w'
        }, 
        // Panels
        {
            command: CommonCommands.COLLAPSE_PANEL.id,
            keybinding: 'alt+c'
        }, {
            command: CommonCommands.TOGGLE_BOTTOM_PANEL.id,
            keybinding: 'ctrlcmd+j',
        }, {
            command: CommonCommands.COLLAPSE_ALL_PANELS.id,
            keybinding: 'alt+shift+c',
        }, {
            command: CommonCommands.TOGGLE_MAXIMIZED.id,
            keybinding: 'alt+m',
        }, 
        // Saving
        {
            command: CommonCommands.SAVE.id,
            keybinding: 'ctrlcmd+s'
        }, {
            command: CommonCommands.SAVE_WITHOUT_FORMATTING.id,
            keybinding: 'ctrlcmd+k s'
        }, {
            command: CommonCommands.SAVE_ALL.id,
            keybinding: 'ctrlcmd+alt+s'
        }, 
        // Theming
        {
            command: CommonCommands.SELECT_COLOR_THEME.id,
            keybinding: 'ctrlcmd+k ctrlcmd+t'
        }, {
            command: CommonCommands.PIN_TAB.id,
            keybinding: 'ctrlcmd+k shift+enter',
            when: '!activeEditorIsPinned'
        }, {
            command: CommonCommands.UNPIN_TAB.id,
            keybinding: 'ctrlcmd+k shift+enter',
            when: 'activeEditorIsPinned'
        });
    }
    async save(options) {
        const widget = this.shell.currentWidget;
        this.saveResourceService.save(widget, options);
    }
    async openAbout() {
        this.aboutDialog.open();
    }
    /**
     * registers event listener which make sure that
     * window doesn't get closed if CMD/CTRL W is pressed.
     * Too many users have that in their muscle memory.
     * Chrome doesn't let us rebind or prevent default the keybinding, so this
     * at least doesn't close the window immediately.
     */
    registerCtrlWHandling() {
        function isCtrlCmd(event) {
            return (os_1.isOSX && event.metaKey) || (!os_1.isOSX && event.ctrlKey);
        }
        window.document.addEventListener('keydown', event => {
            this.shouldPreventClose = isCtrlCmd(event) && event.code === 'KeyW';
        });
        window.document.addEventListener('keyup', () => {
            this.shouldPreventClose = false;
        });
    }
    onWillStop() {
        try {
            if (this.shouldPreventClose || this.shell.canSaveAll()) {
                return { reason: 'Dirty editors present', action: () => (0, dialogs_1.confirmExit)() };
            }
        }
        finally {
            this.shouldPreventClose = false;
        }
    }
    async configureDisplayLanguage() {
        var _a;
        const availableLanguages = await this.localizationProvider.getAvailableLanguages();
        const items = [];
        for (const additionalLanguage of ['en', ...availableLanguages]) {
            items.push({
                label: additionalLanguage,
                execute: async () => {
                    if (additionalLanguage !== nls_1.nls.locale && await this.confirmRestart()) {
                        this.windowService.setSafeToShutDown();
                        window.localStorage.setItem(nls_1.nls.localeId, additionalLanguage);
                        this.windowService.reload();
                    }
                }
            });
        }
        (_a = this.quickInputService) === null || _a === void 0 ? void 0 : _a.showQuickPick(items, {
            placeholder: CommonCommands.CONFIGURE_DISPLAY_LANGUAGE.label,
            activeItem: items.find(item => item.label === (nls_1.nls.locale || 'en'))
        });
    }
    async confirmRestart() {
        const shouldRestart = await new dialogs_1.ConfirmDialog({
            title: nls_1.nls.localizeByDefault('A restart is required for the change in display language to take effect.'),
            msg: nls_1.nls.localizeByDefault('Press the restart button to restart {0} and change the display language.', frontend_application_config_provider_1.FrontendApplicationConfigProvider.get().applicationName),
            ok: nls_1.nls.localizeByDefault('Restart'),
            cancel: dialogs_1.Dialog.CANCEL,
        }).open();
        return shouldRestart === true;
    }
    selectIconTheme() {
        var _a;
        let resetTo = this.iconThemes.current;
        const previewTheme = debounce((id) => this.iconThemes.current = id, 200);
        let items = [];
        for (const iconTheme of this.iconThemes.definitions) {
            items.push({
                id: iconTheme.id,
                label: iconTheme.label,
                description: iconTheme.description,
            });
        }
        items = items.sort((a, b) => {
            if (a.id === 'none') {
                return -1;
            }
            return a.label.localeCompare(b.label);
        });
        (_a = this.quickInputService) === null || _a === void 0 ? void 0 : _a.showQuickPick(items, {
            placeholder: nls_1.nls.localizeByDefault('Select File Icon Theme'),
            activeItem: items.find(item => item.id === resetTo),
            onDidChangeSelection: (quickPick, selectedItems) => {
                resetTo = undefined;
                previewTheme(selectedItems[0].id);
            },
            onDidChangeActive: (quickPick, activeItems) => {
                previewTheme(activeItems[0].id);
            },
            onDidHide: () => {
                if (resetTo) {
                    this.iconThemes.current = resetTo;
                }
            }
        });
    }
    selectColorTheme() {
        var _a;
        let resetTo = this.themeService.getCurrentTheme().id;
        const previewTheme = debounce((id) => this.themeService.setCurrentTheme(id), 200);
        const itemsByTheme = { light: [], dark: [], hc: [] };
        for (const theme of this.themeService.getThemes().sort((a, b) => a.label.localeCompare(b.label))) {
            const themeItems = itemsByTheme[theme.type];
            if (themeItems.length === 0) {
                themeItems.push({
                    type: 'separator',
                    label: (theme.type === 'hc' ? 'high contrast' : theme.type) + ' themes'
                });
            }
            themeItems.push({
                id: theme.id,
                label: theme.label,
                description: theme.description,
            });
        }
        const items = [...itemsByTheme.light, ...itemsByTheme.dark, ...itemsByTheme.hc];
        (_a = this.quickInputService) === null || _a === void 0 ? void 0 : _a.showQuickPick(items, {
            placeholder: nls_1.nls.localizeByDefault('Select Color Theme (Up/Down Keys to Preview)'),
            activeItem: items.find((item) => item.id === resetTo),
            // eslint-disable-next-line @typescript-eslint/no-explicit-any
            onDidChangeSelection: (quickPick, selectedItems) => {
                resetTo = undefined;
                previewTheme(selectedItems[0].id);
            },
            // eslint-disable-next-line @typescript-eslint/no-explicit-any
            onDidChangeActive: (quickPick, activeItems) => {
                previewTheme(activeItems[0].id);
            },
            onDidHide: () => {
                if (resetTo) {
                    this.themeService.setCurrentTheme(resetTo);
                }
            }
        });
    }
    registerColors(colors) {
        colors.register(
        // Base Colors should be aligned with https://code.visualstudio.com/api/references/theme-color#base-colors
        // if not yet contributed by Monaco, check runtime css variables to learn
        { id: 'selection.background', defaults: { dark: '#217daf', light: '#c0dbf1' }, description: 'Overall border color for focused elements. This color is only used if not overridden by a component.' }, { id: 'icon.foreground', defaults: { dark: '#C5C5C5', light: '#424242', hc: '#FFFFFF' }, description: 'The default color for icons in the workbench.' }, { id: 'sash.hoverBorder', defaults: { dark: color_registry_1.Color.transparent('focusBorder', 0.99), light: color_registry_1.Color.transparent('focusBorder', 0.99), hc: color_registry_1.Color.transparent('focusBorder', 0.99) }, description: 'The hover border color for draggable sashes.' }, { id: 'sash.activeBorder', defaults: { dark: 'focusBorder', light: 'focusBorder', hc: 'focusBorder' }, description: 'The active border color for draggable sashes.' }, 
        // Window border colors should be aligned with https://code.visualstudio.com/api/references/theme-color#window-border
        {
            id: 'window.activeBorder', defaults: {
                hc: 'contrastBorder'
            }, description: 'The color used for the border of the window when it is active.'
        }, {
            id: 'window.inactiveBorder', defaults: {
                hc: 'contrastBorder'
            },
            description: 'The color used for the border of the window when it is inactive.'
        }, 
        // Buttons should be aligned with https://code.visualstudio.com/api/references/theme-color#button-control
        // if not yet contributed by Monaco, check runtime css variables to learn
        { id: 'button.foreground', defaults: { dark: color_registry_1.Color.white, light: color_registry_1.Color.white, hc: color_registry_1.Color.white }, description: 'Button foreground color.' }, { id: 'button.background', defaults: { dark: '#0E639C', light: '#007ACC' }, description: 'Button background color.' }, { id: 'button.hoverBackground', defaults: { dark: color_registry_1.Color.lighten('button.background', 0.2), light: color_registry_1.Color.darken('button.background', 0.2) }, description: 'Button background color when hovering.' }, 
        // Activity Bar colors should be aligned with https://code.visualstudio.com/api/references/theme-color#activity-bar
        {
            id: 'activityBar.background', defaults: {
                dark: '#333333',
                light: '#2C2C2C',
                hc: '#000000'
            }, description: 'Activity bar background color. The activity bar is showing on the far left or right and allows to switch between views of the side bar.'
        }, {
            id: 'activityBar.foreground', defaults: {
                dark: color_registry_1.Color.white,
                light: color_registry_1.Color.white,
                hc: color_registry_1.Color.white
            }, description: 'Activity bar item foreground color when it is active. The activity bar is showing on the far left or right and allows to switch between views of the side bar.',
        }, {
            id: 'activityBar.inactiveForeground', defaults: {
                dark: color_registry_1.Color.transparent('activityBar.foreground', 0.4),
                light: color_registry_1.Color.transparent('activityBar.foreground', 0.4),
                hc: color_registry_1.Color.white
            }, description: 'Activity bar item foreground color when it is inactive. The activity bar is showing on the far left or right and allows to switch between views of the side bar.'
        }, {
            id: 'activityBar.border', defaults: {
                hc: 'contrastBorder'
            }, description: 'Activity bar border color separating to the side bar. The activity bar is showing on the far left or right and allows to switch between views of the side bar.'
        }, {
            id: 'activityBar.activeBorder', defaults: {
                dark: 'activityBar.foreground',
                light: 'activityBar.foreground',
            }, description: 'Activity bar border color for the active item. The activity bar is showing on the far left or right and allows to switch between views of the side bar.'
        }, {
            id: 'activityBar.activeFocusBorder',
            description: 'Activity bar focus border color for the active item. The activity bar is showing on the far left or right and allows to switch between views of the side bar.'
        }, { id: 'activityBar.activeBackground', description: 'Activity bar background color for the active item. The activity bar is showing on the far left or right and allows to switch between views of the side bar.' }, {
            id: 'activityBar.dropBackground', defaults: {
                dark: color_registry_1.Color.transparent('#ffffff', 0.12),
                light: color_registry_1.Color.transparent('#ffffff', 0.12),
                hc: color_registry_1.Color.transparent('#ffffff', 0.12),
            }, description: 'Drag and drop feedback color for the activity bar items. The color should have transparency so that the activity bar entries can still shine through. The activity bar is showing on the far left or right and allows to switch between views of the side bar.'
        }, {
            id: 'activityBarBadge.background', defaults: {
                dark: '#007ACC',
                light: '#007ACC',
                hc: '#000000'
            }, description: 'Activity notification badge background color. The activity bar is showing on the far left or right and allows to switch between views of the side bar.'
        }, {
            id: 'activityBarBadge.foreground', defaults: {
                dark: color_registry_1.Color.white,
                light: color_registry_1.Color.white,
                hc: color_registry_1.Color.white
            }, description: 'Activity notification badge foreground color. The activity bar is showing on the far left or right and allows to switch between views of the side bar.'
        }, 
        // Side Bar should be aligned with https://code.visualstudio.com/api/references/theme-color#side-bar
        // if not yet contributed by Monaco, check runtime css variables to learn
        { id: 'sideBar.background', defaults: { dark: '#252526', light: '#F3F3F3', hc: '#000000' }, description: 'Side bar background color. The side bar is the container for views like explorer and search.' }, { id: 'sideBar.foreground', description: 'Side bar foreground color. The side bar is the container for views like explorer and search.' }, { id: 'sideBarSectionHeader.background', defaults: { dark: '#80808033', light: '#80808033' }, description: 'Side bar section header background color. The side bar is the container for views like explorer and search.' }, { id: 'sideBarSectionHeader.foreground', description: 'Side bar foreground color. The side bar is the container for views like explorer and search.' }, { id: 'sideBarSectionHeader.border', defaults: { hc: '#6FC3DF' }, description: 'Side bar section header border color. The side bar is the container for views like explorer and search.' }, 
        // Lists and Trees colors should be aligned with https://code.visualstudio.com/api/references/theme-color#lists-and-trees
        // if not yet contributed by Monaco, check runtime css variables to learn.
        // TODO: Following are not yet supported/no respective elements in theia:
        // list.focusBackground, list.focusForeground, list.inactiveFocusBackground, list.filterMatchBorder,
        // list.dropBackground, listFilterWidget.outline, listFilterWidget.noMatchesOutline
        // list.invalidItemForeground => tree node needs an respective class
        { id: 'list.activeSelectionBackground', defaults: { dark: '#094771', light: '#0074E8' }, description: 'List/Tree background color for the selected item when the list/tree is active. An active list/tree has keyboard focus, an inactive does not.' }, { id: 'list.activeSelectionForeground', defaults: { dark: '#FFF', light: '#FFF' }, description: 'List/Tree foreground color for the selected item when the list/tree is active. An active list/tree has keyboard focus, an inactive does not.' }, { id: 'list.inactiveSelectionBackground', defaults: { dark: '#37373D', light: '#E4E6F1' }, description: 'List/Tree background color for the selected item when the list/tree is inactive. An active list/tree has keyboard focus, an inactive does not.' }, { id: 'list.inactiveSelectionForeground', description: 'List/Tree foreground color for the selected item when the list/tree is inactive. An active list/tree has keyboard focus, an inactive does not.' }, { id: 'list.hoverBackground', defaults: { dark: '#2A2D2E', light: '#F0F0F0' }, description: 'List/Tree background when hovering over items using the mouse.' }, { id: 'list.hoverForeground', description: 'List/Tree foreground when hovering over items using the mouse.' }, { id: 'list.errorForeground', defaults: { dark: '#F88070', light: '#B01011' }, description: 'Foreground color of list items containing errors.' }, { id: 'list.warningForeground', defaults: { dark: '#CCA700', light: '#855F00' }, description: 'Foreground color of list items containing warnings.' }, { id: 'list.filterMatchBackground', defaults: { dark: 'editor.findMatchHighlightBackground', light: 'editor.findMatchHighlightBackground' }, description: 'Background color of the filtered match.' }, { id: 'list.highlightForeground', defaults: { dark: '#18A3FF', light: '#0066BF', hc: 'focusBorder' }, description: 'List/Tree foreground color of the match highlights when searching inside the list/tree.' }, { id: 'list.focusHighlightForeground', defaults: { dark: 'list.highlightForeground', light: 'list.activeSelectionForeground', hc: 'list.highlightForeground' }, description: 'List/Tree foreground color of the match highlights on actively focused items when searching inside the list/tree.' }, { id: 'tree.inactiveIndentGuidesStroke', defaults: { dark: color_registry_1.Color.transparent('tree.indentGuidesStroke', 0.4), light: color_registry_1.Color.transparent('tree.indentGuidesStroke', 0.4), hc: color_registry_1.Color.transparent('tree.indentGuidesStroke', 0.4) }, description: 'Tree stroke color for the inactive indentation guides.' }, 
        // Editor Group & Tabs colors should be aligned with https://code.visualstudio.com/api/references/theme-color#editor-groups-tabs
        {
            id: 'editorGroup.border',
            defaults: {
                dark: '#444444',
                light: '#E7E7E7',
                hc: 'contrastBorder'
            },
            description: 'Color to separate multiple editor groups from each other. Editor groups are the containers of editors.'
        }, {
            id: 'editorGroup.dropBackground',
            defaults: {
                dark: color_registry_1.Color.transparent('#53595D', 0.5),
                light: color_registry_1.Color.transparent('#2677CB', 0.18)
            },
            description: 'Background color when dragging editors around. The color should have transparency so that the editor contents can still shine through.'
        }, {
            id: 'editorGroupHeader.tabsBackground',
            defaults: {
                dark: '#252526',
                light: '#F3F3F3'
            },
            description: 'Background color of the editor group title header when tabs are enabled. Editor groups are the containers of editors.'
        }, {
            id: 'editorGroupHeader.tabsBorder',
            defaults: {
                hc: 'contrastBorder'
            },
            description: 'Border color of the editor group title header when tabs are enabled. Editor groups are the containers of editors.'
        }, {
            id: 'tab.activeBackground',
            defaults: {
                dark: 'editor.background',
                light: 'editor.background',
                hc: 'editor.background'
            },
            description: 'Active tab background color. Tabs are the containers for editors in the editor area. Multiple tabs can be opened in one editor group. There can be multiple editor groups.'
        }, {
            id: 'tab.unfocusedActiveBackground',
            defaults: {
                dark: 'tab.activeBackground',
                light: 'tab.activeBackground',
                hc: 'tab.activeBackground'
            },
            description: 'Active tab background color in an unfocused group. Tabs are the containers for editors in the editor area. Multiple tabs can be opened in one editor group. There can be multiple editor groups.'
        }, {
            id: 'tab.inactiveBackground',
            defaults: {
                dark: '#2D2D2D',
                light: '#ECECEC'
            },
            description: 'Inactive tab background color. Tabs are the containers for editors in the editor area. Multiple tabs can be opened in one editor group. There can be multiple editor groups.'
        }, {
            id: 'tab.activeForeground',
            defaults: {
                dark: color_registry_1.Color.white,
                light: '#333333',
                hc: color_registry_1.Color.white
            }, description: 'Active tab foreground color in an active group. Tabs are the containers for editors in the editor area. Multiple tabs can be opened in one editor group. There can be multiple editor groups.'
        }, {
            id: 'tab.inactiveForeground', defaults: {
                dark: color_registry_1.Color.transparent('tab.activeForeground', 0.5),
                light: color_registry_1.Color.transparent('tab.activeForeground', 0.7),
                hc: color_registry_1.Color.white
            }, description: 'Inactive tab foreground color in an active group. Tabs are the containers for editors in the editor area. Multiple tabs can be opened in one editor group. There can be multiple editor groups.'
        }, {
            id: 'tab.unfocusedActiveForeground', defaults: {
                dark: color_registry_1.Color.transparent('tab.activeForeground', 0.5),
                light: color_registry_1.Color.transparent('tab.activeForeground', 0.7),
                hc: color_registry_1.Color.white
            }, description: 'Active tab foreground color in an unfocused group. Tabs are the containers for editors in the editor area. Multiple tabs can be opened in one editor group. There can be multiple editor groups.'
        }, {
            id: 'tab.unfocusedInactiveForeground', defaults: {
                dark: color_registry_1.Color.transparent('tab.inactiveForeground', 0.5),
                light: color_registry_1.Color.transparent('tab.inactiveForeground', 0.5),
                hc: color_registry_1.Color.white
            }, description: 'Inactive tab foreground color in an unfocused group. Tabs are the containers for editors in the editor area. Multiple tabs can be opened in one editor group. There can be multiple editor groups.'
        }, {
            id: 'tab.border', defaults: {
                dark: '#252526',
                light: '#F3F3F3',
                hc: 'contrastBorder'
            }, description: 'Border to separate tabs from each other. Tabs are the containers for editors in the editor area. Multiple tabs can be opened in one editor group. There can be multiple editor groups.'
        }, {
            id: 'tab.activeBorder',
            description: 'Border on the bottom of an active tab. Tabs are the containers for editors in the editor area. Multiple tabs can be opened in one editor group. There can be multiple editor groups.'
        }, {
            id: 'tab.unfocusedActiveBorder',
            defaults: {
                dark: color_registry_1.Color.transparent('tab.activeBorder', 0.5),
                light: color_registry_1.Color.transparent('tab.activeBorder', 0.7)
            },
            description: 'Border on the bottom of an active tab in an unfocused group. Tabs are the containers for editors in the editor area. Multiple tabs can be opened in one editor group. There can be multiple editor groups.'
        }, {
            id: 'tab.activeBorderTop',
            defaults: {
                dark: 'focusBorder',
                light: 'focusBorder'
            },
            description: 'Border to the top of an active tab. Tabs are the containers for editors in the editor area. Multiple tabs can be opened in one editor group. There can be multiple editor groups.'
        }, {
            id: 'tab.unfocusedActiveBorderTop', defaults: {
                dark: color_registry_1.Color.transparent('tab.activeBorderTop', 0.5),
                light: color_registry_1.Color.transparent('tab.activeBorderTop', 0.7)
            }, description: 'Border to the top of an active tab in an unfocused group. Tabs are the containers for editors in the editor area. Multiple tabs can be opened in one editor group. There can be multiple editor groups.'
        }, {
            id: 'tab.hoverBackground',
            description: 'Tab background color when hovering. Tabs are the containers for editors in the editor area. Multiple tabs can be opened in one editor group. There can be multiple editor groups.'
        }, {
            id: 'tab.unfocusedHoverBackground', defaults: {
                dark: color_registry_1.Color.transparent('tab.hoverBackground', 0.5),
                light: color_registry_1.Color.transparent('tab.hoverBackground', 0.7)
            }, description: 'Tab background color in an unfocused group when hovering. Tabs are the containers for editors in the editor area. Multiple tabs can be opened in one editor group. There can be multiple editor groups.'
        }, {
            id: 'tab.hoverBorder',
            description: 'Border to highlight tabs when hovering. Tabs are the containers for editors in the editor area. Multiple tabs can be opened in one editor group. There can be multiple editor groups.'
        }, {
            id: 'tab.unfocusedHoverBorder', defaults: {
                dark: color_registry_1.Color.transparent('tab.hoverBorder', 0.5),
                light: color_registry_1.Color.transparent('tab.hoverBorder', 0.7)
            }, description: 'Border to highlight tabs in an unfocused group when hovering. Tabs are the containers for editors in the editor area. Multiple tabs can be opened in one editor group. There can be multiple editor groups.'
        }, {
            id: 'tab.activeModifiedBorder', defaults: {
                dark: '#3399CC',
                light: '#33AAEE'
            }, description: 'Border on the top of modified (dirty) active tabs in an active group. Tabs are the containers for editors in the editor area. Multiple tabs can be opened in one editor group. There can be multiple editor groups.'
        }, {
            id: 'tab.inactiveModifiedBorder', defaults: {
                dark: color_registry_1.Color.transparent('tab.activeModifiedBorder', 0.5),
                light: color_registry_1.Color.transparent('tab.activeModifiedBorder', 0.5),
                hc: color_registry_1.Color.white
            }, description: 'Border on the top of modified (dirty) inactive tabs in an active group. Tabs are the containers for editors in the editor area. Multiple tabs can be opened in one editor group. There can be multiple editor groups.'
        }, {
            id: 'tab.unfocusedActiveModifiedBorder', defaults: {
                dark: color_registry_1.Color.transparent('tab.activeModifiedBorder', 0.5),
                light: color_registry_1.Color.transparent('tab.activeModifiedBorder', 0.7),
                hc: color_registry_1.Color.white
            }, description: 'Border on the top of modified (dirty) active tabs in an unfocused group. Tabs are the containers for editors in the editor area. Multiple tabs can be opened in one editor group. There can be multiple editor groups.'
        }, {
            id: 'tab.unfocusedInactiveModifiedBorder', defaults: {
                dark: color_registry_1.Color.transparent('tab.inactiveModifiedBorder', 0.5),
                light: color_registry_1.Color.transparent('tab.inactiveModifiedBorder', 0.5),
                hc: color_registry_1.Color.white
            }, description: 'Border on the top of modified (dirty) inactive tabs in an unfocused group. Tabs are the containers for editors in the editor area. Multiple tabs can be opened in one editor group. There can be multiple editor groups.'
        }, 
        // Status bar colors should be aligned with https://code.visualstudio.com/api/references/theme-color#status-bar-colors
        // Not yet supported:
        // statusBarItem.prominentForeground, statusBarItem.prominentBackground, statusBarItem.prominentHoverBackground
        {
            id: 'statusBar.foreground', defaults: {
                dark: '#FFFFFF',
                light: '#FFFFFF',
                hc: '#FFFFFF'
            }, description: 'Status bar foreground color when a workspace is opened. The status bar is shown in the bottom of the window.'
        }, {
            id: 'statusBar.background', defaults: {
                dark: '#007ACC',
                light: '#007ACC'
            }, description: 'Status bar background color when a workspace is opened. The status bar is shown in the bottom of the window.'
        }, {
            id: 'statusBar.noFolderForeground', defaults: {
                dark: 'statusBar.foreground',
                light: 'statusBar.foreground',
                hc: 'statusBar.foreground'
            }, description: 'Status bar foreground color when no folder is opened. The status bar is shown in the bottom of the window.'
        }, {
            id: 'statusBar.noFolderBackground', defaults: {
                dark: '#68217A',
                light: '#68217A'
            }, description: 'Status bar background color when no folder is opened. The status bar is shown in the bottom of the window.'
        }, {
            id: 'statusBar.border', defaults: {
                hc: 'contrastBorder'
            }, description: 'Status bar border color separating to the sidebar and editor. The status bar is shown in the bottom of the window.'
        }, {
            id: 'statusBar.noFolderBorder', defaults: {
                dark: 'statusBar.border',
                light: 'statusBar.border',
                hc: 'statusBar.border'
            }, description: 'Status bar border color separating to the sidebar and editor when no folder is opened. The status bar is shown in the bottom of the window.'
        }, {
            id: 'statusBarItem.activeBackground', defaults: {
                dark: color_registry_1.Color.rgba(255, 255, 255, 0.18),
                light: color_registry_1.Color.rgba(255, 255, 255, 0.18),
                hc: color_registry_1.Color.rgba(255, 255, 255, 0.18)
            }, description: 'Status bar item background color when clicking. The status bar is shown in the bottom of the window.'
        }, {
            id: 'statusBarItem.hoverBackground', defaults: {
                dark: color_registry_1.Color.rgba(255, 255, 255, 0.12),
                light: color_registry_1.Color.rgba(255, 255, 255, 0.12),
                hc: color_registry_1.Color.rgba(255, 255, 255, 0.12)
            }, description: 'Status bar item background color when hovering. The status bar is shown in the bottom of the window.'
        }, 
        // Quickinput colors should be aligned with https://code.visualstudio.com/api/references/theme-color#quick-picker
        // if not yet contributed by Monaco, check runtime css variables to learn.
        {
            id: 'quickInput.background', defaults: {
                dark: 'sideBar.background',
                light: 'sideBar.background',
                hc: 'sideBar.background'
            }, description: 'Quick Input background color. The Quick Input widget is the container for views like the color theme picker.'
        }, {
            id: 'quickInput.foreground', defaults: {
                dark: 'sideBar.foreground',
                light: 'sideBar.foreground',
                hc: 'sideBar.foreground'
            }, description: 'Quick Input foreground color. The Quick Input widget is the container for views like the color theme picker.'
        }, {
            id: 'quickInput.list.focusBackground', defaults: {
                dark: undefined,
                light: undefined,
                hc: undefined
            }, description: 'quickInput.list.focusBackground deprecation. Please use quickInputList.focusBackground instead'
        }, {
            id: 'quickInputList.focusForeground', defaults: {
                dark: 'list.activeSelectionForeground',
                light: 'list.activeSelectionForeground',
                hc: 'list.activeSelectionForeground'
            }, description: 'Quick picker foreground color for the focused item'
        }, {
            id: 'quickInputList.focusBackground', defaults: {
                dark: 'list.activeSelectionBackground',
                light: 'list.activeSelectionBackground',
                hc: undefined
            }, description: 'Quick picker background color for the focused item.'
        }, 
        // Panel colors should be aligned with https://code.visualstudio.com/api/references/theme-color#panel-colors
        {
            id: 'panel.background', defaults: {
                dark: 'editor.background', light: 'editor.background', hc: 'editor.background'
            }, description: 'Panel background color. Panels are shown below the editor area and contain views like output and integrated terminal.'
        }, {
            id: 'panel.border', defaults: {
                dark: color_registry_1.Color.transparent('#808080', 0.35), light: color_registry_1.Color.transparent('#808080', 0.35), hc: 'contrastBorder'
            }, description: 'Panel border color to separate the panel from the editor. Panels are shown below the editor area and contain views like output and integrated terminal.'
        }, {
            id: 'panel.dropBackground', defaults: {
                dark: color_registry_1.Color.rgba(255, 255, 255, 0.12), light: color_registry_1.Color.transparent('#2677CB', 0.18), hc: color_registry_1.Color.rgba(255, 255, 255, 0.12)
            }, description: 'Drag and drop feedback color for the panel title items. The color should have transparency so that the panel entries can still shine through. Panels are shown below the editor area and contain views like output and integrated terminal.'
        }, {
            id: 'panelTitle.activeForeground', defaults: {
                dark: '#E7E7E7', light: '#424242', hc: color_registry_1.Color.white
            }, description: 'Title color for the active panel. Panels are shown below the editor area and contain views like output and integrated terminal.'
        }, {
            id: 'panelTitle.inactiveForeground', defaults: {
                dark: color_registry_1.Color.transparent('panelTitle.activeForeground', 0.6), light: color_registry_1.Color.transparent('panelTitle.activeForeground', 0.75), hc: color_registry_1.Color.white
            }, description: 'Title color for the inactive panel. Panels are shown below the editor area and contain views like output and integrated terminal.'
        }, {
            id: 'panelTitle.activeBorder', defaults: {
                dark: 'panelTitle.activeForeground', light: 'panelTitle.activeForeground', hc: 'contrastBorder'
            }, description: 'Border color for the active panel title. Panels are shown below the editor area and contain views like output and integrated terminal.'
        }, {
            id: 'panelInput.border', defaults: { light: '#ddd' },
            description: 'Input box border for inputs in the panel.'
        }, {
            id: 'imagePreview.border', defaults: {
                dark: color_registry_1.Color.transparent('#808080', 0.35), light: color_registry_1.Color.transparent('#808080', 0.35), hc: 'contrastBorder'
            }, description: 'Border color for image in image preview.'
        }, 
        // Title Bar colors should be aligned with https://code.visualstudio.com/api/references/theme-color#title-bar-colors
        {
            id: 'titleBar.activeForeground', defaults: {
                dark: '#CCCCCC',
                light: '#333333',
                hc: '#FFFFFF'
            }, description: 'Title bar foreground when the window is active. Note that this color is currently only supported on macOS.'
        }, {
            id: 'titleBar.inactiveForeground', defaults: {
                dark: color_registry_1.Color.transparent('titleBar.activeForeground', 0.6),
                light: color_registry_1.Color.transparent('titleBar.activeForeground', 0.6)
            }, description: 'Title bar foreground when the window is inactive. Note that this color is currently only supported on macOS.'
        }, {
            id: 'titleBar.activeBackground', defaults: {
                dark: '#3C3C3C',
                light: '#DDDDDD',
                hc: '#000000'
            }, description: 'Title bar background when the window is active. Note that this color is currently only supported on macOS.'
        }, {
            id: 'titleBar.inactiveBackground', defaults: {
                dark: color_registry_1.Color.transparent('titleBar.activeBackground', 0.6),
                light: color_registry_1.Color.transparent('titleBar.activeBackground', 0.6)
            }, description: 'Title bar background when the window is inactive. Note that this color is currently only supported on macOS.'
        }, {
            id: 'titleBar.border', defaults: {
                hc: 'contrastBorder'
            }, description: 'Title bar border color. Note that this color is currently only supported on macOS.'
        }, 
        // Menu Bar colors should be aligned with https://code.visualstudio.com/api/references/theme-color#menu-bar-colors
        {
            id: 'menubar.selectionForeground', defaults: {
                dark: 'titleBar.activeForeground',
                light: 'titleBar.activeForeground',
                hc: 'titleBar.activeForeground'
            }, description: 'Foreground color of the selected menu item in the menubar.'
        }, {
            id: 'menubar.selectionBackground', defaults: {
                dark: color_registry_1.Color.transparent('#ffffff', 0.1),
                light: color_registry_1.Color.transparent('#000000', 0.1)
            }, description: 'Background color of the selected menu item in the menubar.'
        }, {
            id: 'menubar.selectionBorder', defaults: {
                hc: 'activeContrastBorder'
            }, description: 'Border color of the selected menu item in the menubar.'
        }, {
            id: 'menu.border', defaults: {
                hc: 'contrastBorder'
            }, description: 'Border color of menus.'
        }, {
            id: 'menu.foreground', defaults: {
                dark: 'dropdown.foreground', light: 'foreground', hc: 'dropdown.foreground'
            },
            description: 'Foreground color of menu items.'
        }, {
            id: 'menu.background', defaults: {
                dark: 'dropdown.background', light: 'dropdown.background', hc: 'dropdown.background'
            }, description: 'Background color of menu items.'
        }, {
            id: 'menu.selectionForeground', defaults: {
                dark: 'list.activeSelectionForeground', light: 'list.activeSelectionForeground', hc: 'list.activeSelectionForeground'
            }, description: 'Foreground color of the selected menu item in menus.'
        }, {
            id: 'menu.selectionBackground', defaults: { dark: 'list.activeSelectionBackground', light: 'list.activeSelectionBackground', hc: 'list.activeSelectionBackground' },
            description: 'Background color of the selected menu item in menus.'
        }, {
            id: 'menu.selectionBorder', defaults: {
                hc: 'activeContrastBorder'
            }, description: 'Border color of the selected menu item in menus.'
        }, {
            id: 'menu.separatorBackground', defaults: {
                dark: '#BBBBBB', light: '#888888', hc: 'contrastBorder'
            },
            description: 'Color of a separator menu item in menus.'
        }, 
        // Welcome Page colors should be aligned with https://code.visualstudio.com/api/references/theme-color#welcome-page
        { id: 'welcomePage.background', description: 'Background color for the Welcome page.' }, { id: 'welcomePage.buttonBackground', defaults: { dark: color_registry_1.Color.rgba(0, 0, 0, .2), light: color_registry_1.Color.rgba(0, 0, 0, .04), hc: color_registry_1.Color.black }, description: 'Background color for the buttons on the Welcome page.' }, { id: 'welcomePage.buttonHoverBackground', defaults: { dark: color_registry_1.Color.rgba(200, 235, 255, .072), light: color_registry_1.Color.rgba(0, 0, 0, .10) }, description: 'Hover background color for the buttons on the Welcome page.' }, { id: 'walkThrough.embeddedEditorBackground', defaults: { dark: color_registry_1.Color.rgba(0, 0, 0, .4), light: '#f4f4f4' }, description: 'Background color for the embedded editors on the Interactive Playground.' }, 
        // Settings Editor colors should be aligned with https://code.visualstudio.com/api/references/theme-color#settings-editor-colors
        {
            id: 'settings.headerForeground', defaults: {
                light: '#444444', dark: '#e7e7e7', hc: '#ffffff'
            }, description: 'The foreground color for a section header or active title.'
        }, {
            id: 'settings.modifiedItemIndicator', defaults: {
                light: color_registry_1.Color.rgba(102, 175, 224),
                dark: color_registry_1.Color.rgba(12, 125, 157),
                hc: color_registry_1.Color.rgba(0, 73, 122)
            }, description: 'The color of the modified setting indicator.'
        }, {
            id: 'settings.dropdownBackground', defaults: { dark: 'dropdown.background', light: 'dropdown.background', hc: 'dropdown.background' },
            description: 'Settings editor dropdown background.'
        }, {
            id: 'settings.dropdownForeground', defaults: {
                dark: 'dropdown.foreground', light: 'dropdown.foreground', hc: 'dropdown.foreground'
            }, description: 'Settings editor dropdown foreground.'
        }, {
            id: 'settings.dropdownBorder', defaults: {
                dark: 'dropdown.border', light: 'dropdown.border', hc: 'dropdown.border'
            }, description: 'Settings editor dropdown border.'
        }, {
            id: 'settings.dropdownListBorder', defaults: {
                dark: 'editorWidget.border', light: 'editorWidget.border', hc: 'editorWidget.border'
            }, description: 'Settings editor dropdown list border. This surrounds the options and separates the options from the description.'
        }, {
            id: 'settings.checkboxBackground', defaults: {
                dark: 'checkbox.background', light: 'checkbox.background', hc: 'checkbox.background'
            }, description: 'Settings editor checkbox background.'
        }, {
            id: 'settings.checkboxForeground', defaults: {
                dark: 'checkbox.foreground', light: 'checkbox.foreground', hc: 'checkbox.foreground'
            }, description: 'Settings editor checkbox foreground.'
        }, {
            id: 'settings.checkboxBorder', defaults: {
                dark: 'checkbox.border', light: 'checkbox.border', hc: 'checkbox.border'
            }, description: 'Settings editor checkbox border.'
        }, {
            id: 'settings.textInputBackground', defaults: {
                dark: 'input.background', light: 'input.background', hc: 'input.background'
            }, description: 'Settings editor text input box background.'
        }, {
            id: 'settings.textInputForeground', defaults: {
                dark: 'input.foreground', light: 'input.foreground', hc: 'input.foreground'
            }, description: 'Settings editor text input box foreground.'
        }, {
            id: 'settings.textInputBorder', defaults: {
                dark: 'input.border', light: 'input.border', hc: 'input.border'
            }, description: 'Settings editor text input box border.'
        }, {
            id: 'settings.numberInputBackground', defaults: {
                dark: 'input.background', light: 'input.background', hc: 'input.background'
            }, description: 'Settings editor number input box background.'
        }, {
            id: 'settings.numberInputForeground', defaults: {
                dark: 'input.foreground', light: 'input.foreground', hc: 'input.foreground'
            }, description: 'Settings editor number input box foreground.'
        }, {
            id: 'settings.numberInputBorder', defaults: {
                dark: 'input.border', light: 'input.border', hc: 'input.border'
            }, description: 'Settings editor number input box border.'
        }, {
            id: 'settings.focusedRowBackground', defaults: {
                dark: color_registry_1.Color.transparent('#808080', 0.14),
                light: color_registry_1.Color.transparent('#808080', 0.03),
                hc: undefined
            }, description: 'The background color of a settings row when focused.'
        }, {
            id: 'settings.rowHoverBackground', defaults: {
                dark: color_registry_1.Color.transparent('#808080', 0.07),
                light: color_registry_1.Color.transparent('#808080', 0.05),
                hc: undefined
            }, description: 'The background color of a settings row when hovered.'
        }, {
            id: 'settings.focusedRowBorder', defaults: {
                dark: color_registry_1.Color.rgba(255, 255, 255, 0.12),
                light: color_registry_1.Color.rgba(0, 0, 0, 0.12),
                hc: 'focusBorder'
            }, description: "The color of the row's top and bottom border when the row is focused."
        }, 
        // Toolbar Action colors should be aligned with https://code.visualstudio.com/api/references/theme-color#action-colors
        {
            id: 'toolbar.hoverBackground', defaults: {
                dark: '#5a5d5e50', light: '#b8b8b850', hc: undefined
            }, description: 'Toolbar background when hovering over actions using the mouse.'
        }, 
        // Theia Variable colors
        {
            id: 'variable.name.color', defaults: {
                dark: '#C586C0',
                light: '#9B46B0',
                hc: '#C586C0'
            },
            description: 'Color of a variable name.'
        }, {
            id: 'variable.value.color', defaults: {
                dark: color_registry_1.Color.rgba(204, 204, 204, 0.6),
                light: color_registry_1.Color.rgba(108, 108, 108, 0.8),
                hc: color_registry_1.Color.rgba(204, 204, 204, 0.6)
            },
            description: 'Color of a variable value.'
        }, {
            id: 'variable.number.variable.color', defaults: {
                dark: '#B5CEA8',
                light: '#09885A',
                hc: '#B5CEA8'
            },
            description: 'Value color of a number variable'
        }, {
            id: 'variable.boolean.variable.color', defaults: {
                dark: '#4E94CE',
                light: '#0000FF',
                hc: '#4E94CE'
            },
            description: 'Value color of a boolean variable'
        }, {
            id: 'variable.string.variable.color', defaults: {
                dark: '#CE9178',
                light: '#A31515',
                hc: '#CE9178'
            },
            description: 'Value color of a string variable'
        }, 
        // Theia ANSI colors
        {
            id: 'ansi.black.color', defaults: {
                dark: '#A0A0A0',
                light: color_registry_1.Color.rgba(128, 128, 128),
                hc: '#A0A0A0'
            },
            description: 'ANSI black color'
        }, {
            id: 'ansi.red.color', defaults: {
                dark: '#A74747',
                light: '#BE1717',
                hc: '#A74747'
            },
            description: 'ANSI red color'
        }, {
            id: 'ansi.green.color', defaults: {
                dark: '#348F34',
                light: '#338A2F',
                hc: '#348F34'
            },
            description: 'ANSI green color'
        }, {
            id: 'ansi.yellow.color', defaults: {
                dark: '#5F4C29',
                light: '#BEB817',
                hc: '#5F4C29'
            },
            description: 'ANSI yellow color'
        }, {
            id: 'ansi.blue.color', defaults: {
                dark: '#6286BB',
                light: color_registry_1.Color.rgba(0, 0, 139),
                hc: '#6286BB'
            },
            description: 'ANSI blue color'
        }, {
            id: 'ansi.magenta.color', defaults: {
                dark: '#914191',
                light: color_registry_1.Color.rgba(139, 0, 139),
                hc: '#914191'
            },
            description: 'ANSI magenta color'
        }, {
            id: 'ansi.cyan.color', defaults: {
                dark: '#218D8D',
                light: color_registry_1.Color.rgba(0, 139, 139),
                hc: '#218D8D'
            },
            description: 'ANSI cyan color'
        }, {
            id: 'ansi.white.color', defaults: {
                dark: '#707070',
                light: '#BDBDBD',
                hc: '#707070'
            },
            description: 'ANSI white color'
        }, 
        // Theia defaults
        // Base
        {
            id: 'errorBackground',
            defaults: {
                dark: 'inputValidation.errorBackground',
                light: 'inputValidation.errorBackground',
                hc: 'inputValidation.errorBackground'
            }, description: 'Background color of error widgets (like alerts or notifications).'
        }, {
            id: 'successBackground',
            defaults: {
                dark: 'editorGutter.addedBackground',
                light: 'editorGutter.addedBackground',
                hc: 'editorGutter.addedBackground'
            }, description: 'Background color of success widgets (like alerts or notifications).'
        }, {
            id: 'warningBackground',
            defaults: {
                dark: 'editorWarning.foreground',
                light: 'editorWarning.foreground',
                hc: 'editorWarning.border'
            }, description: 'Background color of warning widgets (like alerts or notifications).'
        }, {
            id: 'warningForeground',
            defaults: {
                dark: 'inputValidation.warningBackground',
                light: 'inputValidation.warningBackground',
                hc: 'inputValidation.warningBackground'
            }, description: 'Foreground color of warning widgets (like alerts or notifications).'
        }, 
        // Statusbar
        {
            id: 'statusBar.offlineBackground',
            defaults: {
                dark: 'editorWarning.foreground',
                light: 'editorWarning.foreground',
                hc: 'editorWarning.foreground'
            }, description: 'Background of hovered statusbar item in case the theia server is offline.'
        }, {
            id: 'statusBar.offlineForeground',
            defaults: {
                dark: 'editor.background',
                light: 'editor.background',
                hc: 'editor.background'
            }, description: 'Background of hovered statusbar item in case the theia server is offline.'
        }, {
            id: 'statusBarItem.offlineHoverBackground',
            defaults: {
                dark: color_registry_1.Color.lighten('statusBar.offlineBackground', 0.4),
                light: color_registry_1.Color.lighten('statusBar.offlineBackground', 0.4),
                hc: color_registry_1.Color.lighten('statusBar.offlineBackground', 0.4)
            }, description: 'Background of hovered statusbar item in case the theia server is offline.'
        }, {
            id: 'statusBarItem.offlineActiveBackground',
            defaults: {
                dark: color_registry_1.Color.lighten('statusBar.offlineBackground', 0.6),
                light: color_registry_1.Color.lighten('statusBar.offlineBackground', 0.6),
                hc: color_registry_1.Color.lighten('statusBar.offlineBackground', 0.6)
            }, description: 'Background of active statusbar item in case the theia server is offline.'
        }, 
        // Buttons
        {
            id: 'secondaryButton.foreground',
            defaults: {
                dark: 'dropdown.foreground',
                light: 'dropdown.foreground',
                hc: 'dropdown.foreground'
            }, description: 'Foreground color of secondary buttons.'
        }, {
            id: 'secondaryButton.disabledForeground',
            defaults: {
                dark: color_registry_1.Color.transparent('secondaryButton.foreground', 0.5),
                light: color_registry_1.Color.transparent('secondaryButton.foreground', 0.5),
                hc: color_registry_1.Color.transparent('secondaryButton.foreground', 0.5),
            }, description: 'Foreground color of secondary buttons.'
        }, {
            id: 'secondaryButton.background',
            defaults: {
                dark: color_registry_1.Color.lighten('dropdown.background', 0.5),
                light: color_registry_1.Color.lighten('dropdown.background', 0.5)
            }, description: 'Background color of secondary buttons.'
        }, {
            id: 'secondaryButton.hoverBackground',
            defaults: {
                dark: color_registry_1.Color.lighten('secondaryButton.background', 0.2),
                light: color_registry_1.Color.lighten('secondaryButton.background', 0.2)
            }, description: 'Background color when hovering secondary buttons.'
        }, {
            id: 'secondaryButton.disabledBackground',
            defaults: {
                dark: color_registry_1.Color.transparent('secondaryButton.background', 0.6),
                light: color_registry_1.Color.transparent('secondaryButton.background', 0.6)
            }, description: 'Background color when hovering secondary buttons.'
        }, {
            id: 'button.disabledForeground',
            defaults: {
                dark: color_registry_1.Color.transparent('button.foreground', 0.5),
                light: color_registry_1.Color.transparent('button.foreground', 0.5),
                hc: color_registry_1.Color.transparent('button.foreground', 0.5)
            }, description: 'Foreground color of secondary buttons.'
        }, {
            id: 'button.disabledBackground',
            defaults: {
                dark: color_registry_1.Color.transparent('button.background', 0.5),
                light: color_registry_1.Color.transparent('button.background', 0.5)
            }, description: 'Background color of secondary buttons.'
        }, {
            id: 'editorGutter.commentRangeForeground',
            defaults: {
                dark: '#c5c5c5',
                light: '#c5c5c5',
                hc: '#c5c5c5'
            }, description: 'Editor gutter decoration color for commenting ranges.'
        }, {
            id: 'breadcrumb.foreground',
            defaults: {
                dark: color_registry_1.Color.transparent('foreground', 0.8),
                light: color_registry_1.Color.transparent('foreground', 0.8),
                hc: color_registry_1.Color.transparent('foreground', 0.8),
            },
            description: 'Color of breadcrumb item text'
        }, {
            id: 'breadcrumb.background',
            defaults: {
                dark: 'editor.background',
                light: 'editor.background',
                hc: 'editor.background',
            },
            description: 'Color of breadcrumb item background'
        }, {
            id: 'breadcrumb.focusForeground',
            defaults: {
                dark: color_registry_1.Color.lighten('foreground', 0.1),
                light: color_registry_1.Color.darken('foreground', 0.2),
                hc: color_registry_1.Color.lighten('foreground', 0.1),
            },
            description: 'Color of breadcrumb item text when focused'
        }, {
            id: 'breadcrumb.activeSelectionForeground',
            defaults: {
                dark: color_registry_1.Color.lighten('foreground', 0.1),
                light: color_registry_1.Color.darken('foreground', 0.2),
                hc: color_registry_1.Color.lighten('foreground', 0.1),
            },
            description: 'Color of selected breadcrumb item'
        }, {
            id: 'breadcrumbPicker.background',
            defaults: {
                dark: 'editorWidget.background',
                light: 'editorWidget.background',
                hc: 'editorWidget.background',
            },
            description: 'Background color of breadcrumb item picker'
        }, {
            id: 'mainToolbar.background',
            defaults: {
                dark: color_registry_1.Color.lighten('activityBar.background', 0.1),
                light: color_registry_1.Color.darken('activityBar.background', 0.1),
                hc: color_registry_1.Color.lighten('activityBar.background', 0.1),
            },
            description: 'Background color of shell\'s global toolbar'
        }, {
            id: 'mainToolbar.foreground', defaults: {
                dark: color_registry_1.Color.darken('activityBar.foreground', 0.1),
                light: color_registry_1.Color.lighten('activityBar.foreground', 0.1),
                hc: color_registry_1.Color.lighten('activityBar.foreground', 0.1),
            }, description: 'Foreground color of active toolbar item',
        });
    }
};
__decorate([
    (0, inversify_1.inject)(context_key_service_1.ContextKeyService),
    __metadata("design:type", Object)
], CommonFrontendContribution.prototype, "contextKeyService", void 0);
__decorate([
    (0, inversify_1.inject)(resource_context_key_1.ResourceContextKey),
    __metadata("design:type", resource_context_key_1.ResourceContextKey)
], CommonFrontendContribution.prototype, "resourceContextKey", void 0);
__decorate([
    (0, inversify_1.inject)(command_1.CommandRegistry),
    __metadata("design:type", command_1.CommandRegistry)
], CommonFrontendContribution.prototype, "commandRegistry", void 0);
__decorate([
    (0, inversify_1.inject)(storage_service_1.StorageService),
    __metadata("design:type", Object)
], CommonFrontendContribution.prototype, "storageService", void 0);
__decorate([
    (0, inversify_1.inject)(quick_input_1.QuickInputService),
    (0, inversify_1.optional)(),
    __metadata("design:type", Object)
], CommonFrontendContribution.prototype, "quickInputService", void 0);
__decorate([
    (0, inversify_1.inject)(icon_theme_service_1.IconThemeService),
    __metadata("design:type", icon_theme_service_1.IconThemeService)
], CommonFrontendContribution.prototype, "iconThemes", void 0);
__decorate([
    (0, inversify_1.inject)(theming_1.ThemeService),
    __metadata("design:type", theming_1.ThemeService)
], CommonFrontendContribution.prototype, "themeService", void 0);
__decorate([
    (0, inversify_1.inject)(core_preferences_1.CorePreferences),
    __metadata("design:type", Object)
], CommonFrontendContribution.prototype, "preferences", void 0);
__decorate([
    (0, inversify_1.inject)(preferences_1.PreferenceService),
    __metadata("design:type", Object)
], CommonFrontendContribution.prototype, "preferenceService", void 0);
__decorate([
    (0, inversify_1.inject)(clipboard_service_1.ClipboardService),
    __metadata("design:type", Object)
], CommonFrontendContribution.prototype, "clipboardService", void 0);
__decorate([
    (0, inversify_1.inject)(encoding_registry_1.EncodingRegistry),
    __metadata("design:type", encoding_registry_1.EncodingRegistry)
], CommonFrontendContribution.prototype, "encodingRegistry", void 0);
__decorate([
    (0, inversify_1.inject)(env_variables_1.EnvVariablesServer),
    __metadata("design:type", Object)
], CommonFrontendContribution.prototype, "environments", void 0);
__decorate([
    (0, inversify_1.inject)(authentication_service_1.AuthenticationService),
    __metadata("design:type", Object)
], CommonFrontendContribution.prototype, "authenticationService", void 0);
__decorate([
    (0, inversify_1.inject)(window_service_1.WindowService),
    __metadata("design:type", Object)
], CommonFrontendContribution.prototype, "windowService", void 0);
CommonFrontendContribution = __decorate([
    (0, inversify_1.injectable)(),
    __param(0, (0, inversify_1.inject)(application_shell_1.ApplicationShell)),
    __param(1, (0, inversify_1.inject)(selection_service_1.SelectionService)),
    __param(2, (0, inversify_1.inject)(message_service_1.MessageService)),
    __param(3, (0, inversify_1.inject)(opener_service_1.OpenerService)),
    __param(4, (0, inversify_1.inject)(about_dialog_1.AboutDialog)),
    __param(5, (0, inversify_1.inject)(localization_1.AsyncLocalizationProvider)),
    __param(6, (0, inversify_1.inject)(save_resource_service_1.SaveResourceService)),
    __metadata("design:paramtypes", [application_shell_1.ApplicationShell,
        selection_service_1.SelectionService,
        message_service_1.MessageService, Object, about_dialog_1.AboutDialog, Object, save_resource_service_1.SaveResourceService])
], CommonFrontendContribution);
exports.CommonFrontendContribution = CommonFrontendContribution;
//# sourceMappingURL=common-frontend-contribution.js.map