"use strict";
/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
Object.defineProperty(exports, "__esModule", { value: true });
exports.BracketPairsTree = void 0;
const event_1 = require("../../../../../base/common/event");
const lifecycle_1 = require("../../../../../base/common/lifecycle");
const range_1 = require("../../../core/range");
const textModelBracketPairs_1 = require("../../../textModelBracketPairs");
const beforeEditPositionMapper_1 = require("./beforeEditPositionMapper");
const brackets_1 = require("./brackets");
const length_1 = require("./length");
const parser_1 = require("./parser");
const smallImmutableSet_1 = require("./smallImmutableSet");
const tokenizer_1 = require("./tokenizer");
class BracketPairsTree extends lifecycle_1.Disposable {
    constructor(textModel, getLanguageConfiguration) {
        super();
        this.textModel = textModel;
        this.getLanguageConfiguration = getLanguageConfiguration;
        this.didChangeEmitter = new event_1.Emitter();
        this.denseKeyProvider = new smallImmutableSet_1.DenseKeyProvider();
        this.brackets = new brackets_1.LanguageAgnosticBracketTokens(this.denseKeyProvider, this.getLanguageConfiguration);
        this.onDidChange = this.didChangeEmitter.event;
        if (textModel.backgroundTokenizationState === 0 /* Uninitialized */) {
            // There are no token information yet
            const brackets = this.brackets.getSingleLanguageBracketTokens(this.textModel.getLanguageId());
            const tokenizer = new tokenizer_1.FastTokenizer(this.textModel.getValue(), brackets);
            this.initialAstWithoutTokens = (0, parser_1.parseDocument)(tokenizer, [], undefined, true);
            this.astWithTokens = this.initialAstWithoutTokens;
        }
        else if (textModel.backgroundTokenizationState === 2 /* Completed */) {
            // Skip the initial ast, as there is no flickering.
            // Directly create the tree with token information.
            this.initialAstWithoutTokens = undefined;
            this.astWithTokens = this.parseDocumentFromTextBuffer([], undefined, false);
        }
        else if (textModel.backgroundTokenizationState === 1 /* InProgress */) {
            this.initialAstWithoutTokens = this.parseDocumentFromTextBuffer([], undefined, true);
            this.astWithTokens = this.initialAstWithoutTokens;
        }
    }
    didLanguageChange(languageId) {
        return this.brackets.didLanguageChange(languageId);
    }
    //#region TextModel events
    handleDidChangeBackgroundTokenizationState() {
        if (this.textModel.backgroundTokenizationState === 2 /* Completed */) {
            const wasUndefined = this.initialAstWithoutTokens === undefined;
            // Clear the initial tree as we can use the tree with token information now.
            this.initialAstWithoutTokens = undefined;
            if (!wasUndefined) {
                this.didChangeEmitter.fire();
            }
        }
    }
    handleDidChangeTokens({ ranges }) {
        const edits = ranges.map(r => new beforeEditPositionMapper_1.TextEditInfo((0, length_1.toLength)(r.fromLineNumber - 1, 0), (0, length_1.toLength)(r.toLineNumber, 0), (0, length_1.toLength)(r.toLineNumber - r.fromLineNumber + 1, 0)));
        this.astWithTokens = this.parseDocumentFromTextBuffer(edits, this.astWithTokens, false);
        if (!this.initialAstWithoutTokens) {
            this.didChangeEmitter.fire();
        }
    }
    handleContentChanged(change) {
        const edits = change.changes.map(c => {
            const range = range_1.Range.lift(c.range);
            return new beforeEditPositionMapper_1.TextEditInfo((0, length_1.positionToLength)(range.getStartPosition()), (0, length_1.positionToLength)(range.getEndPosition()), (0, length_1.lengthOfString)(c.text));
        }).reverse();
        this.astWithTokens = this.parseDocumentFromTextBuffer(edits, this.astWithTokens, false);
        if (this.initialAstWithoutTokens) {
            this.initialAstWithoutTokens = this.parseDocumentFromTextBuffer(edits, this.initialAstWithoutTokens, false);
        }
    }
    //#endregion
    /**
     * @pure (only if isPure = true)
    */
    parseDocumentFromTextBuffer(edits, previousAst, immutable) {
        // Is much faster if `isPure = false`.
        const isPure = false;
        const previousAstClone = isPure ? previousAst === null || previousAst === void 0 ? void 0 : previousAst.deepClone() : previousAst;
        const tokenizer = new tokenizer_1.TextBufferTokenizer(this.textModel, this.brackets);
        const result = (0, parser_1.parseDocument)(tokenizer, edits, previousAstClone, immutable);
        return result;
    }
    getBracketsInRange(range) {
        const startOffset = (0, length_1.toLength)(range.startLineNumber - 1, range.startColumn - 1);
        const endOffset = (0, length_1.toLength)(range.endLineNumber - 1, range.endColumn - 1);
        const result = new Array();
        const node = this.initialAstWithoutTokens || this.astWithTokens;
        collectBrackets(node, length_1.lengthZero, node.length, startOffset, endOffset, result);
        return result;
    }
    getBracketPairsInRange(range, includeMinIndentation) {
        const result = new Array();
        const startLength = (0, length_1.positionToLength)(range.getStartPosition());
        const endLength = (0, length_1.positionToLength)(range.getEndPosition());
        const node = this.initialAstWithoutTokens || this.astWithTokens;
        const context = new CollectBracketPairsContext(result, includeMinIndentation, this.textModel);
        collectBracketPairs(node, length_1.lengthZero, node.length, startLength, endLength, context);
        return result;
    }
}
exports.BracketPairsTree = BracketPairsTree;
function collectBrackets(node, nodeOffsetStart, nodeOffsetEnd, startOffset, endOffset, result, level = 0) {
    if (node.kind === 4 /* List */) {
        for (const child of node.children) {
            nodeOffsetEnd = (0, length_1.lengthAdd)(nodeOffsetStart, child.length);
            if ((0, length_1.lengthLessThanEqual)(nodeOffsetStart, endOffset) && (0, length_1.lengthGreaterThanEqual)(nodeOffsetEnd, startOffset)) {
                collectBrackets(child, nodeOffsetStart, nodeOffsetEnd, startOffset, endOffset, result, level);
            }
            nodeOffsetStart = nodeOffsetEnd;
        }
    }
    else if (node.kind === 2 /* Pair */) {
        // Don't use node.children here to improve performance
        level++;
        {
            const child = node.openingBracket;
            nodeOffsetEnd = (0, length_1.lengthAdd)(nodeOffsetStart, child.length);
            if ((0, length_1.lengthLessThanEqual)(nodeOffsetStart, endOffset) && (0, length_1.lengthGreaterThanEqual)(nodeOffsetEnd, startOffset)) {
                const range = (0, length_1.lengthsToRange)(nodeOffsetStart, nodeOffsetEnd);
                result.push(new textModelBracketPairs_1.BracketInfo(range, level - 1, !node.closingBracket));
            }
            nodeOffsetStart = nodeOffsetEnd;
        }
        if (node.child) {
            const child = node.child;
            nodeOffsetEnd = (0, length_1.lengthAdd)(nodeOffsetStart, child.length);
            if ((0, length_1.lengthLessThanEqual)(nodeOffsetStart, endOffset) && (0, length_1.lengthGreaterThanEqual)(nodeOffsetEnd, startOffset)) {
                collectBrackets(child, nodeOffsetStart, nodeOffsetEnd, startOffset, endOffset, result, level);
            }
            nodeOffsetStart = nodeOffsetEnd;
        }
        if (node.closingBracket) {
            const child = node.closingBracket;
            nodeOffsetEnd = (0, length_1.lengthAdd)(nodeOffsetStart, child.length);
            if ((0, length_1.lengthLessThanEqual)(nodeOffsetStart, endOffset) && (0, length_1.lengthGreaterThanEqual)(nodeOffsetEnd, startOffset)) {
                const range = (0, length_1.lengthsToRange)(nodeOffsetStart, nodeOffsetEnd);
                result.push(new textModelBracketPairs_1.BracketInfo(range, level - 1, false));
            }
            nodeOffsetStart = nodeOffsetEnd;
        }
    }
    else if (node.kind === 3 /* UnexpectedClosingBracket */) {
        const range = (0, length_1.lengthsToRange)(nodeOffsetStart, nodeOffsetEnd);
        result.push(new textModelBracketPairs_1.BracketInfo(range, level - 1, true));
    }
    else if (node.kind === 1 /* Bracket */) {
        const range = (0, length_1.lengthsToRange)(nodeOffsetStart, nodeOffsetEnd);
        result.push(new textModelBracketPairs_1.BracketInfo(range, level - 1, false));
    }
}
class CollectBracketPairsContext {
    constructor(result, includeMinIndentation, textModel) {
        this.result = result;
        this.includeMinIndentation = includeMinIndentation;
        this.textModel = textModel;
    }
}
function collectBracketPairs(node, nodeOffset, nodeOffsetEnd, startOffset, endOffset, context, level = 0) {
    var _a;
    if (node.kind === 2 /* Pair */) {
        const openingBracketEnd = (0, length_1.lengthAdd)(nodeOffset, node.openingBracket.length);
        let minIndentation = -1;
        if (context.includeMinIndentation) {
            minIndentation = node.computeMinIndentation(nodeOffset, context.textModel);
        }
        context.result.push(new textModelBracketPairs_1.BracketPairWithMinIndentationInfo((0, length_1.lengthsToRange)(nodeOffset, nodeOffsetEnd), (0, length_1.lengthsToRange)(nodeOffset, openingBracketEnd), node.closingBracket
            ? (0, length_1.lengthsToRange)((0, length_1.lengthAdd)(openingBracketEnd, ((_a = node.child) === null || _a === void 0 ? void 0 : _a.length) || length_1.lengthZero), nodeOffsetEnd)
            : undefined, level, minIndentation));
        level++;
    }
    let curOffset = nodeOffset;
    for (const child of node.children) {
        const childOffset = curOffset;
        curOffset = (0, length_1.lengthAdd)(curOffset, child.length);
        if ((0, length_1.lengthLessThanEqual)(childOffset, endOffset) && (0, length_1.lengthLessThanEqual)(startOffset, curOffset)) {
            collectBracketPairs(child, childOffset, curOffset, startOffset, endOffset, context, level);
        }
    }
}
//# sourceMappingURL=bracketPairsTree.js.map