"use strict";
/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
Object.defineProperty(exports, "__esModule", { value: true });
exports.fixBracketsInLine = void 0;
const brackets_1 = require("./bracketPairsTree/brackets");
const length_1 = require("./bracketPairsTree/length");
const parser_1 = require("./bracketPairsTree/parser");
const smallImmutableSet_1 = require("./bracketPairsTree/smallImmutableSet");
const tokenizer_1 = require("./bracketPairsTree/tokenizer");
function fixBracketsInLine(tokens, languageConfigurationService) {
    const denseKeyProvider = new smallImmutableSet_1.DenseKeyProvider();
    const bracketTokens = new brackets_1.LanguageAgnosticBracketTokens(denseKeyProvider, (languageId) => languageConfigurationService.getLanguageConfiguration(languageId));
    const tokenizer = new tokenizer_1.TextBufferTokenizer(new StaticTokenizerSource([tokens]), bracketTokens);
    const node = (0, parser_1.parseDocument)(tokenizer, [], undefined, true);
    let str = '';
    const line = tokens.getLineContent();
    function processNode(node, offset) {
        if (node.kind === 2 /* Pair */) {
            processNode(node.openingBracket, offset);
            offset = (0, length_1.lengthAdd)(offset, node.openingBracket.length);
            if (node.child) {
                processNode(node.child, offset);
                offset = (0, length_1.lengthAdd)(offset, node.child.length);
            }
            if (node.closingBracket) {
                processNode(node.closingBracket, offset);
                offset = (0, length_1.lengthAdd)(offset, node.closingBracket.length);
            }
            else {
                const singleLangBracketTokens = bracketTokens.getSingleLanguageBracketTokens(node.openingBracket.languageId);
                const closingTokenText = singleLangBracketTokens.findClosingTokenText(node.openingBracket.bracketIds);
                str += closingTokenText;
            }
        }
        else if (node.kind === 3 /* UnexpectedClosingBracket */) {
            // remove the bracket
        }
        else if (node.kind === 0 /* Text */ || node.kind === 1 /* Bracket */) {
            str += line.substring((0, length_1.lengthGetColumnCountIfZeroLineCount)(offset), (0, length_1.lengthGetColumnCountIfZeroLineCount)((0, length_1.lengthAdd)(offset, node.length)));
        }
        else if (node.kind === 4 /* List */) {
            for (const child of node.children) {
                processNode(child, offset);
                offset = (0, length_1.lengthAdd)(offset, child.length);
            }
        }
    }
    processNode(node, length_1.lengthZero);
    return str;
}
exports.fixBracketsInLine = fixBracketsInLine;
class StaticTokenizerSource {
    constructor(lines) {
        this.lines = lines;
    }
    getValue() {
        return this.lines.map(l => l.getLineContent()).join('\n');
    }
    getLineCount() {
        return this.lines.length;
    }
    getLineLength(lineNumber) {
        return this.lines[lineNumber - 1].getLineContent().length;
    }
    getLineTokens(lineNumber) {
        return this.lines[lineNumber - 1];
    }
}
//# sourceMappingURL=fixBrackets.js.map