"use strict";
// *****************************************************************************
// Copyright (C) 2018 Red Hat, Inc. and others.
//
// This program and the accompanying materials are made available under the
// terms of the Eclipse Public License v. 2.0 which is available at
// http://www.eclipse.org/legal/epl-2.0.
//
// This Source Code may also be made available under the following Secondary
// Licenses when the conditions for such availability set forth in the Eclipse
// Public License v. 2.0 are satisfied: GNU General Public License, version 2
// with the GNU Classpath Exception which is available at
// https://www.gnu.org/software/classpath/license.html.
//
// SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
// *****************************************************************************
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var GithubPluginDeployerResolver_1;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GithubPluginDeployerResolver = void 0;
const inversify_1 = require("@theia/core/shared/inversify");
const fs = require("fs");
const os = require("os");
const path = require("path");
const request = require("request");
/**
 * Resolver that handle the github: protocol
 * github:<org>/<repo>/<filename>@latest
 * github:<org>/<repo>/<filename>@<version>
 */
let GithubPluginDeployerResolver = GithubPluginDeployerResolver_1 = class GithubPluginDeployerResolver {
    constructor() {
        this.unpackedFolder = path.resolve(os.tmpdir(), 'github-remote');
        if (!fs.existsSync(this.unpackedFolder)) {
            fs.mkdirSync(this.unpackedFolder);
        }
    }
    /**
     * Grab the remote file specified by Github URL
     */
    async resolve(pluginResolverContext) {
        // download the file
        return new Promise((resolve, reject) => {
            // extract data
            const extracted = /^github:(.*)\/(.*)\/(.*)$/gm.exec(pluginResolverContext.getOriginId());
            if (!extracted || extracted === null || extracted.length !== 4) {
                reject(new Error('Invalid extension' + pluginResolverContext.getOriginId()));
                return;
            }
            const orgName = extracted[1];
            const repoName = extracted[2];
            const file = extracted[3];
            // get version if any
            const splitFile = file.split('@');
            let version;
            let filename;
            if (splitFile.length === 1) {
                filename = file;
                version = 'latest';
            }
            else {
                filename = splitFile[0];
                version = splitFile[1];
            }
            // latest version, need to get the redirect
            const url = GithubPluginDeployerResolver_1.GITHUB_ENDPOINT + orgName + '/' + repoName + '/releases/latest';
            // disable redirect to grab the release
            const options = {
                followRedirect: false
            };
            // if latest, resolve first the real version
            if (version === 'latest') {
                request.get(url, options).on('response', response => {
                    // should have a redirect
                    if (response.statusCode === 302) {
                        const redirectLocation = response.headers.location;
                        if (!redirectLocation) {
                            reject(new Error('Invalid github link with latest not being found'));
                            return;
                        }
                        // parse redirect link
                        const taggedValueArray = /^https:\/\/.*tag\/(.*)/gm.exec(redirectLocation);
                        if (!taggedValueArray || taggedValueArray.length !== 2) {
                            reject(new Error('The redirect link for latest is invalid ' + redirectLocation));
                            return;
                        }
                        // grab version of tag
                        this.grabGithubFile(pluginResolverContext, orgName, repoName, filename, taggedValueArray[1], resolve, reject);
                    }
                });
            }
            else {
                this.grabGithubFile(pluginResolverContext, orgName, repoName, filename, version, resolve, reject);
            }
        });
    }
    /*
     * Grab the github file specified by the plugin's ID
     */
    grabGithubFile(pluginResolverContext, orgName, repoName, filename, version, 
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    resolve, reject) {
        const unpackedPath = path.resolve(this.unpackedFolder, path.basename(version + filename));
        const finish = () => {
            pluginResolverContext.addPlugin(pluginResolverContext.getOriginId(), unpackedPath);
            resolve();
        };
        // use of cache. If file is already there use it directly
        if (fs.existsSync(unpackedPath)) {
            finish();
            return;
        }
        const dest = fs.createWriteStream(unpackedPath);
        dest.addListener('finish', finish);
        const url = GithubPluginDeployerResolver_1.GITHUB_ENDPOINT + orgName + '/' + repoName + '/releases/download/' + version + '/' + filename;
        request.get(url)
            .on('error', err => {
            reject(err);
        }).pipe(dest);
    }
    /**
     * Handle only the plugins that starts with github:
     */
    accept(pluginId) {
        return pluginId.startsWith(GithubPluginDeployerResolver_1.PREFIX);
    }
};
GithubPluginDeployerResolver.PREFIX = 'github:';
GithubPluginDeployerResolver.GITHUB_ENDPOINT = 'https://github.com/';
GithubPluginDeployerResolver = GithubPluginDeployerResolver_1 = __decorate([
    (0, inversify_1.injectable)(),
    __metadata("design:paramtypes", [])
], GithubPluginDeployerResolver);
exports.GithubPluginDeployerResolver = GithubPluginDeployerResolver;
//# sourceMappingURL=plugin-github-resolver.js.map