"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TspDataProvider = void 0;
const query_helper_1 = require("tsp-typescript-client/lib/models/query/query-helper");
const annotation_1 = require("tsp-typescript-client/lib/models/annotation");
var ElementType;
(function (ElementType) {
    ElementType["STATE"] = "state";
    ElementType["ANNOTATION"] = "annotation";
})(ElementType || (ElementType = {}));
class TspDataProvider {
    constructor(client, traceUUID, outputId) {
        this.timeGraphEntries = [];
        this.timeGraphRows = [];
        this.client = client;
        this.outputId = outputId;
        this.traceUUID = traceUUID;
        this.totalRange = BigInt(0);
    }
    /**
     * @param ids requested entry ids
     * @param entries time graph entries
     * @param totalTimeRange total time range
     * @param viewRange requested view range relative to start of total time range
     * @param nbTimes number of requested time samples
     * @param annotationMarkers requested annotation categories
     * @returns time graph model
     */
    async getData(ids, entries, totalTimeRange, viewRange, nbTimes, annotationMarkers, markerSetId) {
        this.timeGraphEntries = [...entries];
        if (!this.timeGraphEntries.length || !viewRange || !nbTimes) {
            return {
                id: 'model',
                totalLength: this.totalRange,
                rows: [],
                rangeEvents: [],
                arrows: [],
                data: {}
            };
        }
        this.totalRange = totalTimeRange.getEnd() - totalTimeRange.getStart();
        const start = totalTimeRange.getStart() + viewRange.start;
        const end = totalTimeRange.getStart() + viewRange.end;
        const timeGraphStateParams = query_helper_1.QueryHelper.selectionTimeQuery(query_helper_1.QueryHelper.splitRangeIntoEqualParts(start, end, nbTimes), ids);
        const tspClientResponse = await this.client.fetchTimeGraphStates(this.traceUUID, this.outputId, timeGraphStateParams);
        const stateResponse = tspClientResponse.getModel();
        if (tspClientResponse.isOk() && stateResponse) {
            this.timeGraphRows = stateResponse.model.rows;
            this.timeGraphRowsOrdering(ids);
        }
        else {
            this.timeGraphRows = [];
        }
        // the start time which is normalized to logical 0 in timeline chart.
        const chartStart = totalTimeRange.getStart();
        const rows = [];
        this.timeGraphRows.forEach((row) => {
            const rowId = row.entryId;
            const entry = this.timeGraphEntries.find(tgEntry => tgEntry.id === rowId);
            if (entry) {
                rows.push(this.getRowModel(row, chartStart, rowId, entry));
            }
        });
        const additionalProps = {};
        if (annotationMarkers) {
            additionalProps['requested_marker_categories'] = annotationMarkers;
        }
        if (markerSetId) {
            additionalProps['requested_marker_set'] = markerSetId;
        }
        const annotationParams = query_helper_1.QueryHelper.selectionTimeQuery(query_helper_1.QueryHelper.splitRangeIntoEqualParts(start, end, nbTimes), ids, additionalProps);
        const annotations = new Map();
        const tspClientResponse2 = await this.client.fetchAnnotations(this.traceUUID, this.outputId, annotationParams);
        const annotationsResponse = tspClientResponse2.getModel();
        const rangeEvents = [];
        if (tspClientResponse2.isOk() && annotationsResponse) {
            Object.entries(annotationsResponse.model.annotations).forEach(([category, categoryArray]) => {
                categoryArray.forEach(annotation => {
                    if (annotation.type === annotation_1.Type.CHART) {
                        if (annotation.entryId === -1) {
                            rangeEvents.push(this.getAnnotation(category, annotation, rangeEvents.length, chartStart));
                        }
                        else {
                            let entryArray = annotations.get(annotation.entryId);
                            if (entryArray === undefined) {
                                entryArray = [];
                                annotations.set(annotation.entryId, entryArray);
                            }
                            entryArray.push(this.getAnnotation(category, annotation, entryArray.length, chartStart));
                        }
                    }
                });
            });
        }
        for (const [entryId, entryArray] of annotations.entries()) {
            const row = rows.find(tgEntry => tgEntry.id === entryId);
            if (row) {
                row.annotations = entryArray;
            }
        }
        const arrows = await this.getArrows(ids, viewRange, nbTimes);
        return {
            id: 'model',
            totalLength: this.totalRange,
            rows,
            arrows,
            rangeEvents,
            data: {
                originalStart: chartStart
            }
        };
    }
    async getArrows(ids, viewRange, nbTimes) {
        let timeGraphArrows = [];
        if (viewRange && nbTimes) {
            const start = viewRange.start + this.timeGraphEntries[0].start;
            const end = viewRange.end + this.timeGraphEntries[0].start;
            const fetchParameters = query_helper_1.QueryHelper.selectionTimeQuery(query_helper_1.QueryHelper.splitRangeIntoEqualParts(start, end, nbTimes), ids);
            const tspClientResponseArrows = await this.client.fetchTimeGraphArrows(this.traceUUID, this.outputId, fetchParameters);
            const stateResponseArrows = tspClientResponseArrows.getModel();
            if (tspClientResponseArrows.isOk() && stateResponseArrows && stateResponseArrows.model) {
                timeGraphArrows = stateResponseArrows.model;
            }
        }
        const offset = this.timeGraphEntries[0].start;
        timeGraphArrows = timeGraphArrows.filter(arrow => ids.find(id => id === arrow.sourceId) && ids.find(id => id === arrow.targetId));
        const arrows = timeGraphArrows.map(arrow => ({
            sourceId: ids.indexOf(arrow.sourceId),
            destinationId: ids.indexOf(arrow.targetId),
            range: {
                start: arrow.start - offset,
                end: arrow.end - offset
            }
        }));
        return arrows;
    }
    timeGraphRowsOrdering(orderedIds) {
        const newTimeGraphRows = [];
        orderedIds.forEach(id => {
            const timeGraphRow = this.timeGraphRows.find(row => row.entryId === id);
            if (timeGraphRow) {
                newTimeGraphRows.push(timeGraphRow);
            }
            else {
                const emptyRow = { states: [{ start: BigInt(0), end: BigInt(0), label: '', tags: 0 }], entryId: id };
                newTimeGraphRows.push(emptyRow);
            }
        });
        this.timeGraphRows = newTimeGraphRows;
    }
    getDefaultForGapStyle() {
        // Default color and height for the GAP state
        return {
            parentKey: '',
            values: {
                'background-color': '#CACACA',
                height: 1.0
            }
        };
    }
    getRowModel(row, chartStart, rowId, entry) {
        let gapStyle;
        if (!entry.style) {
            gapStyle = this.getDefaultForGapStyle();
        }
        else {
            gapStyle = entry.style;
        }
        const states = [];
        let prevPossibleState = entry.start;
        let nextPossibleState = entry.end;
        row.states.forEach((state, idx) => {
            const end = state.end - chartStart;
            states.push({
                id: row.entryId + '-' + idx,
                label: state.label,
                range: {
                    start: state.start - chartStart,
                    end
                },
                data: {
                    style: state.style
                }
            });
            this.totalRange = this.totalRange < end ? end : this.totalRange;
            if (idx === 0) {
                prevPossibleState = state.start - chartStart;
            }
            if (idx === row.states.length - 1) {
                nextPossibleState = state.end - chartStart;
            }
        });
        return {
            id: rowId,
            name: entry.labels[0],
            range: {
                start: entry.start - chartStart,
                end: entry.end - chartStart
            },
            states,
            annotations: [],
            prevPossibleState,
            nextPossibleState,
            gapStyle
        };
    }
    getAnnotation(category, annotation, idx, chartStart) {
        return {
            id: annotation.entryId + '-' + idx,
            category: category,
            range: {
                start: annotation.time - chartStart,
                end: annotation.time + annotation.duration - chartStart
            },
            label: annotation.label,
            data: {
                style: annotation.style
            },
        };
    }
    async fetchStateTooltip(element, viewRange) {
        var _a;
        const elementRange = element.model.range;
        const offset = viewRange.getOffset() ? viewRange.getOffset() : BigInt(0);
        // use middle of state for fetching tooltip since hover time is not available
        const time = elementRange.start + (elementRange.end - elementRange.start) / BigInt(2) + (offset ? offset : BigInt(0));
        const requestedElement = {
            elementType: ElementType.STATE,
            time: element.model.range.start + (offset ? offset : BigInt(0)),
            duration: element.model.range.end - element.model.range.start
        };
        const entryId = [element.row.model.id];
        const parameters = query_helper_1.QueryHelper.selectionTimeQuery([time], entryId, { [query_helper_1.QueryHelper.REQUESTED_ELEMENT_KEY]: requestedElement });
        const tooltipResponse = await this.client.fetchTimeGraphTooltip(this.traceUUID, this.outputId, parameters);
        return (_a = tooltipResponse.getModel()) === null || _a === void 0 ? void 0 : _a.model;
    }
    async fetchAnnotationTooltip(element, viewRange) {
        var _a;
        const elementRange = element.model.range;
        const offset = viewRange.getOffset() ? viewRange.getOffset() : BigInt(0);
        const time = elementRange.start + (offset ? offset : BigInt(0));
        const requestedElement = {
            elementType: ElementType.ANNOTATION,
            time: element.model.range.start + (offset ? offset : BigInt(0)),
            duration: element.model.range.end - element.model.range.start,
            entryId: element.row.model.id
        };
        const entryId = [element.row.model.id];
        const parameters = query_helper_1.QueryHelper.selectionTimeQuery([time], entryId, { [query_helper_1.QueryHelper.REQUESTED_ELEMENT_KEY]: requestedElement });
        const tooltipResponse = await this.client.fetchTimeGraphTooltip(this.traceUUID, this.outputId, parameters);
        return (_a = tooltipResponse.getModel()) === null || _a === void 0 ? void 0 : _a.model;
    }
}
exports.TspDataProvider = TspDataProvider;
//# sourceMappingURL=tsp-data-provider.js.map