"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __asyncValues = (this && this.__asyncValues) || function (o) {
    if (!Symbol.asyncIterator) throw new TypeError("Symbol.asyncIterator is not defined.");
    var m = o[Symbol.asyncIterator], i;
    return m ? m.call(o) : (o = typeof __values === "function" ? __values(o) : o[Symbol.iterator](), i = {}, verb("next"), verb("throw"), verb("return"), i[Symbol.asyncIterator] = function () { return this; }, i);
    function verb(n) { i[n] = o[n] && function (v) { return new Promise(function (resolve, reject) { v = o[n](v), settle(resolve, reject, v.done, v.value); }); }; }
    function settle(resolve, reject, d, v) { Promise.resolve(v).then(function(v) { resolve({ value: v, done: d }); }, reject); }
};
Object.defineProperty(exports, "__esModule", { value: true });
// Automatically install clangd binary releases from GitHub.
//
// We don't bundle them with the package because they're big; we'd have to
// include all OS versions, and download them again with every extension update.
//
// There are several entry points:
//  - installation explicitly requested
//  - checking for updates (manual or automatic)
//  - no usable clangd found, try to recover
// These have different flows, but the same underlying mechanisms.
const abort_controller_1 = require("abort-controller");
const child_process = require("child_process");
const fs = require("fs");
const node_fetch_1 = require("node-fetch");
const os = require("os");
const path = require("path");
const readdirp = require("readdirp");
const rimraf = require("rimraf");
const semver = require("semver");
const stream = require("stream");
const unzipper = require("unzipper");
const util_1 = require("util");
const which = require("which");
// Main startup workflow: check whether the configured clangd binary us usable.
// If not, offer to install one. If so, check for updates.
function prepare(ui, checkUpdate) {
    return __awaiter(this, void 0, void 0, function* () {
        try {
            var absPath = yield util_1.promisify(which)(ui.clangdPath);
        }
        catch (e) {
            // Couldn't find clangd - start recovery flow and stop extension
            // loading.
            return { clangdPath: null, background: recover(ui) };
        }
        // Allow extension to load, asynchronously check for updates.
        return {
            clangdPath: absPath,
            background: checkUpdate ? checkUpdates(/*requested=*/ false, ui)
                : Promise.resolve()
        };
    });
}
exports.prepare = prepare;
// The user has explicitly asked to install the latest clangd.
// Do so without further prompting, or report an error.
function installLatest(ui) {
    return __awaiter(this, void 0, void 0, function* () {
        const abort = new abort_controller_1.AbortController();
        try {
            const release = yield Github.latestRelease();
            const asset = yield Github.chooseAsset(release);
            ui.clangdPath = yield Install.install(release, asset, abort, ui);
            ui.promptReload(`clangd ${release.name} is now installed.`);
        }
        catch (e) {
            if (!abort.signal.aborted) {
                console.error('Failed to install clangd: ', e);
                const message = `Failed to install clangd language server: ${e}\n` +
                    'You may want to install it manually.';
                ui.showHelp(message, installURL);
            }
        }
    });
}
exports.installLatest = installLatest;
// We have an apparently-valid clangd (`clangdPath`), check for updates.
function checkUpdates(requested, ui) {
    return __awaiter(this, void 0, void 0, function* () {
        // Gather all the version information to see if there's an upgrade.
        try {
            var release = yield Github.latestRelease();
            yield Github.chooseAsset(release); // Ensure a binary for this platform.
            var upgrade = yield Version.upgrade(release, ui.clangdPath);
        }
        catch (e) {
            console.log('Failed to check for clangd update: ', e);
            // We're not sure whether there's an upgrade: stay quiet unless asked.
            if (requested)
                ui.error(`Failed to check for clangd update: ${e}`);
            return;
        }
        console.log('Checking for clangd update: available=', upgrade.new, ' installed=', upgrade.old);
        // Bail out if the new version is better or comparable.
        if (!upgrade.upgrade) {
            if (requested)
                ui.info(`clangd is up-to-date (you have ${upgrade.old}, latest is ${upgrade.new})`);
            return;
        }
        ui.promptUpdate(upgrade.old, upgrade.new);
    });
}
exports.checkUpdates = checkUpdates;
// The extension has detected clangd isn't available.
// Inform the user, and if possible offer to install or adjust the path.
// Unlike installLatest(), we've had no explicit user request or consent yet.
function recover(ui) {
    return __awaiter(this, void 0, void 0, function* () {
        try {
            const release = yield Github.latestRelease();
            yield Github.chooseAsset(release); // Ensure a binary for this platform.
            ui.promptInstall(release.name);
        }
        catch (e) {
            console.error('Auto-install failed: ', e);
            ui.showHelp('The clangd language server is not installed.', installURL);
        }
    });
}
const installURL = 'https://clangd.llvm.org/installation.html';
// The GitHub API endpoint for the latest binary clangd release.
let githubReleaseURL = 'https://api.github.com/repos/clangd/clangd/releases/latest';
// Set a fake URL for testing.
function fakeGitHubReleaseURL(u) { githubReleaseURL = u; }
exports.fakeGitHubReleaseURL = fakeGitHubReleaseURL;
let lddCommand = 'ldd';
function fakeLddCommand(l) { lddCommand = l; }
exports.fakeLddCommand = fakeLddCommand;
// Bits for talking to github's release API
var Github;
(function (Github) {
    // Fetch the metadata for the latest stable clangd release.
    function latestRelease() {
        return __awaiter(this, void 0, void 0, function* () {
            const response = yield node_fetch_1.default(githubReleaseURL);
            if (!response.ok) {
                console.log(response.url, response.status, response.statusText);
                throw new Error(`Can't fetch release: ${response.statusText}`);
            }
            return yield response.json();
        });
    }
    Github.latestRelease = latestRelease;
    // Determine which release asset should be installed for this machine.
    function chooseAsset(release) {
        return __awaiter(this, void 0, void 0, function* () {
            const variants = {
                'win32': 'windows',
                'linux': 'linux',
                'darwin': 'mac',
            };
            const variant = variants[os.platform()];
            if (variant == 'linux') {
                // Hardcoding this here is sad, but we'd like to offer a nice error message
                // without making the user download the package first.
                const minGlibc = new semver.Range('2.18');
                const oldGlibc = yield Version.oldGlibc(minGlibc);
                if (oldGlibc) {
                    throw new Error('The clangd release is not compatible with your system ' +
                        `(glibc ${oldGlibc.raw} < ${minGlibc.raw}). ` +
                        'Try to install it using your package manager instead.');
                }
            }
            // 32-bit vscode is still common on 64-bit windows, so don't reject that.
            if (variant && (os.arch() == 'x64' || variant == 'windows' ||
                // Mac distribution contains a fat binary working on both x64
                // and arm64s.
                (os.arch() == 'arm64' && variant == 'mac'))) {
                const asset = release.assets.find(a => a.name.indexOf(variant) >= 0);
                if (asset)
                    return asset;
            }
            throw new Error(`No clangd ${release.name} binary available for ${os.platform()}/${os.arch()}`);
        });
    }
    Github.chooseAsset = chooseAsset;
})(Github || (Github = {}));
// Functions to download and install the releases, and manage the files on disk.
//
// File layout:
//  <ui.storagePath>/
//    install/
//      <version>/
//        clangd_<version>/            (outer director from zip file)
//          bin/clangd
//          lib/clang/...
//    download/
//      clangd-platform-<version>.zip  (deleted after extraction)
var Install;
(function (Install) {
    // Download the binary archive `asset` from a github `release` and extract it
    // to the extension's global storage location.
    // The `abort` controller is signaled if the user cancels the installation.
    // Returns the absolute path to the installed clangd executable.
    function install(release, asset, abort, ui) {
        return __awaiter(this, void 0, void 0, function* () {
            const dirs = yield createDirs(ui);
            const extractRoot = path.join(dirs.install, release.tag_name);
            if (yield util_1.promisify(fs.exists)(extractRoot)) {
                const reuse = yield ui.shouldReuse(release.name);
                if (reuse === undefined) {
                    // User dismissed prompt, bail out.
                    abort.abort();
                    throw new Error(`clangd ${release.name} already installed!`);
                }
                if (reuse) {
                    // Find clangd within the existing directory.
                    let files = (yield readdirp.promise(extractRoot)).map(e => e.fullPath);
                    return findExecutable(files);
                }
                else {
                    // Delete the old version.
                    yield util_1.promisify(rimraf)(extractRoot);
                    // continue with installation.
                }
            }
            const zipFile = path.join(dirs.download, asset.name);
            yield download(asset.browser_download_url, zipFile, abort, ui);
            const archive = yield unzipper.Open.file(zipFile);
            const executable = findExecutable(archive.files.map(f => f.path));
            yield ui.slow(`Extracting ${asset.name}`, archive.extract({ path: extractRoot }));
            const clangdPath = path.join(extractRoot, executable);
            yield fs.promises.chmod(clangdPath, 0o755);
            yield fs.promises.unlink(zipFile);
            return clangdPath;
        });
    }
    Install.install = install;
    // Create the 'install' and 'download' directories, and return absolute paths.
    function createDirs(ui) {
        return __awaiter(this, void 0, void 0, function* () {
            const install = path.join(ui.storagePath, 'install');
            const download = path.join(ui.storagePath, 'download');
            for (const dir of [install, download])
                yield fs.promises.mkdir(dir, { 'recursive': true });
            return { install: install, download: download };
        });
    }
    // Find the clangd executable within a set of files.
    function findExecutable(paths) {
        const filename = os.platform() == 'win32' ? 'clangd.exe' : 'clangd';
        const entry = paths.find(f => path.posix.basename(f) == filename ||
            path.win32.basename(f) == filename);
        if (entry == null)
            throw new Error('Didn\'t find a clangd executable!');
        return entry;
    }
    // Downloads `url` to a local file `dest` (whose parent should exist).
    // A progress dialog is shown, if it is cancelled then `abort` is signaled.
    function download(url, dest, abort, ui) {
        return __awaiter(this, void 0, void 0, function* () {
            console.log('Downloading ', url, ' to ', dest);
            return ui.progress(`Downloading ${path.basename(dest)}`, abort, (progress) => __awaiter(this, void 0, void 0, function* () {
                const response = yield node_fetch_1.default(url, { signal: abort.signal });
                if (!response.ok)
                    throw new Error(`Failed to download $url`);
                const size = Number(response.headers.get('content-length'));
                let read = 0;
                response.body.on('data', (chunk) => {
                    read += chunk.length;
                    progress(read / size);
                });
                const out = fs.createWriteStream(dest);
                yield util_1.promisify(stream.pipeline)(response.body, out).catch(e => {
                    // Clean up the partial file if the download failed.
                    fs.unlink(dest, (_) => null); // Don't wait, and ignore error.
                    throw e;
                });
            }));
        });
    }
})(Install || (Install = {}));
// Functions dealing with clangd versions.
//
// We parse both github release numbers and installed `clangd --version` output
// by treating them as SemVer ranges, and offer an upgrade if the version
// is unambiguously newer.
//
// These functions throw if versions can't be parsed (e.g. installed clangd
// is a vendor-modified version).
var Version;
(function (Version) {
    function upgrade(release, clangdPath) {
        return __awaiter(this, void 0, void 0, function* () {
            const releasedVer = released(release);
            const installedVer = yield installed(clangdPath);
            return {
                old: installedVer.raw,
                new: releasedVer.raw,
                upgrade: rangeGreater(releasedVer, installedVer)
            };
        });
    }
    Version.upgrade = upgrade;
    const loose = {
        'loose': true
    };
    // Get the version of an installed clangd binary using `clangd --version`.
    function installed(clangdPath) {
        return __awaiter(this, void 0, void 0, function* () {
            const output = yield run(clangdPath, ['--version']);
            console.log(clangdPath, ' --version output: ', output);
            const prefix = 'clangd version ';
            if (!output.startsWith(prefix))
                throw new Error(`Couldn't parse clangd --version output: ${output}`);
            const rawVersion = output.substr(prefix.length).split(' ', 1)[0];
            return new semver.Range(rawVersion, loose);
        });
    }
    // Get the version of a github release, by parsing the tag or name.
    function released(release) {
        // Prefer the tag name, but fall back to the release name.
        return (!semver.validRange(release.tag_name, loose) &&
            semver.validRange(release.name, loose))
            ? new semver.Range(release.name, loose)
            : new semver.Range(release.tag_name, loose);
    }
    // Detect the (linux) system's glibc version. If older than `min`, return it.
    function oldGlibc(min) {
        return __awaiter(this, void 0, void 0, function* () {
            // ldd is distributed with glibc, so ldd --version should be a good proxy.
            const output = yield run(lddCommand, ['--version']);
            // The first line is e.g. "ldd (Debian GLIBC 2.29-9) 2.29".
            const line = output.split('\n', 1)[0];
            // Require some confirmation this is [e]glibc, and a plausible
            // version number.
            const match = line.match(/^ldd .*glibc.* (\d+(?:\.\d+)+)[^ ]*$/i);
            if (!match || !semver.validRange(match[1], loose)) {
                console.error(`Can't glibc version from ldd --version output: ${line}`);
                return null;
            }
            const version = new semver.Range(match[1], loose);
            console.log('glibc is', version.raw, 'min is', min.raw);
            return rangeGreater(min, version) ? version : null;
        });
    }
    Version.oldGlibc = oldGlibc;
    // Run a system command and capture any stdout produced.
    function run(command, flags) {
        var e_1, _a;
        return __awaiter(this, void 0, void 0, function* () {
            const child = child_process.spawn(command, flags, { stdio: ['ignore', 'pipe', 'ignore'] });
            let output = '';
            try {
                for (var _b = __asyncValues(child.stdout), _c; _c = yield _b.next(), !_c.done;) {
                    const chunk = _c.value;
                    output += chunk;
                }
            }
            catch (e_1_1) { e_1 = { error: e_1_1 }; }
            finally {
                try {
                    if (_c && !_c.done && (_a = _b.return)) yield _a.call(_b);
                }
                finally { if (e_1) throw e_1.error; }
            }
            return output;
        });
    }
    function rangeGreater(newVer, oldVer) {
        return semver.gtr(semver.minVersion(newVer), oldVer);
    }
})(Version || (Version = {}));
//# sourceMappingURL=index.js.map