"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.RestClient = void 0;
const node_fetch_1 = require("node-fetch");
const tsp_client_response_1 = require("./tsp-client-response");
const JSONBigConfig = require("json-bigint");
const JSONBig = JSONBigConfig({
    useNativeBigInt: true,
});
/**
 * Rest client helper to make request.
 * The request response status code indicates if the request is successful.
 * The json object in the response may be undefined when an error occurs.
 */
class RestClient {
    /**
     * Perform GET
     * @template T is the expected type of the json object returned by this request
     * @param url URL to query without query parameters
     * @param parameters Query parameters. Mapped keys and values are used to build the final URL
     */
    static get(url, parameters, normalizer) {
        return __awaiter(this, void 0, void 0, function* () {
            let getUrl = url;
            if (parameters) {
                const urlParameters = this.encodeURLParameters(parameters);
                getUrl = getUrl.concat(urlParameters);
            }
            return this.performRequest('get', getUrl, undefined, normalizer);
        });
    }
    /**
     * Perform POST
     * @template T is the expected type of the json object returned by this request
     * @param url URL to query
     * @param body Query object as defined by the Query interface
     */
    static post(url, body, normalizer) {
        return __awaiter(this, void 0, void 0, function* () {
            return this.performRequest('post', url, body, normalizer);
        });
    }
    /**
     * Perform PUT
     * @template T is the expected type of the json object returned by this request
     * @param url URL to query
     * @param body Query object as defined by the Query interface
     */
    static put(url, body, normalizer) {
        return __awaiter(this, void 0, void 0, function* () {
            return this.performRequest('put', url, body, normalizer);
        });
    }
    /**
     * Perform DELETE
     * @template T is the expected type of the json object returned by this request
     * @param url URL to query without query parameters
     * @param parameters Query parameters. Mapped keys and values are used to build the final URL
     */
    static delete(url, parameters, normalizer) {
        return __awaiter(this, void 0, void 0, function* () {
            let deleteUrl = url;
            if (parameters) {
                const urlParameters = this.encodeURLParameters(parameters);
                deleteUrl = deleteUrl.concat(urlParameters);
            }
            return this.performRequest('delete', deleteUrl, undefined, normalizer);
        });
    }
    static performRequest(method, url, body, normalizer) {
        var _a;
        return __awaiter(this, void 0, void 0, function* () {
            let response;
            try {
                response = yield this.httpRequest({
                    url,
                    method,
                    headers: {
                        'Accept': 'application/json',
                        'Content-Type': 'application/json'
                    },
                    body: this.jsonStringify(body)
                });
                const status = response.status <= 501;
                this.updateConnectionStatus(status);
            }
            catch (err) {
                this.updateConnectionStatus(false);
                return new tsp_client_response_1.TspClientResponse(err.toString(), 503, 'Service Unavailable');
            }
            if (((_a = response.headers) === null || _a === void 0 ? void 0 : _a.get('Content-Type')) === 'application/json') {
                try {
                    const parsed = this.jsonParse(response.text);
                    try {
                        const responseModel = normalizer ? normalizer(parsed) : parsed;
                        return new tsp_client_response_1.TspClientResponse(response.text, response.status, response.statusText, responseModel);
                    }
                    catch (err) {
                        console.log('Error normalizing model: ' + err.toString());
                    }
                }
                catch (err) {
                    console.log('Error parsing response model: ' + JSON.stringify(err));
                }
            }
            return new tsp_client_response_1.TspClientResponse(response.text, response.status, response.statusText);
        });
    }
    static httpRequest(req) {
        return __awaiter(this, void 0, void 0, function* () {
            const { url, method, body, headers } = req;
            const response = yield (0, node_fetch_1.default)(url, { method, headers, body });
            const text = yield response.text();
            return {
                text,
                status: response.status,
                statusText: response.statusText,
                headers: response.headers
            };
        });
    }
    static encodeURLParameters(parameters) {
        if (parameters.size) {
            return '?' + Array.from(parameters, ([key, value]) => `${encodeURIComponent(key)}=${encodeURIComponent(value)}`).join('&');
        }
        return '';
    }
    /**
     * Stringify JS objects. Can stringify `BigInt` values.
     */
    static jsonStringify(data) {
        return JSONBig.stringify(data);
    }
    /**
     * Parse JSON-encoded data. If a number is too large to fit into a regular
     * `number` then it will be deserialized as `BigInt`.
     */
    static jsonParse(text) {
        return JSONBig.parse(text);
    }
    /**
     * Adds a connection status listener.
     * The listener will immediately be called with the current status.
     *
     * @param listener The listener to add
     */
    static addConnectionStatusListener(listener) {
        if (!this.connectionStatusListeners.includes(listener)) {
            this.connectionStatusListeners.push(listener);
        }
        listener(this.status);
    }
    /**
     * Removes a connection status listener.
     *
     * @param listener The listener to remove
     */
    static removeConnectionStatusListener(listener) {
        this.connectionStatusListeners = this.connectionStatusListeners.filter(element => element !== listener);
    }
    static updateConnectionStatus(status) {
        if (this.status === status) {
            return;
        }
        for (const listener of this.connectionStatusListeners) {
            listener(status);
        }
        this.status = status;
    }
}
exports.RestClient = RestClient;
RestClient.status = false;
RestClient.connectionStatusListeners = [];
//# sourceMappingURL=rest-client.js.map