"use strict";
/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
Object.defineProperty(exports, "__esModule", { value: true });
exports.ObservableValue = exports.observableValue = exports.TransactionImpl = exports.getFunctionName = exports.transaction = exports.BaseObservable = exports.ConvenientObservable = exports._setDerived = void 0;
const logging_1 = require("./logging");
let _derived;
/**
 * @internal
 * This is to allow splitting files.
*/
function _setDerived(derived) {
    _derived = derived;
}
exports._setDerived = _setDerived;
class ConvenientObservable {
    get TChange() { return null; }
    /** @sealed */
    read(reader) {
        reader.subscribeTo(this);
        return this.get();
    }
    /** @sealed */
    map(fn) {
        return _derived(() => {
            const name = getFunctionName(fn);
            return name !== undefined ? name : `${this.debugName} (mapped)`;
        }, (reader) => fn(this.read(reader)));
    }
}
exports.ConvenientObservable = ConvenientObservable;
class BaseObservable extends ConvenientObservable {
    constructor() {
        super(...arguments);
        this.observers = new Set();
    }
    /** @sealed */
    addObserver(observer) {
        const len = this.observers.size;
        this.observers.add(observer);
        if (len === 0) {
            this.onFirstObserverAdded();
        }
    }
    /** @sealed */
    removeObserver(observer) {
        const deleted = this.observers.delete(observer);
        if (deleted && this.observers.size === 0) {
            this.onLastObserverRemoved();
        }
    }
    onFirstObserverAdded() { }
    onLastObserverRemoved() { }
}
exports.BaseObservable = BaseObservable;
function transaction(fn, getDebugName) {
    var _a, _b;
    const tx = new TransactionImpl(fn, getDebugName);
    try {
        (_a = (0, logging_1.getLogger)()) === null || _a === void 0 ? void 0 : _a.handleBeginTransaction(tx);
        fn(tx);
    }
    finally {
        tx.finish();
        (_b = (0, logging_1.getLogger)()) === null || _b === void 0 ? void 0 : _b.handleEndTransaction();
    }
}
exports.transaction = transaction;
function getFunctionName(fn) {
    const fnSrc = fn.toString();
    // Pattern: /** @description ... */
    const regexp = /\/\*\*\s*@description\s*([^*]*)\*\//;
    const match = regexp.exec(fnSrc);
    const result = match ? match[1] : undefined;
    return result === null || result === void 0 ? void 0 : result.trim();
}
exports.getFunctionName = getFunctionName;
class TransactionImpl {
    constructor(fn, _getDebugName) {
        this.fn = fn;
        this._getDebugName = _getDebugName;
        this.updatingObservers = [];
    }
    getDebugName() {
        if (this._getDebugName) {
            return this._getDebugName();
        }
        return getFunctionName(this.fn);
    }
    updateObserver(observer, observable) {
        this.updatingObservers.push({ observer, observable });
        observer.beginUpdate(observable);
    }
    finish() {
        const updatingObservers = this.updatingObservers;
        // Prevent anyone from updating observers from now on.
        this.updatingObservers = null;
        for (const { observer, observable } of updatingObservers) {
            observer.endUpdate(observable);
        }
    }
}
exports.TransactionImpl = TransactionImpl;
function observableValue(name, initialValue) {
    return new ObservableValue(name, initialValue);
}
exports.observableValue = observableValue;
class ObservableValue extends BaseObservable {
    constructor(debugName, initialValue) {
        super();
        this.debugName = debugName;
        this.value = initialValue;
    }
    get() {
        return this.value;
    }
    set(value, tx, change) {
        var _a;
        if (this.value === value) {
            return;
        }
        if (!tx) {
            transaction((tx) => {
                this.set(value, tx, change);
            }, () => `Setting ${this.debugName}`);
            return;
        }
        const oldValue = this.value;
        this.value = value;
        (_a = (0, logging_1.getLogger)()) === null || _a === void 0 ? void 0 : _a.handleObservableChanged(this, { oldValue, newValue: value, change, didChange: true });
        for (const observer of this.observers) {
            tx.updateObserver(observer, this);
            observer.handleChange(this, change);
        }
    }
    toString() {
        return `${this.debugName}: ${this.value}`;
    }
}
exports.ObservableValue = ObservableValue;
//# sourceMappingURL=base.js.map