"use strict";
/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
Object.defineProperty(exports, "__esModule", { value: true });
exports.QuickInputBox = void 0;
const dom = require("../../../browser/dom");
const keyboardEvent_1 = require("../../../browser/keyboardEvent");
const mouseEvent_1 = require("../../../browser/mouseEvent");
const findInput_1 = require("../../../browser/ui/findinput/findInput");
const lifecycle_1 = require("../../../common/lifecycle");
const severity_1 = require("../../../common/severity");
require("./media/quickInput.css");
const $ = dom.$;
class QuickInputBox extends lifecycle_1.Disposable {
    constructor(parent) {
        super();
        this.parent = parent;
        this.onKeyDown = (handler) => {
            return dom.addDisposableListener(this.findInput.inputBox.inputElement, dom.EventType.KEY_DOWN, (e) => {
                handler(new keyboardEvent_1.StandardKeyboardEvent(e));
            });
        };
        this.onMouseDown = (handler) => {
            return dom.addDisposableListener(this.findInput.inputBox.inputElement, dom.EventType.MOUSE_DOWN, (e) => {
                handler(new mouseEvent_1.StandardMouseEvent(e));
            });
        };
        this.onDidChange = (handler) => {
            return this.findInput.onDidChange(handler);
        };
        this.container = dom.append(this.parent, $('.quick-input-box'));
        this.findInput = this._register(new findInput_1.FindInput(this.container, undefined, { label: '' }));
    }
    get value() {
        return this.findInput.getValue();
    }
    set value(value) {
        this.findInput.setValue(value);
    }
    select(range = null) {
        this.findInput.inputBox.select(range);
    }
    isSelectionAtEnd() {
        return this.findInput.inputBox.isSelectionAtEnd();
    }
    setPlaceholder(placeholder) {
        this.findInput.inputBox.setPlaceHolder(placeholder);
    }
    get placeholder() {
        return this.findInput.inputBox.inputElement.getAttribute('placeholder') || '';
    }
    set placeholder(placeholder) {
        this.findInput.inputBox.setPlaceHolder(placeholder);
    }
    get ariaLabel() {
        return this.findInput.inputBox.getAriaLabel();
    }
    set ariaLabel(ariaLabel) {
        this.findInput.inputBox.setAriaLabel(ariaLabel);
    }
    get password() {
        return this.findInput.inputBox.inputElement.type === 'password';
    }
    set password(password) {
        this.findInput.inputBox.inputElement.type = password ? 'password' : 'text';
    }
    set enabled(enabled) {
        this.findInput.setEnabled(enabled);
    }
    set toggles(toggles) {
        this.findInput.setAdditionalToggles(toggles);
    }
    hasFocus() {
        return this.findInput.inputBox.hasFocus();
    }
    setAttribute(name, value) {
        this.findInput.inputBox.inputElement.setAttribute(name, value);
    }
    removeAttribute(name) {
        this.findInput.inputBox.inputElement.removeAttribute(name);
    }
    showDecoration(decoration) {
        if (decoration === severity_1.default.Ignore) {
            this.findInput.clearMessage();
        }
        else {
            this.findInput.showMessage({ type: decoration === severity_1.default.Info ? 1 /* MessageType.INFO */ : decoration === severity_1.default.Warning ? 2 /* MessageType.WARNING */ : 3 /* MessageType.ERROR */, content: '' });
        }
    }
    stylesForType(decoration) {
        return this.findInput.inputBox.stylesForType(decoration === severity_1.default.Info ? 1 /* MessageType.INFO */ : decoration === severity_1.default.Warning ? 2 /* MessageType.WARNING */ : 3 /* MessageType.ERROR */);
    }
    setFocus() {
        this.findInput.focus();
    }
    layout() {
        this.findInput.inputBox.layout();
    }
    style(styles) {
        this.findInput.style(styles);
    }
}
exports.QuickInputBox = QuickInputBox;
//# sourceMappingURL=quickInputBox.js.map