"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConnectionImpl = exports.PluginChannel = void 0;
const event_1 = require("@theia/core/lib/common/event");
/**
 * A channel communicating with a counterpart in a plugin host.
 */
class PluginChannel {
    constructor(id, connection) {
        this.id = id;
        this.connection = connection;
        this.messageEmitter = new event_1.Emitter();
        this.errorEmitter = new event_1.Emitter();
        this.closedEmitter = new event_1.Emitter();
    }
    send(content) {
        this.connection.$sendMessage(this.id, content);
    }
    fireMessageReceived(msg) {
        this.messageEmitter.fire(msg);
    }
    fireError(error) {
        this.errorEmitter.fire(error);
    }
    fireClosed() {
        this.closedEmitter.fire();
    }
    onMessage(cb) {
        this.messageEmitter.event(cb);
    }
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    onError(cb) {
        this.errorEmitter.event(cb);
    }
    onClose(cb) {
        this.closedEmitter.event(() => cb(-1, 'closed'));
    }
    close() {
        this.connection.$deleteConnection(this.id);
    }
}
exports.PluginChannel = PluginChannel;
class ConnectionImpl {
    constructor(proxy) {
        this.connections = new Map();
        this.proxy = proxy;
    }
    /**
     * Gets the connection between plugin by id and sends string message to it.
     *
     * @param id connection's id
     * @param message incoming message
     */
    async $sendMessage(id, message) {
        if (this.connections.has(id)) {
            this.connections.get(id).fireMessageReceived(message);
        }
        else {
            console.warn(`Received message for unknown connection: ${id}`);
        }
    }
    /**
     * Instantiates a new connection by the given id.
     * @param id the connection id
     */
    async $createConnection(id) {
        console.debug(`Creating plugin connection: ${id}`);
        await this.doEnsureConnection(id);
    }
    /**
     * Deletes a connection.
     * @param id the connection id
     */
    async $deleteConnection(id) {
        console.debug(`Deleting plugin connection: ${id}`);
        const connection = this.connections.get(id);
        if (connection) {
            this.connections.delete(id);
            connection.fireClosed();
        }
    }
    /**
     * Returns existed connection or creates a new one.
     * @param id the connection id
     */
    async ensureConnection(id) {
        console.debug(`Creating local connection: ${id}`);
        const connection = await this.doEnsureConnection(id);
        await this.proxy.$createConnection(id);
        return connection;
    }
    /**
     * Returns existed connection or creates a new one.
     * @param id the connection id
     */
    async doEnsureConnection(id) {
        const connection = this.connections.get(id) || await this.doCreateConnection(id);
        this.connections.set(id, connection);
        return connection;
    }
    async doCreateConnection(id) {
        const channel = new PluginChannel(id, this.proxy);
        channel.onClose(() => this.connections.delete(id));
        return channel;
    }
}
exports.ConnectionImpl = ConnectionImpl;
//# sourceMappingURL=connection.js.map