"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TimeGraphUnitController = void 0;
var bigint_utils_1 = require("./bigint-utils");
var time_graph_render_controller_1 = require("./time-graph-render-controller");
var TimeGraphUnitController = /** @class */ (function () {
    function TimeGraphUnitController(absoluteRange, viewRange) {
        this.absoluteRange = absoluteRange;
        /**
         * This determines the world size.
         * worldRenderFactor = 1 renders one extra viewRange to the left and right,
         *      so there are three viewRanges rendered.
         */
        this._worldRenderFactor = 1;
        this._offset = BigInt(0);
        this._viewRange = viewRange || { start: BigInt(0), end: absoluteRange };
        this._worldRange = this._viewRange;
        this.viewRangeChangedHandlers = [];
        this.selectionRangeChangedHandlers = [];
        this.updateWorldRangeFromViewRange();
        this._renderer = new time_graph_render_controller_1.TimeGraphRenderController();
    }
    TimeGraphUnitController.prototype.handleViewRangeChange = function (oldRange) {
        var _this = this;
        this.viewRangeChangedHandlers.forEach(function (handler) { return handler(oldRange, _this._viewRange); });
    };
    TimeGraphUnitController.prototype.handleSelectionRangeChange = function () {
        var _this = this;
        this.selectionRangeChangedHandlers.forEach(function (handler) { return handler(_this._selectionRange); });
    };
    TimeGraphUnitController.prototype.onViewRangeChanged = function (handler) {
        this.viewRangeChangedHandlers.push(handler);
    };
    TimeGraphUnitController.prototype.removeViewRangeChangedHandler = function (handler) {
        var index = this.viewRangeChangedHandlers.indexOf(handler);
        if (index > -1) {
            this.viewRangeChangedHandlers.splice(index, 1);
        }
    };
    TimeGraphUnitController.prototype.updateWorldRangeFromViewRange = function () {
        var deltaV = this.viewRange.end - this.viewRange.start;
        var start = this.viewRange.start - bigint_utils_1.BIMath.multiply(deltaV, this.worldRenderFactor);
        var end = this.viewRange.end + bigint_utils_1.BIMath.multiply(deltaV, this.worldRenderFactor);
        return this.worldRange = { start: start, end: end };
    };
    TimeGraphUnitController.prototype.onSelectionRangeChange = function (handler) {
        this.selectionRangeChangedHandlers.push(handler);
    };
    TimeGraphUnitController.prototype.removeSelectionRangeChangedHandler = function (handler) {
        var index = this.selectionRangeChangedHandlers.indexOf(handler);
        if (index > -1) {
            this.selectionRangeChangedHandlers.splice(index, 1);
        }
    };
    Object.defineProperty(TimeGraphUnitController.prototype, "viewRange", {
        get: function () {
            return this._viewRange;
        },
        set: function (newRange) {
            // Making a deep copy
            var oldRange = {
                start: this._viewRange.start,
                end: this._viewRange.end
            };
            if (newRange.end > newRange.start) {
                this._viewRange = { start: newRange.start, end: newRange.end };
            }
            if (newRange.start < 0) {
                this._viewRange.start = BigInt(0);
            }
            if (this._viewRange.end > this.absoluteRange) {
                this._viewRange.end = this.absoluteRange;
            }
            this.handleViewRangeChange(oldRange);
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(TimeGraphUnitController.prototype, "worldRange", {
        get: function () {
            return this._worldRange;
        },
        set: function (newRange) {
            if (newRange.end > newRange.start) {
                this._worldRange = { start: newRange.start, end: newRange.end };
            }
            if (newRange.start < 0) {
                this._worldRange.start = BigInt(0);
            }
            if (this._worldRange.end > this.absoluteRange) {
                this._worldRange.end = this.absoluteRange;
            }
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(TimeGraphUnitController.prototype, "selectionRange", {
        get: function () {
            return this._selectionRange;
        },
        set: function (value) {
            this._selectionRange = value;
            this.handleSelectionRangeChange();
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(TimeGraphUnitController.prototype, "viewRangeLength", {
        get: function () {
            return this._viewRange.end - this._viewRange.start;
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(TimeGraphUnitController.prototype, "worldRangeLength", {
        get: function () {
            return this._worldRange.end - this._worldRange.start;
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(TimeGraphUnitController.prototype, "offset", {
        get: function () {
            return this._offset;
        },
        set: function (offset) {
            this._offset = offset;
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(TimeGraphUnitController.prototype, "worldRenderFactor", {
        get: function () {
            return this._worldRenderFactor;
        },
        set: function (n) {
            this._worldRenderFactor = n;
        },
        enumerable: false,
        configurable: true
    });
    return TimeGraphUnitController;
}());
exports.TimeGraphUnitController = TimeGraphUnitController;
//# sourceMappingURL=time-graph-unit-controller.js.map