"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const utils_1 = require("./utils");
const chai_1 = require("chai");
const path = require("path");
const assert_1 = require("assert");
const os = require("os");
describe('multithread', function () {
    return __awaiter(this, void 0, void 0, function* () {
        let dc;
        const program = path.join(utils_1.testProgramsDir, 'MultiThread');
        const source = path.join(utils_1.testProgramsDir, 'MultiThread.cc');
        const threadNames = {
            monday: 0,
            tuesday: 1,
            wednesday: 2,
            thursday: 3,
            friday: 4,
        };
        const lineTags = {
            LINE_MAIN_ALL_THREADS_STARTED: 0,
            LINE_THREAD_IN_HELLO: 0,
        };
        before(function () {
            (0, utils_1.resolveLineTagLocations)(source, lineTags);
        });
        beforeEach(() => __awaiter(this, void 0, void 0, function* () {
            dc = yield (0, utils_1.standardBeforeEach)();
        }));
        afterEach(() => __awaiter(this, void 0, void 0, function* () {
            yield dc.stop();
        }));
        it('sees all threads', function () {
            return __awaiter(this, void 0, void 0, function* () {
                if (!utils_1.gdbNonStop && os.platform() === 'win32' && utils_1.isRemoteTest) {
                    // The way thread names are set in remote tests on windows is unsupported
                    this.skip();
                }
                yield dc.hitBreakpoint((0, utils_1.fillDefaults)(this.test, {
                    program: program,
                }), {
                    path: source,
                    line: lineTags['LINE_MAIN_ALL_THREADS_STARTED'],
                });
                const threads = yield dc.threadsRequest();
                const nameToId = new Map(threads.body.threads.map((thread) => [thread.name, thread.id]));
                // Make sure all 5 threads are there
                (0, chai_1.expect)(nameToId).to.include.keys(Object.keys(threadNames));
                // and make sure that there is at least 6 threads.
                // We don't care about the name of the "main" thread
                (0, chai_1.expect)(threads.body.threads).length.greaterThanOrEqual(6);
                // check that each thread can be communicated with individually
                for (const [name, idInProgram] of Object.entries(threadNames)) {
                    // There are multiple ids/indexes.
                    // idInProgram cooresponds to the variable thread_id in the C++ source
                    // threadId is the id of the thread in DAP
                    const threadId = nameToId.get(name);
                    if (threadId === undefined) {
                        // unreachable because of expect above
                        (0, assert_1.fail)('unreachable');
                    }
                    if (utils_1.gdbNonStop) {
                        const waitForStopped = dc.waitForEvent('stopped');
                        const pr = dc.pauseRequest({ threadId });
                        yield Promise.all([pr, waitForStopped]);
                    }
                    const stack = yield dc.stackTraceRequest({ threadId });
                    let printHelloFrameId = undefined;
                    let callerFrameId = undefined;
                    for (const frame of stack.body.stackFrames) {
                        if (frame.name === 'PrintHello') {
                            printHelloFrameId = frame.id;
                        }
                        else if (printHelloFrameId !== undefined) {
                            callerFrameId = frame.id;
                            break;
                        }
                    }
                    if (printHelloFrameId === undefined) {
                        (0, assert_1.fail)("Failed to find frame with name 'PrintHello'");
                    }
                    if (callerFrameId === undefined) {
                        (0, assert_1.fail)("Failed to find frame that called 'PrintHello'");
                    }
                    {
                        const scopes = yield dc.scopesRequest({
                            frameId: callerFrameId,
                        });
                        const vr = scopes.body.scopes[0].variablesReference;
                        const vars = yield dc.variablesRequest({
                            variablesReference: vr,
                        });
                        const varnameToValue = new Map(vars.body.variables.map((variable) => [
                            variable.name,
                            variable.value,
                        ]));
                        // Make sure we aren't getting the HelloWorld frame's variables.
                        // The calling method (in glibc or similar) may end up with a local
                        // variable called thread_id, if so, update this heuristic
                        (0, chai_1.expect)(varnameToValue.get('thread_id')).to.be.undefined;
                    }
                    {
                        const scopes = yield dc.scopesRequest({
                            frameId: printHelloFrameId,
                        });
                        const vr = scopes.body.scopes[0].variablesReference;
                        const vars = yield dc.variablesRequest({
                            variablesReference: vr,
                        });
                        const varnameToValue = new Map(vars.body.variables.map((variable) => [
                            variable.name,
                            variable.value,
                        ]));
                        (0, chai_1.expect)(varnameToValue.get('thread_id')).to.equal(idInProgram.toString());
                        // The "name" variable is a pointer, so is displayed as an address + the
                        // extracted nul terminated string
                        (0, chai_1.expect)(varnameToValue.get('name')).to.contain(name);
                    }
                    {
                        // Make sure we can get variables for frame 0,
                        // the contents of those variables don't actually matter
                        // as the thread will probably be stopped in a library
                        // somewhere waiting for a semaphore
                        // This is a test for #235
                        const scopes = yield dc.scopesRequest({
                            frameId: stack.body.stackFrames[0].id,
                        });
                        const vr = scopes.body.scopes[0].variablesReference;
                        const vars = yield dc.variablesRequest({
                            variablesReference: vr,
                        });
                        const varnameToValue = new Map(vars.body.variables.map((variable) => [
                            variable.name,
                            variable.value,
                        ]));
                        // Make sure we aren't getting the HelloWorld frame's variables.
                        // The calling method (in glibc or similar) may end up with a local
                        // variable called thread_id, if so, update this heuristic
                        // We could be stopped PrintHello, so we don't perform the check
                        // if that is the case
                        if (stack.body.stackFrames[0].id !== printHelloFrameId) {
                            (0, chai_1.expect)(varnameToValue.get('thread_id')).to.be.undefined;
                        }
                    }
                }
            });
        });
        it('async resume for gdb-non-stop off', function () {
            return __awaiter(this, void 0, void 0, function* () {
                if (utils_1.gdbNonStop) {
                    // This test is covering only gdb-non-stop off mode
                    this.skip();
                }
                else if (os.platform() === 'win32' && (!utils_1.isRemoteTest || !utils_1.gdbAsync)) {
                    // Only supported in win32 host with remote + mi-async targets
                    this.skip();
                }
                yield dc.launchRequest((0, utils_1.fillDefaults)(this.test, {
                    program,
                }));
                yield dc.setBreakpointsRequest({
                    source: {
                        path: source,
                    },
                    breakpoints: [
                        {
                            line: lineTags['LINE_MAIN_ALL_THREADS_STARTED'],
                        },
                        {
                            line: lineTags['LINE_THREAD_IN_HELLO'],
                        },
                    ],
                });
                yield dc.configurationDoneRequest();
                yield dc.waitForEvent('stopped');
                const threads = yield dc.threadsRequest();
                // make sure that there is at least 2 threads.
                (0, chai_1.expect)(threads.body.threads).length.greaterThanOrEqual(2);
                // Send continue to thread 2
                dc.send('cdt-gdb-tests/executeCommand', {
                    command: '-exec-continue --thread 2',
                });
                const event = yield dc.waitForEvent('continued');
                // In allThreadsContinued:true case we are expecting id of the first thread no matter which thread is continued
                chai_1.assert.deepEqual(event.body, {
                    threadId: threads.body.threads[0].id,
                    allThreadsContinued: true,
                });
            });
        });
        it('async resume for gdb-non-stop on', function () {
            return __awaiter(this, void 0, void 0, function* () {
                if (!utils_1.gdbNonStop) {
                    // This test is covering only gdb-non-stop on
                    this.skip();
                }
                yield dc.hitBreakpoint((0, utils_1.fillDefaults)(this.test, {
                    program: program,
                }), {
                    path: source,
                    line: lineTags['LINE_MAIN_ALL_THREADS_STARTED'],
                });
                const threads = yield dc.threadsRequest();
                // make sure that there is at least 6 threads.
                (0, chai_1.expect)(threads.body.threads).length.greaterThanOrEqual(6);
                // stop the running threads
                const runningThreads = threads.body.threads.filter((t) => t.running);
                for (const thread of runningThreads) {
                    yield dc.pauseRequest({ threadId: thread.id });
                    yield dc.waitForEvent('stopped');
                }
                for (const thread of threads.body.threads) {
                    // Send an async continue request and wait for the continue event.
                    dc.send('cdt-gdb-tests/executeCommand', {
                        command: `-exec-continue --thread ${thread.id}`,
                    });
                    const event = yield dc.waitForEvent('continued');
                    chai_1.assert.deepEqual(event.body, {
                        threadId: thread.id,
                        allThreadsContinued: false,
                    });
                }
            });
        });
    });
});
//# sourceMappingURL=multithread.spec.js.map