/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2005 Imendio AB
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*
 *  lb-module-files.c
 */

#include <config.h>

#include <string.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <gtk/gtkicontheme.h>
#include <gconf/gconf-client.h>
#include <libgnomeui/gnome-icon-lookup.h>
#include <libgnomevfs/gnome-vfs.h>
#include <libgnomevfs/gnome-vfs-mime-handlers.h>
#include <libgnomevfs/gnome-vfs-file-info.h>

#include "lb-item.h"
#include "lb-module-files.h"
#include "lb-utils.h"
#include "lb-item-file.h"

#define GET_PRIV(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), LB_TYPE_MODULE_FILES, LbModuleFilesPriv))

typedef struct _LbModuleFilesPriv LbModuleFilesPriv;
struct _LbModuleFilesPriv {
	gchar *dir_uri;
};

enum {
	PROP_0,
	PROP_DIR_URI
};


static void    module_files_finalize        (GObject            *object);
static void    module_files_set_property    (GObject            *object,
					     guint               property_id,
					     const GValue       *value,
					     GParamSpec         *pspec);
static void    module_files_get_property    (GObject            *object,
					     guint               property_id,
					     GValue             *value,
					     GParamSpec         *pspec);
static GList * module_files_query           (LbModule           *module,
					     const gchar        *match);
static void    module_files_add_actions     (LbModule           *module,
					     LbItem             *item);
static void    module_files_activate_action (LbAction           *action,
					     LbModuleFiles      *module);

G_DEFINE_TYPE (LbModuleFiles, lb_module_files, LB_TYPE_MODULE);
static LbModuleClass *parent_class = NULL;

static void
lb_module_files_class_init (LbModuleFilesClass *klass)
{
	GObjectClass  *object_class = G_OBJECT_CLASS (klass);
	LbModuleClass *module_class = LB_MODULE_CLASS (klass);
	GConfClient   *client;
	gboolean       desktop_is_home;
	gchar         *desktop;
	gchar         *uri;

	parent_class = g_type_class_peek_parent (klass);

	object_class->finalize     = module_files_finalize;
	object_class->set_property = module_files_set_property;
	object_class->get_property = module_files_get_property;

	module_class->query        = module_files_query;
	module_class->add_actions  = module_files_add_actions;

	client = gconf_client_get_default();
	desktop_is_home = gconf_client_get_bool(client, "/apps/nautilus/preferences/desktop_is_home_dir", NULL);

	desktop = g_build_filename (g_get_home_dir (), desktop_is_home ? NULL : "Desktop", NULL);
	uri = g_filename_to_uri (desktop, NULL, NULL);
	g_free (desktop);

	g_object_unref(client);

	g_object_class_install_property (object_class, PROP_DIR_URI,
					 g_param_spec_string ("dir-uri",
							      NULL, NULL,
							      uri,
							      G_PARAM_READWRITE |
							      G_PARAM_CONSTRUCT));

	g_free (uri);

	g_type_class_add_private (object_class,
				  sizeof (LbModuleFilesPriv));
}

static void
lb_module_files_init (LbModuleFiles *files)
{
}

static void
module_files_finalize (GObject *object)
{
	LbModuleFilesPriv *priv = GET_PRIV (object);

	g_free (priv->dir_uri);

	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
module_files_set_property (GObject      *object,
			   guint         property_id,
			   const GValue *value,
			   GParamSpec   *pspec)
{
	LbModuleFilesPriv *priv = GET_PRIV (object);

	switch (property_id) {
	case PROP_DIR_URI:
		g_free (priv->dir_uri);
		priv->dir_uri = g_value_dup_string (value);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}

static void
module_files_get_property (GObject    *object,
			   guint       property_id,
			   GValue     *value,
			   GParamSpec *pspec)
{
	LbModuleFilesPriv *priv = GET_PRIV (object);

	switch (property_id) {
	case PROP_DIR_URI:
		g_value_set_string (value, priv->dir_uri);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}

static GList *
module_files_query (LbModule    *module,
		    const gchar *match)
{
	LbModuleFilesPriv       *priv  = GET_PRIV (module);
	GnomeVFSDirectoryHandle *handle;
	GnomeVFSFileInfo        *info;
	GnomeVFSResult           r;
	GList                   *items = NULL;

	r = gnome_vfs_directory_open (&handle, priv->dir_uri,
				  GNOME_VFS_FILE_INFO_GET_MIME_TYPE);

	if(r != GNOME_VFS_OK) {
		g_warning("error opening %s: %s", priv->dir_uri, gnome_vfs_result_to_string(r));
		return NULL;
	}

	info = gnome_vfs_file_info_new ();

	while (gnome_vfs_directory_read_next (handle, info) == GNOME_VFS_OK) {
		gchar *name;
		gchar *uri;

		name = g_filename_to_utf8 (info->name, -1, NULL, NULL, NULL);
		if (!name) {
			continue;
		}

		uri = g_build_filename (priv->dir_uri, name, NULL);

		if (name[0] != '.' && name[strlen (name) - 1] != '~') {
			gchar *dot = strrchr (name, '.');

			if (dot)
				*dot = '\0';

			if (lb_string_has_substring (name, match)) {
				LbItem    *item;
				gchar     *icon_name;

				icon_name = lb_get_icon_name_for_uri (uri, info);

				item = g_object_new (LB_TYPE_ITEM_FILE,
						     "name",      name,
						     "icon-name", icon_name,
						     "uri",       uri,
						     "mime-type", info->mime_type,
						     NULL);

				g_free (icon_name);

				items = g_list_prepend (items, item);
			}
		}

		g_free (uri);
		g_free (name);
		gnome_vfs_file_info_clear (info);
	}

	gnome_vfs_directory_close (handle);

	return items;
}

static void
module_files_add_actions (LbModule *module,
			  LbItem   *item)
{
	LbItemFile              *item_file;
	GList                   *applications;
	GList                   *list;
	const gchar             *mime_type;
	GnomeVFSMimeApplication *default_application;

	if (! LB_IS_ITEM_FILE (item)) {
		return;
	}

	item_file = LB_ITEM_FILE (item);

	mime_type = lb_item_file_get_mime_type (item_file);

	if (! mime_type) {
		GnomeVFSFileInfo *info;

		info = gnome_vfs_file_info_new ();

		if (gnome_vfs_get_file_info (lb_item_file_get_uri (item_file),
					     info,
					     GNOME_VFS_FILE_INFO_GET_MIME_TYPE) ==
		    GNOME_VFS_OK) {
			lb_item_file_set_mime_type (item_file, info->mime_type);
		}

		gnome_vfs_file_info_unref (info);

		mime_type = lb_item_file_get_mime_type (item_file);
	}

	if (! mime_type) {
		return;
	}

	default_application = gnome_vfs_mime_get_default_application (mime_type);
	applications = gnome_vfs_mime_get_all_applications (mime_type);

	for (list = applications; list ; list = list->next) {
		GnomeVFSMimeApplication *application = list->data;
		LbAction                *action;
		const gchar             *name;
		const gchar             *icon;

		icon = gnome_vfs_mime_application_get_icon (application);
		name = gnome_vfs_mime_application_get_name (application);

		action = g_object_new (LB_TYPE_ACTION,
				       "name",      name,
				       "icon-name", icon,
				       NULL);

		lb_action_set_data (action, application,
				    (GDestroyNotify)
				    gnome_vfs_mime_application_free);

		g_signal_connect (action, "activate",
				  G_CALLBACK (module_files_activate_action),
				  module);

		lb_item_add_action (item, action);

		g_object_unref (action);

#if 0 /* FIXME: Should increase relevance of an action if it's the default
	 action */
		if (gnome_vfs_mime_application_equal (application, default_application)) {
			default_action = action;
		} else {
			actions = g_list_prepend (actions, action);
		}
#endif
	}

	g_list_free (applications);
}

static void
module_files_activate_action (LbAction      *action,
			      LbModuleFiles *module)
{
	GnomeVFSMimeApplication *application;
	LbItemFile              *item;
	GList                    list;

	item = LB_ITEM_FILE (lb_action_get_item (action));

	list.next = NULL;
	list.data = (gpointer) lb_item_file_get_uri (item);

	application = (GnomeVFSMimeApplication *) lb_action_get_data (action);

	gnome_vfs_mime_application_launch (application, &list);

	g_print ("activate me!\n");
}
