#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <errno.h>

#include "kmfoldercachedimap.h"
#include "kmfolderimap.h"
#include "kmundostack.h"
#include "kmfoldermgr.h"
#include "kmmessage.h"
#include "kmacctcachedimap.h"
#include "kmacctmgr.h"
#include "imapprogressdialog.h"

#include <kapplication.h>
#include <kmessagebox.h>
#include <klocale.h>
#include <kdebug.h>
#include <kio/global.h>
#include <kio/scheduler.h>
#include <qbuffer.h>
#include <qfile.h>
#include <qlabel.h>
#include <qlayout.h>

#define UIDCACHE_VERSION 1


DImapTroubleShootDialog::DImapTroubleShootDialog( QWidget* parent,
						  const char* name )
  : KDialogBase( Plain, i18n( "Troubleshooting the IMAP cache" ),
		 Cancel | User1 | User2, Cancel, parent, name, true ),
    rc( Cancel )
{
  QFrame* page = plainPage();
  QVBoxLayout *topLayout = new QVBoxLayout( page, 0 );
  QString txt = i18n( "<p><b>Troubleshooting the IMAP cache.</b></p>"
		      "<p>If you have problems with synchronizing an IMAP "
		      "folder, you should first try rebuilding the index "
		      "file. This will take some time to rebuild, but will "
		      "not cause any problems.</p><p>If that is not enough, "
		      "you can try refreshing the IMAP cache. If you do this, "
		      "you will loose all your local changes for this folder "
		      "and all it's subfolders.</p>" );
  topLayout->addWidget( new QLabel( txt, page ) );
  enableButtonSeparator( true );

  setButtonText( User1, i18n( "Refresh &Cache" ) );
  setButtonText( User2, i18n( "Rebuild &Index" ) );

  connect( this, SIGNAL( user1Clicked () ), this, SLOT( slotRebuildCache() ) );
  connect( this, SIGNAL( user2Clicked () ), this, SLOT( slotRebuildIndex() ) );
}

int DImapTroubleShootDialog::run()
{
  DImapTroubleShootDialog d;
  d.exec();
  return d.rc;
}

void DImapTroubleShootDialog::slotRebuildCache()
{
  rc = User1;
  done( User1 );
}

void DImapTroubleShootDialog::slotRebuildIndex()
{
  rc = User2;
  done( User2 );
}


KMFolderCachedImap::KMFolderCachedImap(KMFolderDir* aParent, const QString& aName)
 : KMFolderCachedImapInherited(aParent, aName), mSyncState(SYNC_STATE_INITIAL),
   mContentState(imapNoInformation),
   mSubfolderState(imapNoInformation), mIsSelected(FALSE), mCheckFlags(TRUE), mAccount(NULL),
   mLastUid(0), mIsConnected(false), mFolderRemoved(false), mResync( false ),
   mSuppressDialog(false), mHoldSyncs( false ), mRemoveRightAway( false )
{
  KConfig* config = KMKernel::config();
  KConfigGroupSaver saver(config, "Folder-" + idString());
  if (mImapPath.isEmpty()) mImapPath = config->readEntry("ImapPath");
  if (aName == "INBOX" && mImapPath == "/INBOX/")
  {
    //mLabel = i18n("inbox");
  }
  mIsSystemFolder = false;
  mNoContent = config->readBoolEntry("NoContent", FALSE);
  mReadOnly = config->readBoolEntry("ReadOnly", FALSE);

  connect( this, SIGNAL( listMessagesComplete() ), this, SLOT( serverSyncInternal() ) );

  setUidValidity("");
  mLastUid=0;
  readUidCache();

  mProgress = 0;
}

KMFolderCachedImap::~KMFolderCachedImap()
{
  if( !mFolderRemoved ) {
    // Only write configuration when the folder haven't been deleted
    KConfig* config = KMKernel::config();
    KConfigGroupSaver saver(config, "Folder-" + idString());
    config->writeEntry("ImapPath", mImapPath);
    config->writeEntry("NoContent", mNoContent);
    config->writeEntry("ReadOnly", mReadOnly);

    writeUidCache();
  }

  if (kernel->undoStack()) kernel->undoStack()->folderDestroyed(this);
}

int KMFolderCachedImap::remove()
{
  mFolderRemoved = true;
  int rc = KMFolderCachedImapInherited::remove();

  if( mRemoveRightAway ) {
    // This is the account folder of an account that was just removed
    // When this happens, be sure to delete all traces of the cache
    QString part1 = path() + "/." + dotEscape(name());
    QString uidCacheFile = part1 + ".uidcache";
    if( QFile::exists(uidCacheFile) )
      unlink( QFile::encodeName( uidCacheFile ) );
    KIO::del( part1 + ".directory" );
  } else {
    // Don't remove the uidcache file here, since presence of that is how
    // we figure out if a directory present on the server have been deleted
    // from the cache or if it's new on the server. The file is removed
    // during the sync
  }

  return rc;
}

QString KMFolderCachedImap::uidCacheLocation() const
{
  QString sLocation(path());
  if (!sLocation.isEmpty()) sLocation += '/';
  return sLocation + '.' + dotEscape(fileName()) + ".uidcache";
}

int KMFolderCachedImap::readUidCache()
{
  QFile uidcache( uidCacheLocation() );
  if( uidcache.open( IO_ReadOnly ) ) {
    char buf[1024];
    int len = uidcache.readLine( buf, sizeof(buf) );
    if( len > 0 ) {
      int cacheVersion;
      sscanf( buf, "# KMail-UidCache V%d\n",  &cacheVersion );
      if( cacheVersion == UIDCACHE_VERSION ) {
	len = uidcache.readLine( buf, sizeof(buf) );
	if( len > 0 ) {
	  setUidValidity( QString::fromLocal8Bit( buf).stripWhiteSpace() );
	  len = uidcache.readLine( buf, sizeof(buf) );
	  if( len > 0 ) {
	    mLastUid = QString::fromLocal8Bit( buf).stripWhiteSpace().toULong();
	    return 0;
	  }
	}
      }
    }
  }
  return -1;
}

int KMFolderCachedImap::writeUidCache()
{
  QFile uidcache( uidCacheLocation() );
  if( uidcache.open( IO_WriteOnly ) ) {
    QTextStream str( &uidcache );
    str << "# KMail-UidCache V" << UIDCACHE_VERSION << endl;
    str << uidValidity() << endl;
    str << lastUid() << endl;
    uidcache.flush();
    fsync( uidcache.handle() ); /* this is probably overkill */
    uidcache.close();
    return 0;
  } else {
    return errno; /* does QFile set errno? */
  }
}

int KMFolderCachedImap::create(bool imap)
{
  int rc = KMFolderCachedImapInherited::create(imap);
  mLastUid = 0;
  mUidValidity = "";
  if( !rc ) return writeUidCache();
  return rc;
}

void KMFolderCachedImap::reloadUidMap()
{
  uidMap.clear();
  uidRevMap.clear();
  open();
  for( int i = 0; i < count(); ++i ) {
    bool unget = !isMessage(i);
    bool ok;
    KMMessage *msg = getMsg(i);
    if( !msg ) continue;
    ulong uid = msg->headerField("X-UID").toULong(&ok);
    if (unget) unGetMsg(i);
    if( ok ) {
      uidMap.insert( uid, i );
      uidRevMap.insert( i, uid );
      if( uid > mLastUid ) setLastUid( uid );
    }
  }
  close();
}

/* Reimplemented from KMFolderMaildir */
KMMessage* KMFolderCachedImap::take(int idx)
{
  QMap<int,ulong>::Iterator it = uidRevMap.find(idx);
  if( it != uidRevMap.end() ) {
    uidMap.remove( it.data() );
    uidRevMap.remove( idx );
  }
  return KMFolderCachedImapInherited::take(idx);
}

// Add a message without clearing it's X-UID field.
int KMFolderCachedImap::addMsgInternal(KMMessage* msg, int* index_return)
{
  bool ok;
  int idx_return;

  // Add the message
  ulong uid = msg->headerField("X-UID").toULong(&ok);
  int rc = KMFolderCachedImapInherited::addMsg(msg, &idx_return);
  if( index_return ) *index_return = idx_return;

  // Put it in the uid maps
  // TODO: Do we really want to add it if there is no X-UID header?
  if( ok && !rc && idx_return >= 0 ) {
    uidMap.insert( uid, idx_return );
    uidRevMap.insert( idx_return, uid );
    if( uid > mLastUid ) setLastUid( uid );
  }

  return rc;
}

/* Reimplemented from KMFolderMaildir */
int KMFolderCachedImap::addMsg(KMMessage* msg, int* index_return)
{
  assert(msg);

  // Strip the IMAP UID
  msg->removeHeaderField( "X-UID" );

  // Add it to storage
  return addMsgInternal( msg, index_return );
}


/* Reimplemented from KMFolderMaildir */
void KMFolderCachedImap::removeMsg(int idx, bool imapQuiet)
{
  // kdDebug(5006) << "KMFolderCachedImap::removeMsg(" << idx << ", " << imapQuiet << ")" << endl;

  // Remove it from the maps
  QMap<int,ulong>::Iterator it = uidRevMap.find(idx);
  if( it != uidRevMap.end() ) {
    ulong uid = it.data();
    uidMap.remove( uid );
    uidRevMap.remove( idx );
  }

  // Remove it from disk
  KMFolderCachedImapInherited::removeMsg(idx,imapQuiet);

  // TODO (Bo): Shouldn't this be "emit changed();"?
  kernel->imapFolderMgr()->contentsChanged();
}

bool KMFolderCachedImap::canRemoveFolder() const {
  // If this has subfolders it can't be removed
  if( child() != 0 && child()->count() > 0 )
    return false;

  // No special condition here, so let base class decide
  return KMFolderCachedImapInherited::canRemoveFolder();
}

/* Reimplemented from KMFolderDir */
int KMFolderCachedImap::rename(const QString& aName, KMFolderDir* /*aParent*/)
{
  if ( aName == name() )
    // Stupid user trying to rename it to it's old name :)
    return 0;

  if( mSyncState != SYNC_STATE_INITIAL ) {
    KMessageBox::error( 0, i18n("You can't rename a folder when a sync is in progress") );
    return -1;
  }

  if( account() == 0 ) {
    QString err = i18n("You must synchronize with the server before renaming IMAP folders.");
    KMessageBox::error( 0, err );
    return -1;
  }

  bool ok;
  new KMail::CachedImapJob( aName, ok, KMail::CachedImapJob::tRenameFolder, this );
  return 0;
}

void KMFolderCachedImap::setLastUid( ulong uid ) {
  mLastUid = uid;
  writeUidCache();
}

ulong KMFolderCachedImap::lastUid() {
  // kdDebug(5006) << "KMFolderCachedImap::lastUid("<<name()<<") = " << mLastUid << endl;
  return mLastUid;
}

KMMessage* KMFolderCachedImap::findByUID( ulong uid )
{
  QMap<ulong,int>::Iterator it = uidMap.find( uid );
  if( it != uidMap.end() ) {
    bool unget = !isMessage(count() - 1);
    KMMessage* msg = getMsg( *it );
    if( msg && msg->headerField("X-UID").toULong() == uid )
      return msg;
    else if( unget )
      unGetMsg( *it );
  }

  // Not found by now. That probably mean there is a problem in the
  // maps. Rebuild them and try again
  reloadUidMap();

  it = uidMap.find( uid );
  if( it != uidMap.end() )
    // Since the uid map is just rebuilt, no need for the sanity check
    return getMsg( *it );

  // Then it's not here
  return 0;
}

// This finds and sets the proper account for this folder if it has not been done
KMAcctCachedImap *KMFolderCachedImap::account()
{
  if( (KMAcctCachedImap *)mAccount == 0 ) {
    // Find the account
    mAccount = static_cast<KMAcctCachedImap *>( kernel->acctMgr()->find( name() ) );
  }

  return mAccount;
}

void KMFolderCachedImap::slotTroubleshoot()
{
  int rc = DImapTroubleShootDialog::run();

  if( rc == KDialogBase::User1 ) {
    // Refresh cache
    if( !account() ) {
      KMessageBox::sorry( 0, i18n("No account setup for this folder.\n"
				  "Please try running a sync before this.") );
      return;
    }
    QString str = i18n("Are you sure you want to refresh the IMAP cache of "
		       "the folder %1 and all it's subfolders?\nThis will "
		       "remove all changes you have done locally to your "
		       "folders").arg( name() );
    QString s1 = i18n("Refresh IMAP Cache");
    QString s2 = i18n("&Refresh");
    if( KMessageBox::warningContinueCancel( 0, str, s1, s2 ) ==
	KMessageBox::Continue )
      account()->invalidateIMAPFolders( this );
  } else if( rc == KDialogBase::User2 ) {
    // Rebuild index file
    createIndexFromContents();
    KMessageBox::information( 0, i18n( "The index of this folder have been "
				       "recreated." ) );
  }
}

void KMFolderCachedImap::processNewMail()
{
  if( account() )
    account()->processNewMail( this, true );
}

void KMFolderCachedImap::serverSync( bool suppressDialog )
{
  assert( account() );

  reloadUidMap();
  // kdDebug(5006) << "KMFolderCachedImap::serverSync(), imapPath()=" << imapPath() << ", path()="
  //    << path() << " name()="<< name() << endl;

  // Connect to the imap progress dialog
  mSuppressDialog = suppressDialog;
  if( mIsConnected != mAccount->isProgressDialogEnabled() &&
      suppressDialog )
  {
    if( !mIsConnected )
      connect( this, SIGNAL( newState( const QString&, int, const QString& ) ),
	       account()->imapProgressDialog(),
	       SLOT( syncState( const QString&, int, const QString& ) ) );
    else
      disconnect( this, SIGNAL( newState( const QString&, int, const QString& ) ),
	       account()->imapProgressDialog(),
	       SLOT( syncState( const QString&, int, const QString& ) ) );
    mIsConnected = mAccount->isProgressDialogEnabled();
  }

  if( mHoldSyncs ) {
    // All done for this folder.
    mProgress = 100; // all done
    emit newState( name(), mProgress, i18n("Synchronization skipped"));
    mAccount->displayProgress();
    mSyncState = SYNC_STATE_INITIAL;
    emit statusMsg( i18n("%1: Synchronization done").arg(name()) );
    emit folderComplete( this, true );
    return;
  }

  mResync = false;
  serverSyncInternal();
}

QString KMFolderCachedImap::state2String( int state ) const
{
  switch( state ) {
  case SYNC_STATE_INITIAL:           return "SYNC_STATE_INITIAL";
  case SYNC_STATE_PUT_MESSAGES:      return "SYNC_STATE_PUT_MESSAGES";
  case SYNC_STATE_CREATE_SUBFOLDERS: return "SYNC_STATE_CREATE_SUBFOLDERS";
  case SYNC_STATE_LIST_SUBFOLDERS:   return "SYNC_STATE_LIST_SUBFOLDERS";
  case SYNC_STATE_LIST_SUBFOLDERS2:  return "SYNC_STATE_LIST_SUBFOLDERS2";
  case SYNC_STATE_DELETE_SUBFOLDERS: return "SYNC_STATE_DELETE_SUBFOLDERS";
  case SYNC_STATE_LIST_MESSAGES:     return "SYNC_STATE_LIST_MESSAGES";
  case SYNC_STATE_DELETE_MESSAGES:   return "SYNC_STATE_DELETE_MESSAGES";
  case SYNC_STATE_GET_MESSAGES:      return "SYNC_STATE_GET_MESSAGES";
  case SYNC_STATE_FIND_SUBFOLDERS:   return "SYNC_STATE_FIND_SUBFOLDERS";
  case SYNC_STATE_SYNC_SUBFOLDERS:   return "SYNC_STATE_SYNC_SUBFOLDERS";
  case SYNC_STATE_EXPUNGE_MESSAGES:  return "SYNC_STATE_EXPUNGE_MESSAGES";
  case SYNC_STATE_HANDLE_INBOX:      return "SYNC_STATE_HANDLE_INBOX";
  case SYNC_STATE_CHECK_UIDVALIDITY: return "SYNC_STATE_CHECK_UIDVALIDITY";
  default:                           return "Unknown state";
  }
}


// While the server synchronization is running, mSyncState will hold
// the state that should be executed next
void KMFolderCachedImap::serverSyncInternal()
{
  switch( mSyncState ) {
  case SYNC_STATE_INITIAL:
  {
    mProgress = 0;
    emit statusMsg( i18n("%1: Synchronizing").arg(name()) );
    emit newState( name(), mProgress, i18n("Syncronizing"));

    open();

    kdDebug() << k_funcinfo << " making connection" << endl;
    // Connect to the server (i.e. prepare the slave)
    KMail::ImapAccountBase::ConnectionState cs = mAccount->makeConnection();
    if ( cs == KMail::ImapAccountBase::Error ) // cancelled by user, or slave can't start
    {
        kdDebug(5006) << "makeConnection said Error, aborting." << endl;
        // We stop here. We're already in SYNC_STATE_INITIAL for the next time.
        emit folderComplete(this, FALSE);
        break;
    } else if ( cs == KMail::ImapAccountBase::Connecting )
    {
        kdDebug(5006) << "makeConnection said Connecting, waiting for signal." << endl;
        // We'll wait for the connectionResult signal from the account.
        connect( mAccount, SIGNAL( connectionResult(int) ),
                 this, SLOT( slotConnectionResult(int) ) );
        break;
    } else // Connected
    {
        kdDebug(5006) << "makeConnection said Connected, proceeding." << endl;
        mSyncState = SYNC_STATE_CHECK_UIDVALIDITY;
        // Fall through to next state
    }
  }
  case SYNC_STATE_CHECK_UIDVALIDITY:
    emit syncRunning( this, true );
    mSyncState = SYNC_STATE_CREATE_SUBFOLDERS;
    if( !noContent() ) {
      // TODO (Bo): How can we obtain the UID validity on a noContent folder?
      // TODO (Bo): Is it perhaps not necessary to do so?
      checkUidValidity();
      break;
    }
    // Else carry on

  case SYNC_STATE_CREATE_SUBFOLDERS:
    mSyncState = SYNC_STATE_PUT_MESSAGES;
    createNewFolders();
    break;

  case SYNC_STATE_PUT_MESSAGES:
    mSyncState = SYNC_STATE_LIST_SUBFOLDERS;
    if( !noContent() ) {
      uploadNewMessages();
      break;
    }
    // Else carry on

  case SYNC_STATE_LIST_SUBFOLDERS:
    mSyncState = SYNC_STATE_LIST_SUBFOLDERS2;
    mProgress += 10;
    //kdDebug() << name() << ": +10 -> " << mProgress << "%" << endl;
    emit statusMsg( i18n("%1: Retrieving folderlist").arg(name()) );
    emit newState( name(), mProgress, i18n("Retrieving folderlist"));
    if( !listDirectory() ) {
      mSyncState = SYNC_STATE_INITIAL;
      KMessageBox::error(0, i18n("Error during listDirectory()"));
    }
    break;

  case SYNC_STATE_LIST_SUBFOLDERS2:
    mSyncState = SYNC_STATE_DELETE_SUBFOLDERS;
    mProgress += 10;
    //kdDebug() << name() << ": +10 -> " << mProgress << "%" << endl;
    emit newState( name(), mProgress, i18n("Retrieving subfolders"));
    listDirectory2();
    break;

  case SYNC_STATE_DELETE_SUBFOLDERS:
    mSyncState = SYNC_STATE_LIST_MESSAGES;
    emit syncState( SYNC_STATE_DELETE_SUBFOLDERS, foldersForDeletionOnServer.count() );
    if( !foldersForDeletionOnServer.isEmpty() ) {
      emit statusMsg( i18n("%1: Deleting folders %2 from server").arg(name())
		      .arg( foldersForDeletionOnServer.join(", ") ) );
      emit newState( name(), mProgress, i18n("Deleting folders from server"));
      bool ok;
      KMail::CachedImapJob* job =
	new KMail::CachedImapJob( foldersForDeletionOnServer, ok,
				  KMail::CachedImapJob::tDeleteFolders, this );
      if( job && ok ) {
	connect( job, SIGNAL( finished() ),
		 this, SLOT( serverSyncInternal() ) );
	break;
      } // else carry on
    } else
      emit newState( name(), mProgress, i18n("No folders to delete from server"));
      // Carry on

  case SYNC_STATE_LIST_MESSAGES:
    mSyncState = SYNC_STATE_DELETE_MESSAGES;
    mProgress += 10;
    //kdDebug() << name() << ": +10 -> " << mProgress << "%" << endl;
    if( !noContent() ) {
      emit statusMsg( i18n("%1: Retrieving messagelist").arg(name()) );
      emit newState( name(), mProgress, i18n("Retrieving messagelist"));
      // emit syncState( SYNC_STATE_LIST_MESSAGES, foldersForDeletionOnServer.count() );
      listMessages();
      break;
    }
    // Else carry on

  case SYNC_STATE_DELETE_MESSAGES:
    mSyncState = SYNC_STATE_EXPUNGE_MESSAGES;
    if( !noContent() ) {
      if( deleteMessages() ) {
	// Fine, we will continue with the next state
      } else {
	// No messages to delete, skip to GET_MESSAGES
	emit newState( name(), mProgress, i18n("No messages to delete..."));
	mSyncState = SYNC_STATE_GET_MESSAGES;
	serverSyncInternal();
      }
      break;
    }
    // Else carry on

  case SYNC_STATE_EXPUNGE_MESSAGES:
    mSyncState = SYNC_STATE_GET_MESSAGES;
    if( !noContent() ) {
      mProgress += 10;
      //kdDebug() << name() << ": +10 -> " << mProgress << "%" << endl;
      emit statusMsg( i18n("%1: Expunging deleted messages").arg(name()) );
      emit newState( name(), mProgress, i18n("Expunging deleted messages"));
      bool ok;
      KMail::CachedImapJob *job =
	new KMail::CachedImapJob( QString::null, ok,
				  KMail::CachedImapJob::tExpungeFolder, this );
      if( job && ok ) {
	connect( job, SIGNAL( finished() ),
		 this, SLOT( serverSyncInternal() ) );
	break;
      } // Else carry on
    }
    // Else carry on

  case SYNC_STATE_GET_MESSAGES:
    mSyncState = SYNC_STATE_HANDLE_INBOX;
    if( !noContent() ) {
      //emit syncState( SYNC_STATE_GET_MESSAGES, mMsgsForDownload.count() );
      if( !mMsgsForDownload.isEmpty() ) {
	emit statusMsg( i18n("%1: Retrieving new messages").arg(name()) );
	emit newState( name(), mProgress, i18n("Retrieving new messages"));
	bool ok;
	KMail::CachedImapJob *job =
	  new KMail::CachedImapJob( mMsgsForDownload, ok,
				    KMail::CachedImapJob::tGetMessage,
				    this );
	if( job && ok ) {
	  connect( job, SIGNAL( progress(unsigned long, unsigned long) ),
		   this, SLOT( slotProgress(unsigned long, unsigned long) ) );
	  connect( job, SIGNAL( finished() ),
		   this, SLOT( serverSyncInternal() ) );
	  break;
	}
	mMsgsForDownload.clear();
	// Carry on
      } else {
	emit newState( name(), mProgress, i18n("No new messages from server"));
      }
    }
    // Else carry on

  case SYNC_STATE_HANDLE_INBOX:
    // Wrap up the 'download emails' stage (which has a 20% span)
    mProgress += 20;
    //kdDebug() << name() << ": +20 -> " << mProgress << "%" << endl;

    if( mResync ) {
      // Some conflict have been resolved, so restart the sync
      mResync = false;
      mSyncState = SYNC_STATE_INITIAL;
    } else
      // Continue with the subfolders
      mSyncState = SYNC_STATE_FIND_SUBFOLDERS;

    if( imapPath() == "/INBOX/" ) {
      KMFolderNode* node = 0;
      if( kernel->groupware().isEnabled() )
	node = child()->hasNamedFolder( kernel->groupware().folderName( KFolderTreeItem::Inbox ) );
      if( node && !node->isDir() ) {
	// Here we need to move messages from INBOX to the "real" inbox
	KMFolder* inboxFolder = static_cast<KMFolder*>(node);
	open();
	inboxFolder->open();
	mAccount->setFolder( inboxFolder );
	while( count() > 0) {
	  KMMessage* m = getMsg(0);
	  inboxFolder->moveMsg(m);
	  mAccount->processNewMsg(m);
	}
	compact();
	close();
	inboxFolder->close();
	// Delete from INBOX
	if( deleteMessages() ) {
	  break;
	}
      } else {
	// Filter new messages
	QValueList<ulong>::Iterator it = mUidsForDownload.begin();
	QValueList<ulong>::Iterator end = mUidsForDownload.end();
	for ( ; it != end; ++it ) {
	  kdDebug(5006) << "/INBOX/: Considering " << *it << endl;
	  KMMessage* msg = findByUID( *it );
// 	  kdDebug(5006) << "msg: " << msg << " unread: "
// 			<< (msg?QString::number(msg->isUnread()):QString("-"))<< endl;
	  if( msg && msg->isUnread() ) {
// 	    kdDebug(5006) << "/INBOX/: Processing message\n";
// 	    kdDebug(5006) << "msg->parent: " << msg->parent()->location() << endl;
	    mAccount->processNewMsg( msg );
	  }
	}
      }
    }
    // Don't carry on - it might be that we must resync
    serverSyncInternal();
    break;

  case SYNC_STATE_FIND_SUBFOLDERS:
    {
      emit newState( name(), mProgress, i18n("Updating cache file"));

      // last state got new messages, update cache file
      writeUidCache();

      mSyncState = SYNC_STATE_SYNC_SUBFOLDERS;
      mSubfoldersForSync.clear();
      mCurrentSubfolder = 0;
      if( child() ) {
	KMFolderNode *node = child()->first();
	while( node ) {
	  if( !node->isDir() ) {
	    // kdDebug(5006) << "child folder " << node->name() << " is a "
             // << node->className() << endl;
	    if ( static_cast<KMFolderCachedImap*>(node)->imapPath() != "" )
	      // Only sync folders that have been accepted by the server
	      mSubfoldersForSync << static_cast<KMFolderCachedImap*>(node);
	  }
	  node = child()->next();
	}
      }
    }

    // All done for this folder.
    mProgress = 100; // all done
    emit newState( name(), mProgress, i18n("Synchronization done"));
    emit syncRunning( this, false );
    mAccount->displayProgress();
    // Carry on

  case SYNC_STATE_SYNC_SUBFOLDERS:
    {
      if( mCurrentSubfolder ) {
	disconnect( mCurrentSubfolder, SIGNAL( folderComplete(KMFolderCachedImap*, bool) ),
		    this, SLOT( serverSyncInternal() ) );
	mCurrentSubfolder = 0;
      }

      if( mSubfoldersForSync.isEmpty() ) {
	mSyncState = SYNC_STATE_INITIAL;
	emit statusMsg( i18n("%1: Synchronization done").arg(name()) );
	emit folderComplete( this, TRUE );
	close();
      } else {
	mCurrentSubfolder = mSubfoldersForSync.front();
	mSubfoldersForSync.pop_front();
	connect( mCurrentSubfolder, SIGNAL( folderComplete(KMFolderCachedImap*, bool) ),
		 this, SLOT( serverSyncInternal() ) );

	// kdDebug(5006) << "Sync'ing subfolder " << mCurrentSubfolder->imapPath() << endl;
	assert( mCurrentSubfolder->imapPath() != "" );
	mCurrentSubfolder->setAccount( account() );
	mCurrentSubfolder->serverSync( mSuppressDialog );
      }
    }
    break;

  default:
    kdDebug(5006) << "KMFolderCachedImap::serverSyncInternal() WARNING: no such state "
	      << mSyncState << endl;
  }
}

/* Connected to the imap account's connectionResult signal.
   Emitted when the slave connected or failed to connect.
*/
void KMFolderCachedImap::slotConnectionResult( int errorCode )
{
    kdDebug() << k_funcinfo << errorCode << endl;
    disconnect( mAccount, SIGNAL( connectionResult(int) ),
                this, SLOT( slotConnectionResult(int) ) );
    if ( !errorCode ) // success
    {
        mSyncState = SYNC_STATE_CHECK_UIDVALIDITY;
        serverSyncInternal();
    }
    else // error (error message already shown by the account)
    {
        emit folderComplete(this, FALSE);
    }
}

/* find new messages (messages without a UID) */
QValueList<KMMessage*> KMFolderCachedImap::findNewMessages()
{
  //kdDebug(5006) << "KMFolderCachedImap::findNewMessages(), message count is " << count() << endl;
  QValueList<KMMessage*> result;
  for( int i = 0; i < count(); ++i ) {
    bool unget = !isMessage(i);
    KMMessage *msg = getMsg(i);
    if( !msg ) continue; /* what goes on if getMsg() returns 0? */
    if( msg->headerField("X-UID").isEmpty() ) {
      result << msg;
    } else {
      if (unget) unGetMsg(i);
    }
  }
  return result;
}

/* Upload new messages to server */
void KMFolderCachedImap::uploadNewMessages()
{
  QValueList<KMMessage*> newMsgs = findNewMessages();
  emit syncState( SYNC_STATE_PUT_MESSAGES, newMsgs.count() );
  mProgress += 10;
  //kdDebug() << name() << ": +10 (uploadNewMessages) -> " << mProgress << "%" << endl;
  if( !newMsgs.isEmpty() ) {
    emit statusMsg( i18n("%1: Uploading messages to server").arg(name()) );

    emit newState( name(), mProgress, i18n("Uploading messages to server"));
    bool ok;
    KMail::CachedImapJob *job =
      new KMail::CachedImapJob( newMsgs, ok,
				KMail::CachedImapJob::tPutMessage, this );
    if( job && ok )
      connect( job, SIGNAL( finished() ), this, SLOT( serverSyncInternal() ) );
    else
      serverSyncInternal();
  } else {
    emit newState( name(), mProgress, i18n("No messages to upload to server"));

    serverSyncInternal();
  }
}

/* Upload new folders to server */
void KMFolderCachedImap::createNewFolders()
{
  QValueList<KMFolderCachedImap*> newFolders = findNewFolders();
  //emit syncState( SYNC_STATE_CREATE_SUBFOLDERS, newFolders.count() );
  mProgress += 10;
  //kdDebug() << name() << ": +10 (createNewFolders) -> " << mProgress << "%" << endl;
  if( !newFolders.isEmpty() ) {
    emit statusMsg( i18n("%1: Creating subfolders on server").arg(name()) );
    emit newState( name(), mProgress, i18n("Creating subfolders on server"));
    bool ok;
    KMail::CachedImapJob* job =
      new KMail::CachedImapJob( newFolders, ok,
				KMail::CachedImapJob::tAddSubfolders, this );
    if( job && ok )
      connect( job, SIGNAL( finished() ), this, SLOT( serverSyncInternal() ) );
    else {
      // All done for this folder.
      mProgress = 100; // all done
      emit newState( name(), mProgress, i18n("Synchronization skipped"));
      mAccount->displayProgress();
      mSyncState = SYNC_STATE_INITIAL;
      emit statusMsg( i18n("%1: Synchronization done").arg(name()) );
      emit folderComplete( this, true );
      return;
    }
  } else {
    serverSyncInternal();
  }
}

QValueList<KMFolderCachedImap*> KMFolderCachedImap::findNewFolders()
{
  QValueList<KMFolderCachedImap*> newFolders;
  if( child() ) {
    KMFolderNode *node = child()->first();
    while( node ) {
      if( !node->isDir() ) {
	if( !node->isA("KMFolderCachedImap") ) {
	  kdDebug(5006) << "KMFolderCachedImap::findNewFolders(): ARGH!!! " << node->name()
		    << " is not an IMAP folder. It is a " << node->className() << endl;
	  node = child()->next();
	  assert(0);
	}
	KMFolderCachedImap* folder = static_cast<KMFolderCachedImap*>(node);
	if( folder->imapPath() == "" ) newFolders << folder;
      }
      node = child()->next();
    }
  }
  return newFolders;
}

bool KMFolderCachedImap::deleteMessages()
{
  /* Delete messages from cache that are gone from the server */
  QPtrList<KMMsgBase> msgsForDeletion;
  reloadUidMap();
  // It is not possible to just go over all indices and remove them one by one
  // because the index list can get resized under us. So use msg pointers instead
  for( QMap<ulong,int>::Iterator it = uidMap.begin(); it != uidMap.end(); ++it ) {
    // kdDebug(5006) << "looking at " << it.key() << " in cache " << imapPath() << endl;
    if( !uidsOnServer.contains( it.key() ) ) {
      // kdDebug(5006) << "Uid" << it.key() << "(idx="<<it.data()<< ") not present on server" << endl;
      msgsForDeletion.append( mMsgList[it.data()] );
    }
  }

  if( !msgsForDeletion.isEmpty() ) {
    emit statusMsg( i18n("%1: Deleting removed messages from cache").arg(name()) );
    open();
    for( KMMsgBase *msg = msgsForDeletion.first(); msg; msg = msgsForDeletion.next() )
      KMFolder::removeMsg( msg );
    compact();
    close();

    // It is quite possible that the list have been resized, so we need to rebuild
    // the entire uid map
    reloadUidMap();
  }

  //emit syncState( SYNC_STATE_DELETE_MESSAGES, uidsForDeletionOnServer.count() );

  mProgress += 10;
  //kdDebug() << name() << ": +10 (deleteMessages) -> " << mProgress << "%" << endl;
  emit newState( name(), mProgress, i18n("Deleting removed messages from server"));

  /* Delete messages from the server that we dont have anymore */
  if( !uidsForDeletionOnServer.isEmpty() ) {
    emit statusMsg( i18n("%1: Deleting removed messages from server").arg(name()) );
    QStringList sets = makeSets( uidsForDeletionOnServer, true );
    uidsForDeletionOnServer.clear();
    if( sets.count() > 1 ) {
      //KMessageBox::error( 0, i18n("The number of messages scheduled for deletion is too large") );
      // Rerun the sync until the messages are all deleted
      mResync = true;
    }
    //kdDebug(5006) << "Deleting " << sets.front() << " from sever folder " << imapPath() << endl;
    bool ok;
    KMail::CachedImapJob *job =
      new KMail::CachedImapJob( sets.front(), ok,
				KMail::CachedImapJob::tDeleteMessage, this );
    if( job && ok ) {
      connect( job, SIGNAL( finished() ), this, SLOT( serverSyncInternal() ) );
      return true;
    } else
      return false;
  } else {
    return false;
  }
}

void KMFolderCachedImap::checkUidValidity() {
  // IMAP root folders doesn't seem to have a UID validity setting.
  // Also, don't try the uid validity on new folders
  if( imapPath().isEmpty() || imapPath() == "/" )
    // Just proceed
    serverSyncInternal();
  else {
    mProgress += 10;
    //kdDebug() << name() << ": +10 (checkUidValidity) -> " << mProgress << "%" << endl;
    emit newState( name(), mProgress, i18n("Checking folder validity"));
    emit statusMsg( i18n("%1: Checking folder validity").arg(name()) );
    bool ok;
    KMail::CachedImapJob *job =
      new KMail::CachedImapJob( ok, KMail::CachedImapJob::tCheckUidValidity,
				this );
    if( job && ok )
      connect( job, SIGNAL( finished() ), this, SLOT( serverSyncInternal() ) );
    else
      serverSyncInternal();
  }
}

/* This will only list the messages in a folder.
   No directory listing done*/
void KMFolderCachedImap::listMessages() {
  if( imapPath() == "/" ) {
    // Don't list messages on the root folder
    serverSyncInternal();
    return;
  }

  if( !mAccount->makeConnection() ) {
    emit listMessagesComplete();
    emit folderComplete(this, FALSE);
    return;
  }
  uidsOnServer.clear();
  uidsForDeletionOnServer.clear();
  mMsgsForDownload.clear();
  mUidsForDownload.clear();
  KURL url = mAccount->getUrl();
  url.setPath(imapPath() + ";UID=1:*;SECTION=ENVELOPE");
  KMAcctCachedImap::jobData jd( url.url(), this );

  KIO::SimpleJob *newJob = KIO::get(url, FALSE, FALSE);
  KIO::Scheduler::assignJobToSlave(mAccount->slave(), newJob);
  mAccount->insertJob(newJob, jd);

  connect( newJob, SIGNAL( result( KIO::Job* ) ),
	   this, SLOT( slotGetLastMessagesResult( KIO::Job* ) ) );
  connect( newJob, SIGNAL( data( KIO::Job*, const QByteArray& ) ),
	   this, SLOT( slotGetMessagesData( KIO::Job* , const QByteArray& ) ) );
}

void KMFolderCachedImap::slotGetLastMessagesResult(KIO::Job * job)
{
  getMessagesResult(job, true);
}


//-----------------------------------------------------------------------------
void KMFolderCachedImap::slotGetMessagesResult(KIO::Job * job)
{
  getMessagesResult(job, false);
}

// All this should be moved to CachedImapJob obviously...
void KMFolderCachedImap::slotGetMessagesData(KIO::Job * job, const QByteArray & data)
{
  KMAcctCachedImap::JobIterator it = mAccount->findJob(job);
  if ( it == mAccount->jobsEnd() ) { // Shouldn't happen
    kdDebug(5006) << "could not find job!?!?!" << endl;
    serverSyncInternal(); /* HACK^W Fix: we should at least try to keep going */
    return;
  }
  (*it).cdata += QCString(data, data.size() + 1);
  int pos = (*it).cdata.find("\r\n--IMAPDIGEST");
  if (pos > 0) {
    int p = (*it).cdata.find("\r\nX-uidValidity:");
    if (p != -1)
      setUidValidity((*it).cdata.mid(p + 17, (*it).cdata.find("\r\n", p+1) - p - 17));
    (*it).cdata.remove(0, pos);
  }
  pos = (*it).cdata.find("\r\n--IMAPDIGEST", 1);

  int flags;
  while (pos >= 0) {
    KMMessage *msg = new KMMessage;
    msg->fromString((*it).cdata.mid(16, pos - 16));
    flags = msg->headerField("X-Flags").toInt();
    bool ok;
    ulong uid = msg->headerField("X-UID").toULong(&ok);
    if( ok ) uidsOnServer.append( uid );
    if ( /*flags & 8 ||*/ uid <= lastUid()) {
      // kdDebug(5006) << "KMFolderCachedImap::slotGetMessagesData() : folder "<<name()<<" already has msg="<<msg->headerField("Subject") << ", UID="<<uid << ", lastUid = " << mLastUid << endl;
      /* If this message UID is not present locally, then it must
	 have been deleted by the user, so we delete it on the
	 server also.
      */
      KMMsgBase *existingMessage = findByUID(uid);
      if( !existingMessage ) {
	// kdDebug(5006) << "message with uid " << uid << " is gone from local cache. Must be deleted on server!!!" << endl;
	uidsForDeletionOnServer << uid;
      } else {
	/* The message is OK, update flags */
	existingMessage->setStatus( flagsToStatus( flags ) );
      }
      delete msg;
    } else {
      ulong size = msg->headerField("X-Length").toULong();
      mMsgsForDownload << KMail::CachedImapJob::MsgForDownload(uid, flags, size);
      if( imapPath() == "/INBOX/" )
	mUidsForDownload << uid;
#if 0
      msg->setStatus(flagsToStatus(flags));
      open();
      //KMFolderCachedImapInherited::addMsg(msg, NULL);
      msg->setComplete( FALSE );
      addMsgInternal(msg, NULL);
      //sync();
      //if (count() > 1) unGetMsg(count() - 1);
      //mLastUid = uid;
      close();
#endif
      /*
      QValueList<KMMessage*> msgList;
      msgList << msg;
      (void)new KMail::CachedImapJob( msgList, KMail::CachedImapJob::tGetMessage, this );
      */
/*      if ((*it).total > 20 &&
        ((*it).done + 1) * 5 / (*it).total > (*it).done * 5 / (*it).total)
      {
        quiet(FALSE);
        quiet(TRUE);
      } */
    }
    (*it).cdata.remove(0, pos);
    (*it).done++;
    pos = (*it).cdata.find("\r\n--IMAPDIGEST", 1);
    mAccount->displayProgress();
  }
}

void KMFolderCachedImap::getMessagesResult(KIO::Job * job, bool lastSet)
{
  if (lastSet) quiet( false );

  KMAcctCachedImap::JobIterator it = mAccount->findJob(job);
  if ( it == mAccount->jobsEnd() ) { // Shouldn't happen
    kdDebug(5006) << "could not find job!?!?!" << endl;
    serverSyncInternal(); /* HACK^W Fix: we should at least try to keep going */
    return;
  }

  if( job->error() ) {
    mAccount->slotSlaveError( mAccount->slave(), job->error(), job->errorText() );
    mContentState = imapNoInformation;
    emit folderComplete(this, FALSE);
  } else if (lastSet) mContentState = imapFinished;

  mAccount->removeJob(it);
  if( lastSet )
    emit listMessagesComplete();
}

void KMFolderCachedImap::slotProgress(unsigned long done, unsigned long total)
{
  //kdDebug() << "KMFolderCachedImap::slotProgress done=" << done << " total=" << total << "=> progress=" << mProgress + ( 20 * done ) / total << endl;
  // Progress info while retrieving new emails
  // (going from mProgress to mProgress+20)
  emit newState( name(), mProgress + (20 * done) / total, QString::null);
}

KMMsgStatus KMFolderCachedImap::flagsToStatus(int flags, bool newMsg)
{
  if (flags & 4) return KMMsgStatusFlag;
  if (flags & 2) return KMMsgStatusReplied;
  if (flags & 1) return KMMsgStatusOld;
  return (newMsg) ? KMMsgStatusNew : KMMsgStatusUnread;
}

QCString KMFolderCachedImap::statusToFlags(KMMsgStatus status)
{
  QCString flags = "";
  switch (status)
  {
    case KMMsgStatusNew:
    case KMMsgStatusUnread:
      break;
    case KMMsgStatusDeleted:
      flags = "\\DELETED";
      break;
    case KMMsgStatusReplied:
      flags = "\\SEEN \\ANSWERED";
      break;
    case KMMsgStatusFlag:
      flags = "\\SEEN \\FLAGGED";
      break;
    default:
      flags = "\\SEEN";
  }
  return flags;
}


void KMFolderCachedImap::setAccount(KMAcctCachedImap *aAccount)
{
  // kdDebug(5006) << "KMFolderCachedImap::setAccount( " << aAccount->name() << " )" << endl;
  assert( aAccount->isA("KMAcctCachedImap") );
  mAccount = aAccount;
  if( imapPath()=="/" ) aAccount->setFolder(this);

  if( !mChild || mChild->count() == 0) return;
  for( KMFolderNode* node = mChild->first(); node; node = mChild->next() )
    if (!node->isDir())
      static_cast<KMFolderCachedImap*>(node)->setAccount(aAccount);
}


// This synchronizes the subfolders with the server
bool KMFolderCachedImap::listDirectory() {
  // kdDebug(5006) << "KMFolderCachedImap::listDirectory " << "imapPath() = "
  //    << imapPath() << " mAccount->prefix() = " << mAccount->prefix() << endl;
  reloadUidMap();

  mSubfolderState = imapInProgress;
  KURL url = mAccount->getUrl();
  url.setPath(imapPath() + ";TYPE="
	      + (mAccount->onlySubscribedFolders() ? "LSUB" : "LIST"));
  KMAcctCachedImap::jobData jd( url.url(), this );
  mSubfolderNames.clear();
  mSubfolderPaths.clear();
  mSubfolderMimeTypes.clear();

  kdDebug(5006) << "listDirectory(): listing url " << url.url() << endl;
  if (!mAccount->makeConnection()) {
    emit folderComplete(this, FALSE);
    return FALSE;
  }

  kdDebug(5006) << "listDirectory(): Making the job " << endl;
  KIO::SimpleJob *job = KIO::listDir(url, FALSE);
  KIO::Scheduler::assignJobToSlave(mAccount->slave(), job);
  mAccount->insertJob(job, jd);
  connect(job, SIGNAL(result(KIO::Job *)),
          this, SLOT(slotListResult(KIO::Job *)));
  connect(job, SIGNAL(entries(KIO::Job *, const KIO::UDSEntryList &)),
          this, SLOT(slotListEntries(KIO::Job *, const KIO::UDSEntryList &)));

  return TRUE;
}

void KMFolderCachedImap::slotListResult(KIO::Job * job)
{
  KMAcctCachedImap::JobIterator it = mAccount->findJob(job);
  if ( it == mAccount->jobsEnd() ) { // Shouldn't happen
    kdDebug(5006) << "could not find job!?!?!" << endl;
    serverSyncInternal(); /* HACK^W Fix: we should at least try to keep going */
    return;
  }

  if (job->error()) {
    kdDebug(5006) << "listDirectory() - slotListResult: Job error\n";
    mAccount->slotSlaveError( mAccount->slave(), job->error(), job->errorText() );
  }

  mSubfolderState = imapFinished;
  mAccount->removeJob(it);

  if (!job->error()) {
    kernel->imapFolderMgr()->quiet(TRUE);
    createChildFolder();

    // Find all subfolders present on disk but not on the server
    KMFolderCachedImap *folder;
    KMFolderNode *node = mChild->first();
    QPtrList<KMFolder> toRemove;
    while (node) {
      if (!node->isDir() ) {
	if( mSubfolderNames.findIndex(node->name()) == -1) {
	  // This subfolder isn't present on the server
	  kdDebug(5006) << node->name() << " isn't on the server." << endl;
	  folder = static_cast<KMFolderCachedImap*>(node);
	  if (folder->uidValidity() != "") {
	    // The folder have a uidValidity setting, so it has been on the
	    // server before. Delete it locally.
	    toRemove.append( folder );
	  }
	}
      }
      node = mChild->next();
    }
    // Remove all folders
    for ( KMFolder* doomed=toRemove.first(); doomed; doomed = toRemove.next() )
      kernel->imapFolderMgr()->remove( doomed );

    mAccount->displayProgress();
    serverSyncInternal();
  }
}


void KMFolderCachedImap::listDirectory2() {
  foldersForDeletionOnServer.clear();

  // Find all subfolders present on server but not on disk
  for (uint i = 0; i < mSubfolderNames.count(); i++) {
    KMFolderCachedImap *folder = 0;

    // Find the subdir, if already present
    KMFolderNode *node;
    for (node = mChild->first(); node; node = mChild->next())
      if (!node->isDir() && node->name() == mSubfolderNames[i]) break;

    if (!node) {
      // This folder is not present here
      QString part1 = path() + "/." + dotEscape(name()) + ".directory/."
	+ dotEscape(mSubfolderNames[i]);
      QString uidCacheFile = part1 + ".uidcache";
      if( QFile::exists(uidCacheFile) ) {
	// This is an old folder that is deleted locally - delete it on the server
	unlink( QFile::encodeName( uidCacheFile ) );
	foldersForDeletionOnServer << mSubfolderPaths[i];
	// Make sure all trace of the dir is gone
	KIO::del( part1 + ".directory" );
      } else {
	// This is a new folder, create the local cache
	folder = static_cast<KMFolderCachedImap*>
	  (mChild->createFolder(mSubfolderNames[i], false, KMFolderTypeCachedImap));
	if (folder) {
	  folder->close();
	  folder->setAccount(mAccount);
	  kernel->imapFolderMgr()->contentsChanged();
	} else {
	  kdDebug(5006) << "can't create folder " << mSubfolderNames[i] <<endl;
	}
      }
    } else {
      // kdDebug(5006) << "node " << node->name() << " is a " << node->className() << endl;
      if( node->isA("KMFolderCachedImap") )
	folder = static_cast<KMFolderCachedImap*>(node);
    }

    if (folder && folder->imapPath() == "") {
      // kdDebug(5006) << "folder("<<folder->name()<<")->imapPath()=" << folder->imapPath()
      // << "\nAssigning new imapPath " << mSubfolderPaths[i] << endl;
      // Write folder settings
      folder->setAccount(mAccount);
      folder->setNoContent(mSubfolderMimeTypes[i] == "inode/directory");
      folder->setImapPath(mSubfolderPaths[i]);
    }
  }

  kernel->imapFolderMgr()->quiet(FALSE);
  emit listComplete(this);
  serverSyncInternal();
}


//-----------------------------------------------------------------------------
void KMFolderCachedImap::slotListEntries(KIO::Job * job, const KIO::UDSEntryList & uds)
{
  // kdDebug(5006) << "KMFolderCachedImap::slotListEntries("<<name()<<")" << endl;
  KMAcctCachedImap::JobIterator it = mAccount->findJob(job);
  if (it == mAccount->jobsEnd()) return;

  QString name;
  KURL url;
  QString mimeType;
  for (KIO::UDSEntryList::ConstIterator udsIt = uds.begin();
    udsIt != uds.end(); udsIt++)
  {
    // kdDebug(5006) << "slotListEntries start" << endl;
    mimeType = QString::null;

    // Find the info on this subfolder
    for (KIO::UDSEntry::ConstIterator eIt = (*udsIt).begin();
      eIt != (*udsIt).end(); eIt++)
    {
      //kdDebug(5006) << "slotListEntries got type " << (*eIt).m_uds << " str " << (*eIt).m_str << endl;
      if ((*eIt).m_uds == KIO::UDS_NAME)
        name = (*eIt).m_str;
      else if ((*eIt).m_uds == KIO::UDS_URL)
        url = KURL((*eIt).m_str, 106); // utf-8
      else if ((*eIt).m_uds == KIO::UDS_MIME_TYPE)
        mimeType = (*eIt).m_str;
    }

    // kdDebug(5006) << "slotListEntries end. mimetype = " << mimeType
    //      << ", name = " << name << ", path = " << url.path() << endl;

    // If this was a subfolder, add it to the list
    if ((mimeType == "inode/directory" || mimeType == "message/digest"
        || mimeType == "message/directory")
        && name != ".." && (mAccount->hiddenFolders() || name.at(0) != '.'))
    {
      // Some servers send _lots_ of duplicates
      if (mSubfolderNames.findIndex(name) == -1) {
	mSubfolderNames.append(name);
	mSubfolderPaths.append(url.path());
	mSubfolderMimeTypes.append(mimeType);
      }
    }
  }
}

void KMFolderCachedImap::slotSimpleData(KIO::Job * job, const QByteArray & data)
{
  KMAcctCachedImap::JobIterator it = mAccount->findJob(job);
  if (it == mAccount->jobsEnd()) return;
  QBuffer buff((*it).data);
  buff.open(IO_WriteOnly | IO_Append);
  buff.writeBlock(data.data(), data.size());
  buff.close();
}


QStringList KMFolderCachedImap::makeSets(QStringList& uids, bool sort)
{
  QValueList<ulong> tmp;
  for ( QStringList::Iterator it = uids.begin(); it != uids.end(); ++it )
    tmp.append( (*it).toInt() );
  return makeSets(tmp, sort);
}

QStringList KMFolderCachedImap::makeSets(QValueList<ulong>& uids, bool sort)
{
  QStringList sets;
  QString set;

  if (uids.size() == 1)
  {
    sets.append(QString::number(uids.first()));
    return sets;
  }

  if (sort) qHeapSort(uids);

  ulong last = 0;
  // needed to make a uid like 124 instead of 124:124
  bool inserted = false;
  /* iterate over uids and build sets like 120:122,124,126:150 */
  for ( QValueList<ulong>::Iterator it = uids.begin(); it != uids.end(); ++it )
  {
    if (it == uids.begin() || set.isEmpty()) {
      set = QString::number(*it);
      inserted = true;
    } else
    {
      if (last+1 != *it)
      {
        // end this range
        if (inserted)
          set += ',' + QString::number(*it);
        else
          set += ':' + QString::number(last) + ',' + QString::number(*it);
        inserted = true;
        if (set.length() > 100)
        {
          // just in case the server has a problem with longer lines..
          sets.append(set);
          set = "";
        }
      } else {
        inserted = false;
      }
    }
    last = *it;
  }
  // last element
  if (!inserted)
    set += ':' + QString::number(uids.last());

  if (!set.isEmpty()) sets.append(set);

  return sets;
}

#include "kmfoldercachedimap.moc"
