<?php
/**
 * The IMP_Search:: class contains all functions related to handling
 * searching mailboxes in IMP.
 *
 * $Horde: imp/lib/Search.php,v 1.29 2003/07/21 21:57:30 slusarz Exp $
 *
 * Copyright 2002-2003 Michael Slusarz <slusarz@bigworm.colorado.edu>
 *
 * See the enclosed file COPYING for license information (GPL). If you
 * did not receive this file, see http://www.fsf.org/copyleft/gpl.html.
 *
 * @author  Michael Slusarz <slusarz@bigworm.colorado.edu>
 * @version $Revision: 1.29 $
 * @since   IMP 4.0
 * @package imp
 */
class IMP_Search {

    /**
     * Run a search.
     *
     * @access public
     *
     * @return array  The sorted list.
     */
    function runSearch()
    {
        global $imp, $prefs;

        $mbox = '';
        $sorted = array();
        $use_pop3 = ($imp['base_protocol'] == 'pop3');

        if (empty($imp['searchquery']) || empty($imp['searchfolders'])) {
            return array();
        }

        require_once HORDE_BASE . '/lib/IMAP/Search.php';
        $imap_search = &IMAP_Search::singleton(array('pop3' => $use_pop3));

        /* Prepare the search query. */
        $query = unserialize($imp['searchquery']);

        /* How do we want to sort results? */
        $sortby = $prefs->getValue('sortby');
        if ($sortby == SORTTHREAD) {
            $sortby = SORTDATE;
        }
        $sortdir = $prefs->getValue('sortdir');

        /* Prepare the IMAP string. */
        if ($use_pop3) {
            $stream = &$imp['stream'];
        } else {
            require_once IMP_BASE . '/lib/IMAP.php';
            $imp_imap = &IMP_IMAP::singleton();
            $stream = &$imp_imap->openIMAPStream(null, OP_READONLY);
        }

        if (!$stream) {
            return $sorted;
        }

        for ($i = 0; $i < count($imp['searchfolders']); $i++) {
            if (!$use_pop3) {
                $mbox = IMP::serverString($imp['searchfolders'][$i]);
            }

            $results = $imap_search->searchSortMailbox($query, $stream, $mbox, $sortby, $sortdir);

            if (is_array($results)) {
                for ($j = 0; $j < count($results); $j++) {
                    $sorted[] = $results[$j] . IMP_IDX_SEP . $imp['searchfolders'][$i];
                }
            }
        }

        /* Close connection if not POP3. */
        if (!$use_pop3) {
            @imap_close($stream);
        }

        return $sorted;
    }

    /**
     * Set the IMAP search query in the IMP session.
     *
     * @access public
     *
     * @param object IMAP_Search_Query $query  The search query object.
     * @param array $folders                   The list of folders to search.
     */
    function createSearchQuery($query, $folders)
    {
        global $imp;

        $imp['searchquery'] = serialize($query);
        $imp['searchfolders'] = $folders;
    }

    /**
     * Generates a URL with any necessary information required for handling
     * a search mailbox added to the parameters.
     *
     * @access public
     *
     * @param string $page     Page name to link to.
     * @param string $mailbox  The mailbox to use on the linked page.
     *
     * @return string  URL to $page with any necessary search information
     *                 added to the parameter list of the URL.
     */
    function generateSearchUrl($page, $mailbox)
    {
        $link = Horde::applicationUrl($page);

        foreach (IMP_Search::getSearchParameters($mailbox) as $key => $val) {
            $link = Horde::addParameter($link, $key, $val);
        }

        return $link;
    }

    /**
     * Returns a list of parameters necessary for handling a search mailbox.
     *
     * @access public
     *
     * @param string $mailbox  The mailbox to use on the linked page.
     *
     * @return array  The list of parameters needed for handling a search
     *                mailbox (may be empty if not currently in a search
     *                mailbox).
     */
    function getSearchParameters($mailbox)
    {
        global $imp;

        $params = array();

        if ($imp['mailbox'] == IMP_SEARCH_MBOX) {
            $params['thismailbox'] = $mailbox;
            $params['mailbox'] = $imp['mailbox'];
        }

        return $params;
    }

}
