/*
 * Driver for C-Media CMI8338 and 8738 PCI soundcards.
 * Copyright (c) 2000-2001 by Takashi Iwai <tiwai@suse.de>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
 
#define __SND_OSS_COMPAT__
#define SND_MAIN_OBJECT_FILE

#include "../include/driver.h"
#include "../include/info.h"
#include "../include/control.h"
#include "../include/mixer.h"
#include "../include/pcm.h"
#include "../include/rawmidi.h"
#include "../include/mpu401.h"
#include "../include/sb.h"	/* for SB16 mixer definitions */
#include "../include/initval.h"

EXPORT_NO_SYMBOLS;
MODULE_DESCRIPTION("\
Driver: C-Media CMI8x38 PCI\n\
Card: C-Media CMI8338 PCI\n\
Card: C-Media CMI8738 PCI\n\
PCI: 0x13f6=0x0100\n\
PCI: 0x13f6=0x0101\n\
PCI: 0x13f6=0x0111\n\
PCI: 0x13f6=0x0112\n\
");
MODULE_LICENSE("GPL");

#define CARD_NAME "C-Media PCI"
#define DRIVER_NAME CARD_NAME

static int snd_index[SND_CARDS] = SND_DEFAULT_IDX;	/* Index 0-MAX */
static char *snd_id[SND_CARDS] = SND_DEFAULT_STR;	/* ID for this card */
static int snd_enable[SND_CARDS] = {[0 ... (SND_CARDS - 1)] = 1}; /* all enabled */
static int snd_dac_frame_size[SND_CARDS] = {[0 ... (SND_CARDS - 1)] = 128};
static int snd_adc_frame_size[SND_CARDS] = {[0 ... (SND_CARDS - 1)] = 128};
static int snd_enable_midi[SND_CARDS] = {[0 ... (SND_CARDS-1)] = 0};
static int snd_enable_fm[SND_CARDS] = {[0 ... (SND_CARDS-1)] = 0};
static int snd_mpu_port[SND_CARDS] = {0x330, [1 ... (SND_CARDS-1)]=-1};
static int snd_mpu_irq[SND_CARDS] = {[0 ... (SND_CARDS-1)]=-1};
static int snd_fm_port[SND_CARDS] = {0x388, [1 ... (SND_CARDS-1)]=-1};

MODULE_PARM(snd_index, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_index, "Index value for " CARD_NAME " soundcard.");
MODULE_PARM(snd_id, "1-" __MODULE_STRING(SND_CARDS) "s");
MODULE_PARM_DESC(snd_id, "ID string for " CARD_NAME " soundcard.");
MODULE_PARM(snd_enable, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_enable, "Enable this soundcard. [BOOL]");
MODULE_PARM(snd_dac_frame_size, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_dac_frame_size, "DAC frame size in kB for " CARD_NAME " soundcard.");
MODULE_PARM(snd_adc_frame_size, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_adc_frame_size, "ADC frame size in kB for " CARD_NAME " soundcard.");
MODULE_PARM(snd_enable_midi, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_enable_midi, "Enable MPU-401 port. [BOOL]");
MODULE_PARM(snd_mpu_port, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_mpu_port, "MPU-401 port # for " DRIVER_NAME " driver. [list=0x330,0x320,0x310,0x300]");
MODULE_PARM(snd_mpu_irq, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_mpu_irq, "MPU-401 irq # for " DRIVER_NAME " driver. (-1 = shared with pcm)");
MODULE_PARM(snd_enable_fm, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_enable_fm, "Enable FM OPL-3 synth. [BOOL]");
MODULE_PARM(snd_fm_port, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_fm_port, "FM port # for " DRIVER_NAME " driver. [list=0x388,0x3c8,0x3e0,0x3e8]");


/*
 * CM8x38 registers definition
 */

#define CM_REG_FUNCTRL0		0x00
#define CM_RST_CH1		0x00080000
#define CM_RST_CH0		0x00040000
#define CM_CHEN1		0x00020000
#define CM_CHEN0		0x00010000
#define CM_PAUSE1		0x00000008
#define CM_PAUSE0		0x00000004
#define CM_CHADC1		0x00000002	/* ch1, 0:playback, 1:record */
#define CM_CHADC0		0x00000001	/* ch0, 0:playback, 1:record */

#define CM_REG_FUNCTRL1		0x04
#define CM_ASFC_MASK		0x0000E000	/* ADC sampling frequency */
#define CM_ASFC_SHIFT		13
#define CM_DSFC_MASK		0x00001C00	/* DAC sampling frequency */
#define CM_DSFC_SHIFT		10
#define CM_SPDF_1		0x00000200	/* SPDIF IN/OUT at channel B */
#define CM_SPDF_0		0x00000100	/* SPDIF OUT only channel A */
#define CM_SPDFLOOP		0x00000080	/* ext. SPDIIF/OUT -> IN loopback */
#define CM_SPDO2DAC		0x00000040	/* SPDIF/OUT can be heard from internal DAC */
#define CM_INTRM		0x00000020	/* master control block (MCB) interrupt enabled */
#define CM_BREQ			0x00000010	/* bus master enabled */
#define CM_VOICE_EN		0x00000008	/* legacy voice (SB16,FM) */
#define CM_UART_EN		0x00000004	/* UART */
#define CM_JYSTK_EN		0x00000002	/* joy stick */

#define CM_REG_CHFORMAT		0x08

#define CM_CHB3D5C		0x80000000	/* 5 channels */
#define CM_CHB3D		0x20000000	/* 4,5,6 channels */

#define CM_CHIP_MASK1		0x1f000000
#define CM_CHIP_037		0x01000000

#define CM_AC3EN1		0x00100000	/* enable AC3: model 037 */
#define CM_SPD24SEL		0x00020000	/* 24bit spdif: model 037 */

#define CM_ADCBITLEN_MASK	0x0000C000	
#define CM_ADCBITLEN_16		0x00000000
#define CM_ADCBITLEN_15		0x00004000
#define CM_ADCBITLEN_14		0x00008000
#define CM_ADCBITLEN_13		0x0000C000

#define CM_ADCDACLEN_MASK	0x00003000
#define CM_ADCDACLEN_060	0x00000000
#define CM_ADCDACLEN_066	0x00001000
#define CM_ADCDACLEN_130	0x00002000
#define CM_ADCDACLEN_280	0x00003000

#define CM_CH1_SRATE_176K	0x00000800
#define CM_CH1_SRATE_88K	0x00000400
#define CM_CH0_SRATE_176K	0x00000200
#define CM_CH0_SRATE_88K	0x00000100

#define CM_CH1FMT_MASK		0x0000000C
#define CM_CH1FMT_SHIFT		2
#define CM_CH0FMT_MASK		0x00000003
#define CM_CH0FMT_SHIFT		0

#define CM_REG_INT_HLDCLR	0x0C
#define CM_CHIP_MASK2		0xff000000
#define CM_CHIP_039		0x04000000
#define CM_CHIP_039_6CH		0x01000000
#define CM_TDMA_INT_EN		0x00040000
#define CM_CH1_INT_EN		0x00020000
#define CM_CH0_INT_EN		0x00010000
#define CM_INT_HOLD		0x00000002
#define CM_INT_CLEAR		0x00000001

#define CM_REG_INT_STATUS	0x10
#define CM_INTR			0x80000000
#define CM_UARTINT		0x00010000
#define CM_LTDMAINT		0x00008000
#define CM_HTDMAINT		0x00004000
#define CM_CH1BUSY		0x00000008
#define CM_CH0BUSY		0x00000004
#define CM_CHINT1		0x00000002
#define CM_CHINT0		0x00000001

#define CM_REG_LEGACY_CTRL	0x14
#define CM_NXCHG		0x80000000	/* h/w multi channels? */
#define CM_VMPU_MASK		0x60000000	/* MPU401 i/o port address */
#define CM_VMPU_330		0x00000000
#define CM_VMPU_320		0x20000000
#define CM_VMPU_310		0x40000000
#define CM_VMPU_300		0x60000000
#define CM_VSBSEL_MASK		0x0C000000	/* SB16 base address */
#define CM_VSBSEL_220		0x00000000
#define CM_VSBSEL_240		0x04000000
#define CM_VSBSEL_260		0x08000000
#define CM_VSBSEL_280		0x0C000000
#define CM_FMSEL_MASK		0x03000000	/* FM OPL3 base address */
#define CM_FMSEL_388		0x00000000
#define CM_FMSEL_3C8		0x01000000
#define CM_FMSEL_3E0		0x02000000
#define CM_FMSEL_3E8		0x03000000
#define CM_ENSPDOUT		0x00800000	/* enable XPDIF/OUT to I/O interface */
#define CM_SPDCOPYRHT		0x00400000	/* set copyright spdif in/out */
#define CM_DAC2SPDO		0x00200000	/* enable wave+fm_midi -> SPDIF/OUT */
#define CM_SETRETRY		0x00010000	/* 0: legacy i/o wait (default), 1: legacy i/o bus retry */
#define CM_CHB3D6C		0x00008000	/* 5.1 channels support */
#define CM_LINE_AS_BASS		0x00006000	/* use line-in as bass */

#define CM_REG_MISC_CTRL	0x18
#define CM_PWD			0x80000000
#define CM_RESET		0x40000000
#define CM_SFIL_MASK		0x30000000
#define CM_TXVX			0x08000000
#define CM_N4SPK3D		0x04000000	/* analog 4ch output (duplicate) */
#define CM_SPDO5V		0x02000000	/* 5V spdif output */
#define CM_SPDIF48K		0x01000000	/* write */
#define CM_SPATUS48K		0x01000000	/* read */
#define CM_ENDBDAC		0x00800000	/* enable dual dac */
#define CM_XCHGDAC		0x00400000	/* 0: front=ch0, 1: front=ch1 */
#define CM_SPD32SEL		0x00200000	/* 0: 16bit SPDIF, 1: 32bit */
#define CM_SPDFLOOPI		0x00100000	/* int. SPDIF-IN -> int. OUT */
#define CM_FM_EN		0x00080000	/* enalbe FM */
#define CM_AC3EN2		0x00040000	/* enable AC3: model 039 */
#define CM_VIDWPDSB		0x00010000 
#define CM_SPDF_AC97		0x00008000	/* 0: SPDIF/OUT 44.1K, 1: 48K */
#define CM_MASK_EN		0x00004000
#define CM_VIDWPPRT		0x00002000
#define CM_SFILENB		0x00001000
#define CM_MMODE_MASK		0x00000E00
#define CM_ENCENTER		0x00000080	/* shared with FLINKON? */
#define CM_FLINKON		0x00000080
#define CM_FLINKOFF		0x00000040
#define CM_MIDSMP		0x00000010
#define CM_UPDDMA_MASK		0x0000000C
#define CM_TWAIT_MASK		0x00000003

	/* byte */
#define CM_REG_MIXER0		0x20

#define CM_REG_SB16_DATA	0x22
#define CM_REG_SB16_ADDR	0x23

#define CM_REG_MIXER1		0x24
#define CM_FMMUTE		0x80	/* mute FM */
#define CM_WSMUTE		0x40	/* mute PCM */
#define CM_SPK4			0x20	/* lin-in -> rear line out */
#define CM_REAR2FRONT		0x10	/* exchange rear/front */
#define CM_WAVEINL		0x08	/* digital wave rec. left chan */
#define CM_WAVEINR		0x04	/* digical wave rec. right */
#define CM_X3DEN		0x02	/* 3D surround enable */
#define CM_CDPLAY		0x01	/* enable SPDIF/IN PCM -> DAC */

#define CM_REG_MIXER2		0x25
#define CM_RAUXREN		0x80	/* AUX right capture */
#define CM_RAUXLEN		0x40	/* AUX left capture */
#define CM_VAUXRM		0x20	/* AUX right mute */
#define CM_VAUXLM		0x10	/* AUX left mute */
#define CM_VADMIC_MASK		0x0e	/* mic gain level (0-3) << 1 */
#define CM_MICGAINZ		0x01	/* mic boost */

#define CM_REG_AUX_VOL		0x26
#define CM_VAUXL_MASK		0xf0
#define CM_VAUXR_MASK		0x0f

#define CM_REG_MISC		0x27
#define CM_XGPO1		0x20
#define CM_XGPBIO		0x04
#define CM_SPDVALID		0x02	/* spdif input valid check */
#define CM_DMAUTO		0x01

#define CM_REG_AC97		0x28	/* hmmm.. do we have ac97 link? */

/*
 * extended registers
 */
#define CM_REG_CH0_FRAME1	0x80	/* base address */
#define CM_REG_CH0_FRAME2	0x84
#define CM_REG_CH1_FRAME1	0x88	/* 0-15: count of samples at bus master; buffer size */
#define CM_REG_CH1_FRAME2	0x8C	/* 16-31: count of samples at codec; fragment size */

/*
 * size of i/o region
 */
#define CM_EXTENT_CODEC	  0x100
#define CM_EXTENT_MIDI	  0x2
#define CM_EXTENT_SYNTH	  0x4

/*
 */

#define CM_OPEN_NONE	0
#define CM_OPEN_CH0	1
#define CM_OPEN_CH1	2
#define CM_OPEN_DAC	0x10
#define CM_OPEN_ADC	0x20
#define CM_OPEN_SPDIF	0x40
#define CM_OPEN_PLAYBACK	(CM_OPEN_CH0 | CM_OPEN_DAC)
#define CM_OPEN_PLAYBACK2	(CM_OPEN_CH1 | CM_OPEN_DAC)
#define CM_OPEN_CAPTURE		(CM_OPEN_CH1 | CM_OPEN_ADC)
#define CM_OPEN_SPDIF_PLAYBACK	(CM_OPEN_CH0 | CM_OPEN_DAC | CM_OPEN_SPDIF)
#define CM_OPEN_SPDIF_CAPTURE	(CM_OPEN_CH1 | CM_OPEN_ADC | CM_OPEN_SPDIF)

typedef struct snd_stru_cmipci cmipci_t;
typedef struct snd_stru_cmipci_pcm cmipci_pcm_t;

struct snd_stru_cmipci_pcm {
	snd_pcm_subchn_t *subchn;
	snd_dma_t *dmaptr;
	int running;		/* dac/adc running? */
	unsigned int dma_size;	/* in frames */
	unsigned int frag_size;	/* in frames */
	unsigned int offset;	/* physical address of the buffer */
	unsigned int fmt;	/* format bits */
	int ch;			/* channel (0/1) */
	int is_dac;		/* is dac? */
	int bytes_per_frame;
	int ac3_shift;	/* extra shift: 1 on soft ac3 mode */
};

struct snd_stru_cmipci {
	snd_card_t *card;

	struct pci_dev *pci;
	snd_irq_t *irqptr;
	snd_irq_t *midi_irqptr;

	unsigned long iobase, iomidi, iosynth;
	unsigned int ctrl;	/* FUNCTRL0 current value */

	snd_pcm_t *pcm;		/* DAC/ADC PCM */
	snd_pcm_t *pcm2;	/* 2nd DAC */
	snd_pcm_t *pcm_spdif;	/* SPDIF */

	int chip_version;
	int max_channels;
	unsigned int has_dual_dac: 1;
	unsigned int can_ac3_sw: 1;
	unsigned int can_ac3_hw: 1;
	unsigned int can_multi_ch: 1;
	int spdif_counter;	/* for software AC3 */

	snd_pcm_hardware_t *hw_info[3]; /* for playbacks */

	int opened[2];	/* open mode */
	struct semaphore open_mutex;

	int pcm_mute_changed: 1; /* to resume after spdif close */
	int pcm_mute_state: 1; /* to resume after spdif close */

	cmipci_pcm_t channel[2];

	snd_pcm_digital_t dig_mask;

	/* mixers */
	snd_kmixer_t *kmixer;

	snd_kmixer_element_t *me_mux_mic;
	snd_kmixer_element_t *me_mux_line;
	snd_kmixer_element_t *me_mux_cd;
	snd_kmixer_element_t *me_mux;
	snd_kmixer_element_t *me_in_accu;
	snd_kmixer_element_t *me_out_accu;
	snd_kmixer_element_t *me_playback;
	snd_kmixer_element_t *me_capture;
	snd_kmixer_element_t *me_in_speaker;
	snd_kmixer_element_t *me_vol_speaker;
	snd_kmixer_element_t *me_in_mic;
	snd_kmixer_element_t *me_vol_mic;
	snd_kmixer_element_t *me_sw1_mic_output;
	snd_kmixer_element_t *me_sw1_mic_input;
	snd_kmixer_element_t *me_mic_gain_vol;
	snd_kmixer_element_t *me_sw1_mic_gain;
	snd_kmixer_element_t *me_in_line;
	snd_kmixer_element_t *me_vol_line;
	snd_kmixer_element_t *me_sw1_line_output;
	snd_kmixer_element_t *me_sw3_line_input;
	snd_kmixer_element_t *me_in_cd;
	snd_kmixer_element_t *me_vol_cd;
	snd_kmixer_element_t *me_sw1_cd_output;
	snd_kmixer_element_t *me_sw3_cd_input;
	snd_kmixer_element_t *me_in_synth;
	snd_kmixer_element_t *me_in_aux;
	snd_kmixer_element_t *me_vol_aux;
	snd_kmixer_element_t *me_sw1_aux_output;
	snd_kmixer_element_t *me_sw1_aux_input;
	snd_kmixer_element_t *me_vol_synth;
	snd_kmixer_element_t *me_sw2_synth;
	snd_kmixer_element_t *me_sw3_synth_input;
	snd_kmixer_element_t *me_vol_pcm;
	snd_kmixer_element_t *me_sw1_pcm_input;
	snd_kmixer_element_t *me_sw2_mute_pcm;
	snd_kmixer_element_t *me_out_master;
	snd_kmixer_element_t *me_sw1_3dse;
	snd_kmixer_element_t *me_vol_master;
	
	/* external MIDI */
	snd_rawmidi_t *rmidi;

	spinlock_t reg_lock;
	snd_info_entry_t *proc_entry;
};


/* read/write operations for dword register */
inline static void snd_cmipci_write(cmipci_t *cm, unsigned int cmd, unsigned int data)
{
	outl(data, cm->iobase + cmd);
}
inline static unsigned int snd_cmipci_read(cmipci_t *cm, unsigned int cmd)
{
	return inl(cm->iobase + cmd);
}

/* read/write operations for word register */
inline static void snd_cmipci_write_w(cmipci_t *cm, unsigned int cmd, unsigned short data)
{
	outw(data, cm->iobase + cmd);
}
inline static unsigned short snd_cmipci_read_w(cmipci_t *cm, unsigned int cmd)
{
	return inw(cm->iobase + cmd);
}

/* read/write operations for byte register */
inline static void snd_cmipci_write_b(cmipci_t *cm, unsigned int cmd, unsigned char data)
{
	outb(data, cm->iobase + cmd);
}

inline static unsigned char snd_cmipci_read_b(cmipci_t *cm, unsigned int cmd)
{
	return inb(cm->iobase + cmd);
}

/* bit operations for dword register */
static void snd_cmipci_set_bit(cmipci_t *cm, unsigned int cmd, unsigned int flag)
{
	unsigned int val;
	val = inl(cm->iobase + cmd);
	val |= flag;
	outl(val, cm->iobase + cmd);
}

static void snd_cmipci_clear_bit(cmipci_t *cm, unsigned int cmd, unsigned int flag)
{
	unsigned int val;
	val = inl(cm->iobase + cmd);
	val &= ~flag;
	outl(val, cm->iobase + cmd);
}

/* bit operations for byte register */
static void snd_cmipci_set_bit_b(cmipci_t *cm, unsigned int cmd, unsigned char flag)
{
	unsigned char val;
	val = inb(cm->iobase + cmd);
	val |= flag;
	outb(val, cm->iobase + cmd);
}

static void snd_cmipci_clear_bit_b(cmipci_t *cm, unsigned int cmd, unsigned char flag)
{
	unsigned char val;
	val = inb(cm->iobase + cmd);
	val &= ~flag;
	outb(val, cm->iobase + cmd);
}


/*
 * PCM interface
 */

/*
 * calculate frequency
 */
static struct {
	unsigned	rate;
	unsigned	lower;
	unsigned	upper;
	unsigned char	freq;
} rate_lookup[] =
{
	{ 5512,		(0 + 5512) / 2,		(5512 + 8000) / 2,	0 },
	{ 8000,		(5512 + 8000) / 2,	(8000 + 11025) / 2,	4 },
	{ 11025,	(8000 + 11025) / 2,	(11025 + 16000) / 2,	1 },
	{ 16000,	(11025 + 16000) / 2,	(16000 + 22050) / 2,	5 },
	{ 22050,	(16000 + 22050) / 2,	(22050 + 32000) / 2,	2 },
	{ 32000,	(22050 + 32000) / 2,	(32000 + 44100) / 2,	6 },
	{ 44100,	(32000 + 44100) / 2,	(44100 + 48000) / 2,	3 },
	{ 48000,	(44100 + 48000) /2,	48000,			7 }
};

static int freq_tbl[] = { 5512, 11025, 22050, 44100, 8000, 16000, 32000, 48000 };

static unsigned int snd_cmipci_fixed_rate(unsigned int rate)
{
	int i, freq = 4;

	if (rate > 48000)
		rate = 48000;
	if (rate < 5512)
		rate = 5512;
	for (i = 0; i < sizeof(rate_lookup) / sizeof(rate_lookup[0]); i++) {
		if (rate > rate_lookup[i].lower && rate <= rate_lookup[i].upper) {
			freq = rate_lookup[i].freq;
			break;
	    	}
	}
	return freq;
}

/*
 */
static int set_dac_channels(cmipci_t *cm, cmipci_pcm_t *rec, int channels)
{
	unsigned long flags;

	if (channels > 2) {
		if (! cm->can_multi_ch)
			return -EINVAL;
		if (channels >= cm->max_channels || channels == 3)
			return -EINVAL;
		if (rec->fmt != 0x03) /* stereo 16bit only */
			return -EINVAL;

		spin_lock_irqsave(&cm->reg_lock, flags);
		snd_cmipci_set_bit(cm, CM_REG_LEGACY_CTRL, CM_NXCHG);
		if (channels > 4) {
			snd_cmipci_clear_bit(cm, CM_REG_CHFORMAT, CM_CHB3D);
			snd_cmipci_set_bit(cm, CM_REG_CHFORMAT, CM_CHB3D5C);
		} else {
			snd_cmipci_set_bit(cm, CM_REG_CHFORMAT, CM_CHB3D);
			snd_cmipci_clear_bit(cm, CM_REG_CHFORMAT, CM_CHB3D5C);
		}
		if (channels == 6) {
			snd_cmipci_set_bit(cm, CM_REG_LEGACY_CTRL, CM_CHB3D6C);
			snd_cmipci_set_bit(cm, CM_REG_MISC_CTRL, CM_ENCENTER);
		} else {
			snd_cmipci_clear_bit(cm, CM_REG_LEGACY_CTRL, CM_CHB3D6C);
			snd_cmipci_clear_bit(cm, CM_REG_MISC_CTRL, CM_ENCENTER);
		}
		spin_unlock_irqrestore(&cm->reg_lock, flags);

	} else {
		if (cm->can_multi_ch) {
			spin_lock_irqsave(&cm->reg_lock, flags);
			snd_cmipci_clear_bit(cm, CM_REG_CHFORMAT, CM_CHB3D);
			snd_cmipci_clear_bit(cm, CM_REG_CHFORMAT, CM_CHB3D5C);
			snd_cmipci_clear_bit(cm, CM_REG_LEGACY_CTRL, CM_CHB3D6C);
			snd_cmipci_clear_bit(cm, CM_REG_MISC_CTRL, CM_ENCENTER);
			spin_unlock_irqrestore(&cm->reg_lock, flags);
		}
	}
	return 0;
}


/*
 * prepare playback/capture channel
 * channel to be used must have been set in rec->ch.
 */
static int snd_cmipci_pcm_prepare(cmipci_t *cm, cmipci_pcm_t *rec,
				  snd_pcm_subchn_t *subchn)
{
	unsigned long flags;
	unsigned int reg, freq, val;
	snd_pcm_runtime_t *runtime = subchn->runtime;

	rec->fmt = 0;
	rec->bytes_per_frame = 1;
	if (snd_pcm_format_width(runtime->format.format) >= 16) {
		rec->fmt |= 0x02;
		rec->bytes_per_frame <<= 1;
	}
	if (runtime->format.voices > 1) {
		rec->fmt |= 0x01;
		/* FIXME: is this correct? */
		rec->bytes_per_frame *= runtime->format.voices;
	}
	if (rec->is_dac && set_dac_channels(cm, rec, runtime->format.voices) < 0)
		return -EINVAL;

	rec->offset = virt_to_bus(runtime->dma_area->buf);
	/* buffer and fragment sizes in frame */
	rec->dma_size = snd_pcm_lib_transfer_size(subchn) / rec->bytes_per_frame;
	rec->frag_size = snd_pcm_lib_transfer_fragment(subchn) / rec->bytes_per_frame;
	rec->frag_size <<= rec->ac3_shift;
	/* sanity check.. */
	if ((rec->dma_size % rec->frag_size) != 0 ||
	    rec->dma_size / rec->frag_size < 2)
		return -EINVAL;

	spin_lock_irqsave(&cm->reg_lock, flags);

	/* set buffer address */
	reg = rec->ch ? CM_REG_CH1_FRAME1 : CM_REG_CH0_FRAME1;
	snd_cmipci_write(cm, reg, rec->offset);
	/* program sample counts */
	reg = rec->ch ? CM_REG_CH1_FRAME2 : CM_REG_CH0_FRAME2;
	snd_cmipci_write_w(cm, reg, rec->dma_size - 1);
	snd_cmipci_write_w(cm, reg + 2, rec->frag_size - 1);

	/* set adc/dac flag */
	val = rec->ch ? CM_CHADC1 : CM_CHADC0;
	if (rec->is_dac)
		cm->ctrl &= ~val;
	else
		cm->ctrl |= val;
	snd_cmipci_write(cm, CM_REG_FUNCTRL0, cm->ctrl);
	//snd_printd("cmipci: functrl0 = %08x\n", cm->ctrl);

	/* set sample rate */
	freq = snd_cmipci_fixed_rate(runtime->format.rate);
	val = snd_cmipci_read(cm, CM_REG_FUNCTRL1);
	if (rec->ch) {
		val &= ~CM_ASFC_MASK;
		val |= (freq << CM_ASFC_SHIFT) & CM_ASFC_MASK;
	} else {
		val &= ~CM_DSFC_MASK;
		val |= (freq << CM_DSFC_SHIFT) & CM_DSFC_MASK;
	}
	runtime->format.rate = freq_tbl[freq];
	snd_cmipci_write(cm, CM_REG_FUNCTRL1, val);
	//snd_printd("cmipci: functrl1 = %08x\n", val);

	/* set format */
	val = snd_cmipci_read(cm, CM_REG_CHFORMAT);
	if (rec->ch) {
		val &= ~CM_CH1FMT_MASK;
		val |= rec->fmt << CM_CH1FMT_SHIFT;
	} else {
		val &= ~CM_CH0FMT_MASK;
		val |= rec->fmt << CM_CH0FMT_SHIFT;
	}
	snd_cmipci_write(cm, CM_REG_CHFORMAT, val);
	//snd_printd("cmipci: chformat = %08x\n", val);

	rec->running = 0;
	spin_unlock_irqrestore(&cm->reg_lock, flags);

	return 0;
}

/*
 * ioctl - change rate
 */
static int snd_cmipci_pcm_ioctl(cmipci_t *cm, cmipci_pcm_t *rec,
				snd_pcm_subchn_t *subchn,
				unsigned int cmd, unsigned long *arg)
{

	if (cmd == SND_PCM_IOCTL1_PARAMS) {
		int rate = snd_cmipci_fixed_rate(subchn->runtime->format.rate);
		subchn->runtime->format.rate = freq_tbl[rate];
	}
	return 0;
}

/*
 * PCM trigger/stop
 */
static int snd_cmipci_pcm_trigger(cmipci_t *cm, cmipci_pcm_t *rec,
				 snd_pcm_subchn_t *subchn, int cmd)
{
	unsigned long flags;
	unsigned int inthld, chen, reset, pause;

	if (cmd == SND_PCM_TRIGGER_SYNC_GO)
		return -EINVAL;

	inthld = CM_CH0_INT_EN << rec->ch;
	chen = CM_CHEN0 << rec->ch;
	reset = CM_RST_CH0 << rec->ch;
	pause = CM_PAUSE0 << rec->ch;

	spin_lock_irqsave(&cm->reg_lock, flags);
	switch (cmd) {
	case SND_PCM_TRIGGER_GO:
		rec->running = 1;
		/* set interrupt */
		snd_cmipci_set_bit(cm, CM_REG_INT_HLDCLR, inthld);
		cm->ctrl |= chen;
		/* enable channel */
		snd_cmipci_write(cm, CM_REG_FUNCTRL0, cm->ctrl);
		//snd_printd("cmipci: functrl0 = %08x\n", cm->ctrl);
		break;
	case SND_PCM_TRIGGER_STOP:
		rec->running = 0;
		/* disable interrupt */
		snd_cmipci_clear_bit(cm, CM_REG_INT_HLDCLR, inthld);
		/* reset */
		cm->ctrl &= ~chen;
		snd_cmipci_write(cm, CM_REG_FUNCTRL0, cm->ctrl | reset);
		snd_cmipci_write(cm, CM_REG_FUNCTRL0, cm->ctrl);
		break;
	case SND_PCM_TRIGGER_PAUSE_PUSH:
		cm->ctrl |= pause;
		snd_cmipci_write(cm, CM_REG_FUNCTRL0, cm->ctrl);
		break;
	case SND_PCM_TRIGGER_PAUSE_RELEASE:
		cm->ctrl &= ~pause;
		snd_cmipci_write(cm, CM_REG_FUNCTRL0, cm->ctrl);
		break;
	}
	spin_unlock_irqrestore(&cm->reg_lock, flags);
	return 0;
}

/*
 * return the current pointer
 */
static unsigned int snd_cmipci_pcm_pointer(cmipci_t *cm, cmipci_pcm_t *rec,
					  snd_pcm_subchn_t *subchn)
{
	unsigned long flags;
	unsigned int ptr, reg;
	if (! rec->running)
		return 0;
	spin_lock_irqsave(&cm->reg_lock, flags);
#if 1
	reg = rec->ch ? CM_REG_CH1_FRAME2 : CM_REG_CH0_FRAME2;
	ptr = rec->dma_size - (snd_cmipci_read_w(cm, reg) + 1);
	ptr *= rec->bytes_per_frame; /* convert to bytes */
#else
	reg = rec->ch ? CM_REG_CH1_FRAME1 : CM_REG_CH0_FRAME1;
	ptr = snd_cmipci_read(cm, reg) - rec->offset;
#endif
	ptr >>= rec->ac3_shift;
	spin_unlock_irqrestore(&cm->reg_lock, flags);
	return ptr;
}

/*
 * playback
 */

static int snd_cmipci_playback_prepare(void *private_data,
				      snd_pcm_subchn_t *subchn)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, private_data, -ENXIO);
	return snd_cmipci_pcm_prepare(cm, &cm->channel[0], subchn);
}

static int snd_cmipci_playback_ioctl(void *private_data,
				    snd_pcm_subchn_t *subchn,
				    unsigned int cmd,
				    unsigned long *arg)
{
	int result;
	cmipci_t *cm = snd_magic_cast(cmipci_t, private_data, -ENXIO);

	result = snd_pcm_lib_ioctl(private_data, subchn, cmd, arg);
	if (result < 0)
		return result;
	return snd_cmipci_pcm_ioctl(cm, &cm->channel[0], subchn, cmd, arg);
}

static int snd_cmipci_playback_trigger(void *private_data,
				      snd_pcm_subchn_t *subchn,
				      int cmd)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, private_data, -ENXIO);
	return snd_cmipci_pcm_trigger(cm, &cm->channel[0], subchn, cmd);
}

static unsigned int snd_cmipci_playback_pointer(void *private_data,
					       snd_pcm_subchn_t *subchn)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, private_data, -ENXIO);
	return snd_cmipci_pcm_pointer(cm, &cm->channel[0], subchn);
}



/*
 * capture
 */

static int snd_cmipci_capture_prepare(void *private_data,
				      snd_pcm_subchn_t *subchn)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, private_data, -ENXIO);
	return snd_cmipci_pcm_prepare(cm, &cm->channel[1], subchn);
}

static int snd_cmipci_capture_ioctl(void *private_data,
				   snd_pcm_subchn_t *subchn,
				   unsigned int cmd,
				   unsigned long *arg)
{
	int result;
	cmipci_t *cm = snd_magic_cast(cmipci_t, private_data, -ENXIO);

	result = snd_pcm_lib_ioctl(private_data, subchn, cmd, arg);
	if (result < 0)
		return result;
	return snd_cmipci_pcm_ioctl(cm, &cm->channel[1], subchn, cmd, arg);
}

static int snd_cmipci_capture_trigger(void *private_data,
				     snd_pcm_subchn_t *subchn,
				     int cmd)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, private_data, -ENXIO);
	return snd_cmipci_pcm_trigger(cm, &cm->channel[1], subchn, cmd);
}

static unsigned int snd_cmipci_capture_pointer(void *private_data,
					      snd_pcm_subchn_t *subchn)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, private_data, -ENXIO);
	return snd_cmipci_pcm_pointer(cm, &cm->channel[1], subchn);
}


/*
 * special tricks for software ac3 transfer:
 * expand two 16bit data to two 32bit data.
 */

/* find parity for bit 4~30 */
static unsigned parity(unsigned int data)
{
	unsigned int parity = 0;
	int counter = 4;

	data >>= 4;	// start from bit 4
	while (counter <= 30) {
		if (data & 1)
			parity++;
		data >>= 1;
		counter++;
	}
	return parity & 1;
}


inline static u32 convert_ac3_32bit(cmipci_t *cm, u32 val)
{
	u32 data = (u32)val << 12;

	if (cm->spdif_counter == 2 || cm->spdif_counter == 3)
		data |= 0x40000000;	/* indicate AC-3 raw data */
	if (parity(data))		/* parity bit 4-30 */
		data |= 0x80000000;
	if (cm->spdif_counter == 0)
		data |= 3;		/* preamble 'M' */
	else if (cm->spdif_counter & 1)
		data |= 5;		/* odd, 'W' */
	else
		data |= 9;		/* even, 'M' */

	cm->spdif_counter++;
	if (cm->spdif_counter == 384)
		cm->spdif_counter = 0;

	return data;
}

static int snd_cmipci_ac3_memcpy(snd_pcm_subchn_t *subchn, int voice, int pos,
				 void *src, int count)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, subchn->pcm->private_data, -ENXIO);
	u32 *dst;
	u16 *srcp = src, val;
	int offset;

	if (! access_ok(VERIFY_READ, src, count))
		return -EFAULT;

	/* frame = 16bit stereo */
	offset = (pos << 1) % (cm->channel[0].dma_size << 2);
	dst = (u32*)((char*)subchn->runtime->dma_area->buf + offset);

	count /= 2;
	while (count-- > 0) {
		get_user(val, srcp);
		srcp++;
		*dst++ = convert_ac3_32bit(cm, val);
	}

	return 0;
}

static int snd_cmipci_ac3_memset(snd_pcm_subchn_t *subchn, int voice, int pos,
				 int c, int count)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, subchn->pcm->private_data, -ENXIO);
	u32 *dst;
	int offset;

	/* frame = 16bit stereo */
	offset = (pos << 1) % (cm->channel[0].dma_size << 2);
	dst = (u32*)((char*)subchn->runtime->dma_area->buf + offset);

	count /= 2;
	while (count-- > 0) {
		*dst++ = convert_ac3_32bit(cm, c);
	}

	return 0;
}


/*
 * spdif prepare
 */
static int snd_cmipci_playback_spdif_prepare(void *private_data,
					     snd_pcm_subchn_t *subchn)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, private_data, -ENXIO);
	snd_pcm_runtime_t *runtime = subchn->runtime;
	unsigned long flags;
	int err, do_ac3;

	if ((runtime->format.format != SND_PCM_SFMT_S16_LE &&
	     runtime->format.format != SND_PCM_SFMT_S24_LE) ||
	    runtime->format.voices != 2 ||
	    (runtime->format.rate != 44100 && runtime->format.rate != 48000))
		return -EINVAL;
	if (! runtime->dig_mask)
		return -EINVAL;
	if (! runtime->dig_mask->dig_valid)
		return -EINVAL;

	do_ac3 = (runtime->dig_mask && runtime->dig_mask->dig_valid &&
		  (runtime->dig_mask->dig_status[0] & SND_PCM_DIG0_NONAUDIO));

	spin_lock_irqsave(&cm->reg_lock, flags);

	if (do_ac3) {
		/* mute pcm analog output */
		/* remember the mute state to resume after spdif close.
		 * FIXME: association with mixer.
		 */
		cm->pcm_mute_changed = 1;
		cm->pcm_mute_state = snd_cmipci_read_b(cm, CM_REG_MIXER1) & CM_WSMUTE ? 1 : 0;
		/* to resume after spdif close */
		snd_cmipci_set_bit_b(cm, CM_REG_MIXER1, CM_WSMUTE);
	} else
		cm->pcm_mute_changed = 0;
	/* snd_cmipci_set_bit(cm, CM_REG_LEGACY_CTRL, CM_ENSPDOUT); */
	snd_cmipci_set_bit(cm, CM_REG_FUNCTRL1, CM_SPDF_0);

	runtime->hw_memcpy = NULL;
	runtime->hw_memset = NULL;
	cm->channel[0].ac3_shift = 0;
	cm->spdif_counter = 0;

	if (do_ac3) {

		/* AC3EN for 037 */
		snd_cmipci_set_bit(cm, CM_REG_CHFORMAT, CM_AC3EN1);
		/* AC3EN for 039 */
		snd_cmipci_set_bit(cm, CM_REG_MISC_CTRL, CM_AC3EN2);
	
		if (cm->can_ac3_hw) {
			/* SPD24SEL for 037, 0x02 */
			/* SPD24SEL for 039, 0x20, but cannot be set */
			snd_cmipci_set_bit(cm, CM_REG_CHFORMAT, CM_SPD24SEL);
		} else {
			runtime->hw_memcpy = snd_cmipci_ac3_memcpy;
			runtime->hw_memset = snd_cmipci_ac3_memset;

			/* SPD32SEL for 037 & 039, 0x20 */
			snd_cmipci_set_bit(cm, CM_REG_MISC_CTRL, CM_SPD32SEL);
			/* set 176K sample rate to fix 033 HW bug */
			if (cm->chip_version == 33) {
				if (runtime->format.rate == 48000)
					snd_cmipci_set_bit(cm, CM_REG_CHFORMAT, CM_CH1_SRATE_176K);
				else
					snd_cmipci_clear_bit(cm, CM_REG_CHFORMAT, CM_CH1_SRATE_176K);
			}
			cm->channel[0].ac3_shift = 1; /* convert 16->32bit */
		}

	} else {
		snd_cmipci_clear_bit(cm, CM_REG_CHFORMAT, CM_AC3EN1);
		snd_cmipci_clear_bit(cm, CM_REG_MISC_CTRL, CM_AC3EN2);

		if (cm->can_ac3_hw) {
			if (runtime->format.format == SND_PCM_SFMT_S24_LE)
				snd_cmipci_set_bit(cm, CM_REG_CHFORMAT, CM_SPD24SEL);
			else
				snd_cmipci_clear_bit(cm, CM_REG_CHFORMAT, CM_SPD24SEL);
		} else {
			snd_cmipci_clear_bit(cm, CM_REG_MISC_CTRL, CM_SPD32SEL);
			snd_cmipci_clear_bit(cm, CM_REG_CHFORMAT, CM_CH1_SRATE_176K);
		}
	}

	spin_unlock_irqrestore(&cm->reg_lock, flags);

	err = snd_cmipci_pcm_prepare(cm, &cm->channel[0], subchn);

	spin_lock_irqsave(&cm->reg_lock, flags);
	/* it seems that this must be here */
	if (runtime->format.rate == 48000)
		snd_cmipci_set_bit(cm, CM_REG_MISC_CTRL, CM_SPDF_AC97 | CM_SPDIF48K);
	else
		snd_cmipci_clear_bit(cm, CM_REG_MISC_CTRL, CM_SPDF_AC97 | CM_SPDIF48K);
	spin_unlock_irqrestore(&cm->reg_lock, flags);

	return err;
}

static int snd_cmipci_capture_spdif_prepare(void *private_data,
					    snd_pcm_subchn_t *subchn)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, private_data, -ENXIO);
	snd_pcm_runtime_t *runtime = subchn->runtime;

	if (runtime->format.format != SND_PCM_SFMT_S16_LE ||
	    runtime->format.voices != 2 ||
	    (runtime->format.rate != 44100 && runtime->format.rate != 48000))
		return -EINVAL;

#if 0 /* these bits are only for playback?? */
	if (runtime->format.rate == 48000)
		snd_cmipci_set_bit(cm, CM_REG_MISC_CTRL, CM_SPDIF48K | CM_SPDF_AC97);
	else
		snd_cmipci_clear_bit(cm, CM_REG_MISC_CTRL, CM_SPDIF48K | CM_SPDF_AC97);
#endif

	snd_cmipci_set_bit(cm, CM_REG_FUNCTRL1, CM_SPDF_1);

	return snd_cmipci_pcm_prepare(cm, &cm->channel[1], subchn);
}

/*
 * both for playback/capture
 */
static int snd_cmipci_spdif_ioctl(void *private_data,
				  snd_pcm_subchn_t *subchn,
				  unsigned int cmd,
				  unsigned long *arg)
{
	int result;

	result = snd_pcm_lib_ioctl(private_data, subchn, cmd, arg);
	if (result < 0)
		return result;
	if (cmd == SND_PCM_IOCTL1_PARAMS) {
		if (subchn->runtime->format.rate >= 48000)
			subchn->runtime->format.rate = 48000;
		else
			subchn->runtime->format.rate = 44100;
	}
	return 0;
}


/*
 * interrupt handler
 */
static void snd_cmipci_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, dev_id,);
	unsigned int status;
	
	/* fastpath out, to ease interrupt sharing */
	status = snd_cmipci_read(cm, CM_REG_INT_STATUS);
	if (!(status & CM_INTR))
		return;

	/* acknowledge interrupt */
	spin_lock(&cm->reg_lock);
	if (status & CM_CHINT0) {
		snd_cmipci_clear_bit(cm, CM_REG_INT_HLDCLR, CM_CH0_INT_EN);
		snd_cmipci_set_bit(cm, CM_REG_INT_HLDCLR, CM_CH0_INT_EN);
	}
	if (status & CM_CHINT1) {
		snd_cmipci_clear_bit(cm, CM_REG_INT_HLDCLR, CM_CH1_INT_EN);
		snd_cmipci_set_bit(cm, CM_REG_INT_HLDCLR, CM_CH1_INT_EN);
	}
	spin_unlock(&cm->reg_lock);

	if (cm->rmidi) {
		if (status & CM_UARTINT) {
			snd_mpu401_uart_interrupt(cm->rmidi);
		}
	}

	if (cm->pcm) {
		if ((status & CM_CHINT0) && cm->channel[0].running)
			snd_pcm_transfer_done(cm->channel[0].subchn);
		if ((status & CM_CHINT1) && cm->channel[1].running)
			snd_pcm_transfer_done(cm->channel[1].subchn);
	}
}

/*
 * for external MIDI (only when snd_mpu_irq options is given)
 */
static void snd_cmipci_midi_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, dev_id,);
	if (cm->rmidi)
		snd_mpu401_uart_interrupt(cm->rmidi);
}

/*
 */
static snd_pcm_hardware_t snd_cmipci_playback =
{
	chninfo:	(SND_PCM_CHNINFO_MMAP | SND_PCM_CHNINFO_STREAM |
			 SND_PCM_CHNINFO_BLOCK | SND_PCM_CHNINFO_INTERLEAVE |
			 SND_PCM_CHNINFO_BLOCK_TRANSFER |
			 SND_PCM_CHNINFO_MMAP_VALID |
			 SND_PCM_CHNINFO_PAUSE),
	formats:	SND_PCM_FMT_U8 | SND_PCM_FMT_S16_LE,
	rates:		SND_PCM_RATE_KNOT | SND_PCM_RATE_8000_48000,
	min_rate:	5512,
	max_rate:	48000,
	min_voices:	1,
	max_voices:	2,
	min_fragment_size:	64,
	max_fragment_size:	(32*1024),
	fragment_align:	31,
	fifo_size:	0,
	transfer_block_size:	4,
	ioctl:		snd_cmipci_playback_ioctl,
	prepare:	snd_cmipci_playback_prepare,
	trigger:	snd_cmipci_playback_trigger,
	pointer:	snd_cmipci_playback_pointer,
};

static snd_pcm_hardware_t snd_cmipci_capture =
{
	chninfo:	(SND_PCM_CHNINFO_MMAP | SND_PCM_CHNINFO_STREAM |
			 SND_PCM_CHNINFO_BLOCK | SND_PCM_CHNINFO_INTERLEAVE |
			 SND_PCM_CHNINFO_BLOCK_TRANSFER |
			 SND_PCM_CHNINFO_MMAP_VALID |
			 SND_PCM_CHNINFO_PAUSE),
	formats:	SND_PCM_FMT_U8 | SND_PCM_FMT_S16_LE,
	rates:		SND_PCM_RATE_KNOT | SND_PCM_RATE_8000_48000,
	min_rate:	5512,
	max_rate:	48000,
	min_voices:	1,
	max_voices:	2,
	min_fragment_size:	64,
	max_fragment_size:	(32*1024),
	fragment_align:	31,
	fifo_size:	0,
	transfer_block_size:	4,
	ioctl:		snd_cmipci_capture_ioctl,
	prepare:	snd_cmipci_capture_prepare,
	trigger:	snd_cmipci_capture_trigger,
	pointer:	snd_cmipci_capture_pointer,
};

static snd_pcm_hardware_t snd_cmipci_playback_spdif =
{
	chninfo:	(SND_PCM_CHNINFO_BLOCK | SND_PCM_CHNINFO_INTERLEAVE |
			 SND_PCM_CHNINFO_BLOCK_TRANSFER |
			 SND_PCM_CHNINFO_PAUSE),
	formats:	SND_PCM_FMT_S16_LE,
	rates:		SND_PCM_RATE_44100 | SND_PCM_RATE_48000,
	min_rate:	44100,
	max_rate:	48000,
	min_voices:	2,
	max_voices:	2,
	min_fragment_size:	64,
	max_fragment_size:	(32*1024),
	fragment_align:	31,
	fifo_size:	0,
	transfer_block_size:	4,
	ioctl:		snd_cmipci_spdif_ioctl,
	prepare:	snd_cmipci_playback_spdif_prepare,
	trigger:	snd_cmipci_playback_trigger,
	pointer:	snd_cmipci_playback_pointer,
};

static snd_pcm_hardware_t snd_cmipci_capture_spdif =
{
	chninfo:	(SND_PCM_CHNINFO_MMAP | SND_PCM_CHNINFO_STREAM |
			 SND_PCM_CHNINFO_BLOCK | SND_PCM_CHNINFO_INTERLEAVE |
			 SND_PCM_CHNINFO_BLOCK_TRANSFER |
			 SND_PCM_CHNINFO_MMAP_VALID |
			 SND_PCM_CHNINFO_PAUSE),
	formats:	SND_PCM_FMT_S16_LE,
	rates:		SND_PCM_RATE_44100 | SND_PCM_RATE_48000,
	min_rate:	44100,
	max_rate:	48000,
	min_voices:	2,
	max_voices:	2,
	min_fragment_size:	64,
	max_fragment_size:	(32*1024),
	fragment_align:	31,
	fifo_size:	0,
	transfer_block_size:	4,
	ioctl:		snd_cmipci_spdif_ioctl,
	prepare:	snd_cmipci_capture_spdif_prepare,
	trigger:	snd_cmipci_capture_trigger,
	pointer:	snd_cmipci_capture_pointer,
};


/*
 * check device open/close
 */
static int open_device_check(cmipci_t *cm, int mode, snd_pcm_subchn_t *subs,
			     char *dma_str)
{
	unsigned long flags;
	int ch = (mode & CM_OPEN_CH0) ? 0 : 1;
	int err;

	/* FIXME: a file should wait until the device becomes free
	 * when it's opened on blocking mode.  however, since the current
	 * pcm framework doesn't pass file pointer before actually opened,
	 * we can't know whether blocking mode or not in open callback..
	 */
	down(&cm->open_mutex);
	if (cm->opened[ch]) {
		up(&cm->open_mutex);
		return -EBUSY;
	}
	if ((err = snd_pcm_dma_alloc(subs, cm->channel[ch].dmaptr, dma_str)) < 0) {
		up(&cm->open_mutex);
		return err;
	}
	cm->opened[ch] = mode;
	cm->channel[ch].subchn = subs;
	if (mode & CM_OPEN_DAC) {
		if (! cm->channel[ch].is_dac) {
			cm->channel[ch].is_dac = 1;
			spin_lock_irqsave(&cm->reg_lock, flags);
			snd_cmipci_set_bit(cm, CM_REG_MISC_CTRL, CM_ENDBDAC);
			spin_unlock_irqrestore(&cm->reg_lock, flags);
		}
	} else {
		if (cm->channel[ch].is_dac) {
			cm->channel[ch].is_dac = 1;
			spin_lock_irqsave(&cm->reg_lock, flags);
			snd_cmipci_clear_bit(cm, CM_REG_MISC_CTRL, CM_ENDBDAC);
			spin_unlock_irqrestore(&cm->reg_lock, flags);
		}
	}
	up(&cm->open_mutex);
	return 0;
}

static void close_device_check(cmipci_t *cm, int mode)
{
	int ch = (mode & CM_OPEN_CH0) ? 0 : 1;

	down(&cm->open_mutex);
	if (cm->opened[ch] == mode) {
		if (cm->channel[ch].subchn)
			snd_pcm_dma_free(cm->channel[ch].subchn);
		cm->channel[ch].running = 0;
		cm->channel[ch].ac3_shift = 0;
		cm->channel[ch].subchn = NULL;
		cm->opened[ch] = 0;
	}
	up(&cm->open_mutex);
}

/*
 */
static int snd_cmipci_playback_open(void *private_data,
				    snd_pcm_subchn_t *subchn)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, private_data, -ENXIO);
	int err;

	if ((err = open_device_check(cm, CM_OPEN_PLAYBACK, subchn,
				     "C-Media PCI - DAC")) < 0)
		return err;
	subchn->runtime->hw = cm->hw_info[0];
	snd_pcm_set_mixer(subchn, cm->kmixer->device, cm->me_playback);
	snd_pcm_set_sync(subchn);

	return 0;
}

static int snd_cmipci_capture_open(void *private_data,
				   snd_pcm_subchn_t * subchn)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, private_data, -ENXIO);
	int err;

	if ((err = open_device_check(cm, CM_OPEN_CAPTURE, subchn,
				     "C-Media PCI - ADC")) < 0)
		return err;
	subchn->runtime->hw = &snd_cmipci_capture;
	snd_pcm_set_mixer(subchn, cm->kmixer->device, cm->me_capture);
	snd_pcm_set_sync(subchn);

	return 0;
}

static int snd_cmipci_playback2_open(void *private_data,
				     snd_pcm_subchn_t * subchn)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, private_data, -ENXIO);
	int err;

	if ((err = open_device_check(cm, CM_OPEN_PLAYBACK2, subchn,
				     "C-Media PCI - DAC2")) < 0)
		return err;
	subchn->runtime->hw = cm->hw_info[1];
	snd_pcm_set_mixer(subchn, cm->kmixer->device, cm->me_playback);
	snd_pcm_set_sync(subchn);

	return 0;
}

static int snd_cmipci_playback_spdif_open(void *private_data,
					  snd_pcm_subchn_t *subchn)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, private_data, -ENXIO);
	snd_pcm_runtime_t *runtime = subchn->runtime;
	int err;

	if ((err = open_device_check(cm, CM_OPEN_SPDIF_PLAYBACK, subchn,
				     "C-Media PCI - SPDIF Out")) < 0)
		return err;

	runtime->dig_mask = &cm->dig_mask;
	runtime->dig_mask_free = NULL;
	runtime->dig_mask->dig_valid = 1;
	runtime->dig_mask->dig_status[0] = SND_PCM_DIG0_PROFESSIONAL | SND_PCM_DIG0_PRO_FS_44100;
	runtime->dig_mask->dig_status[1] = 0;
	runtime->dig_mask->dig_status[2] = 0;
	runtime->dig_mask->dig_status[3] = SND_PCM_DIG3_CON_FS_44100;

	subchn->runtime->hw = cm->hw_info[2];
	snd_pcm_set_sync(subchn);

	return 0;
}

static int snd_cmipci_capture_spdif_open(void *private_data,
					 snd_pcm_subchn_t *subchn)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, private_data, -ENXIO);
	int err;

	if ((err = open_device_check(cm, CM_OPEN_SPDIF_CAPTURE, subchn,
				     "C-Media PCI - SPDIF In")) < 0)
		return err;

	subchn->runtime->hw = &snd_cmipci_capture_spdif;
	snd_pcm_set_sync(subchn);

	return 0;
}


/*
 */

static int snd_cmipci_playback_close(void *private_data,
				    snd_pcm_subchn_t * subchn)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, private_data, -ENXIO);
	close_device_check(cm, CM_OPEN_PLAYBACK);
	return 0;
}

static int snd_cmipci_capture_close(void *private_data,
				   snd_pcm_subchn_t * subchn)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, private_data, -ENXIO);
	close_device_check(cm, CM_OPEN_CAPTURE);
	return 0;
}

static int snd_cmipci_playback2_close(void *private_data,
				      snd_pcm_subchn_t * subchn)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, private_data, -ENXIO);
	close_device_check(cm, CM_OPEN_PLAYBACK2);
	return 0;
}

static int snd_cmipci_playback_spdif_close(void *private_data,
					   snd_pcm_subchn_t *subchn)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, private_data, -ENXIO);
	unsigned long flags;

	spin_lock_irqsave(&cm->reg_lock, flags);
	/* snd_cmipci_clear_bit(cm, CM_REG_LEGACY_CTRL, CM_ENSPDOUT); */
	snd_cmipci_clear_bit(cm, CM_REG_FUNCTRL1, CM_SPDF_0);
	/* resume pcm mute button */
	if (cm->pcm_mute_changed && ! cm->pcm_mute_state)
		snd_cmipci_clear_bit_b(cm, CM_REG_MIXER1, CM_WSMUTE);
	spin_unlock_irqrestore(&cm->reg_lock, flags);

	close_device_check(cm, CM_OPEN_SPDIF_PLAYBACK);

	return 0;
}

static int snd_cmipci_capture_spdif_close(void *private_data,
					  snd_pcm_subchn_t *subchn)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, private_data, -ENXIO);
	unsigned long flags;

	spin_lock_irqsave(&cm->reg_lock, flags);
	snd_cmipci_clear_bit(cm, CM_REG_FUNCTRL1, CM_SPDF_1);
	spin_unlock_irqrestore(&cm->reg_lock, flags);

	close_device_check(cm, CM_OPEN_SPDIF_CAPTURE);

	return 0;
}

/*
 */

static int __init snd_cmipci_pcm_new(cmipci_t *cm, int device)
{
	snd_pcm_t *pcm;
	snd_pcm_hardware_t *hw;
	int err;

	err = snd_pcm_new(cm->card, cm->card->shortname, device, 1, 1, &pcm);
	if (err < 0)
		return err;

	pcm->chn[SND_PCM_CHANNEL_PLAYBACK].private_data = cm;
	pcm->chn[SND_PCM_CHANNEL_PLAYBACK].open = snd_cmipci_playback_open;
	pcm->chn[SND_PCM_CHANNEL_PLAYBACK].close = snd_cmipci_playback_close;

	pcm->chn[SND_PCM_CHANNEL_CAPTURE].private_data = cm;
	pcm->chn[SND_PCM_CHANNEL_CAPTURE].open = snd_cmipci_capture_open;
	pcm->chn[SND_PCM_CHANNEL_CAPTURE].close = snd_cmipci_capture_close;

	pcm->private_data = cm;
	pcm->info_flags = SND_PCM_INFO_PLAYBACK | SND_PCM_INFO_CAPTURE |
			  SND_PCM_INFO_DUPLEX;
	strcpy(pcm->name, "C-Media PCI DAC/ADC");
	cm->pcm = pcm;

	cm->channel[0].ch = 0;
	cm->channel[0].is_dac = 1;
	cm->channel[1].ch = 1;
	cm->channel[1].is_dac = 0;

	hw = cm->hw_info[0] = snd_kmalloc(sizeof(snd_pcm_hardware_t), GFP_KERNEL);
	if (! hw)
		return -ENOMEM;
	*hw = snd_cmipci_playback;
	if (cm->can_multi_ch)
		hw->max_voices = cm->max_channels;

	return 0;
}

/* second DAC */
static int __init snd_cmipci_pcm2_new(cmipci_t *cm, int device)
{
	snd_pcm_t *pcm;
	snd_pcm_hardware_t *hw;
	int err;

	err = snd_pcm_new(cm->card, cm->card->shortname, device, 1, 0, &pcm);
	if (err < 0)
		return err;

	pcm->chn[SND_PCM_CHANNEL_PLAYBACK].private_data = cm;
	pcm->chn[SND_PCM_CHANNEL_PLAYBACK].open = snd_cmipci_playback2_open;
	pcm->chn[SND_PCM_CHANNEL_PLAYBACK].close = snd_cmipci_playback2_close;

	pcm->private_data = cm;
	pcm->info_flags = SND_PCM_INFO_PLAYBACK;
	strcpy(pcm->name, "C-Media PCI 2nd DAC");
	cm->pcm2 = pcm;

	hw = cm->hw_info[1] = snd_kmalloc(sizeof(snd_pcm_hardware_t), GFP_KERNEL);
	if (! hw)
		return -ENOMEM;
	*hw = snd_cmipci_capture; /* use the ADC as 2nd DAC */
	/* stereo 16bit only?? */
	hw->min_voices = hw->max_voices = 2;
	hw->formats = SND_PCM_FMT_S16_LE;

	return 0;
}

/* spdif i/o */
static int __init snd_cmipci_pcm_spdif_new(cmipci_t *cm, int device)
{
	snd_pcm_t *pcm;
	snd_pcm_hardware_t *hw;
	int err;

	err = snd_pcm_new(cm->card, cm->card->shortname, device, 1, 1, &pcm);
	if (err < 0)
		return err;

	pcm->chn[SND_PCM_CHANNEL_PLAYBACK].private_data = cm;
	pcm->chn[SND_PCM_CHANNEL_PLAYBACK].open = snd_cmipci_playback_spdif_open;
	pcm->chn[SND_PCM_CHANNEL_PLAYBACK].close = snd_cmipci_playback_spdif_close;

	pcm->chn[SND_PCM_CHANNEL_CAPTURE].private_data = cm;
	pcm->chn[SND_PCM_CHANNEL_CAPTURE].open = snd_cmipci_capture_spdif_open;
	pcm->chn[SND_PCM_CHANNEL_CAPTURE].close = snd_cmipci_capture_spdif_close;

	pcm->private_data = cm;
	pcm->info_flags = SND_PCM_INFO_PLAYBACK | SND_PCM_INFO_CAPTURE |
		SND_PCM_INFO_DUPLEX | SND_PCM_INFO_DUPLEX_RATE;
	strcpy(pcm->name, "C-Media PCI SPDIF");
	cm->pcm_spdif = pcm;

	hw = cm->hw_info[2] = snd_kmalloc(sizeof(snd_pcm_hardware_t), GFP_KERNEL);
	if (! hw)
		return -ENOMEM;
	*hw = snd_cmipci_playback_spdif;
	if (cm->can_ac3_hw) {
		hw->chninfo |= SND_PCM_CHNINFO_MMAP | SND_PCM_CHNINFO_MMAP_VALID;
		hw->formats |= SND_PCM_FMT_S24_LE;
	}

	return 0;
}


/*
 * mixer interface:
 * - CM8338/8738 has a compatible mixer interface with SB16, but
 *   lack of some elements like tone control, i/o gain and AGC.
 * - Access to native registers:
 *   - A 3D switch
 *   - Output mute switches
 *   - AUX in
 *   - mic volume, boost
 */
static void snd_cmipci_mixer_write(cmipci_t *cm, unsigned char idx, unsigned char data)
{
	snd_cmipci_write_b(cm, CM_REG_SB16_ADDR, idx);
	snd_cmipci_write_b(cm, CM_REG_SB16_DATA, data);
}

static unsigned char snd_cmipci_mixer_read(cmipci_t *cm, unsigned char idx)
{
	snd_cmipci_write_b(cm, CM_REG_SB16_ADDR, idx);
	return snd_cmipci_read_b(cm, CM_REG_SB16_DATA);
}

static int snd_cmipci_input_route(snd_kmixer_element_t * element,
				 int w_flag,
				 unsigned int *prsw,
				 unsigned char left_bit,
				 unsigned char right_bit)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, element->private_data, -ENXIO);
	unsigned long flags;
	unsigned char oleft, oright;
	int change = 0, tmp;

	left_bit = 1 << left_bit;
	right_bit = 1 << right_bit;
	spin_lock_irqsave(&cm->reg_lock, flags);
	oleft = snd_cmipci_mixer_read(cm, SB_DSP4_INPUT_LEFT);
	oright = snd_cmipci_mixer_read(cm, SB_DSP4_INPUT_RIGHT);
	if (!w_flag) {
		snd_mixer_set_bit(prsw, 0, oleft & left_bit);
		snd_mixer_set_bit(prsw, 1, oright & left_bit);
		snd_mixer_set_bit(prsw, 2, oleft & right_bit);
		snd_mixer_set_bit(prsw, 3, oright & right_bit);
	} else {
		tmp = snd_mixer_get_bit(prsw, 0);
		if (tmp != ((oleft & left_bit) != 0)) {
			change = 1;
			oleft &= ~left_bit;
			if (tmp)
				oleft |= left_bit;
		}
		tmp = snd_mixer_get_bit(prsw, 1);
		if (tmp != ((oright & left_bit) != 0)) {
			change = 1;
			oright &= ~left_bit;
			if (tmp)
				oright |= left_bit;
		}
		tmp = snd_mixer_get_bit(prsw, 2);
		if (tmp != ((oleft & right_bit) != 0)) {
			change = 1;
			oleft &= ~right_bit;
			if (tmp)
				oleft |= right_bit;
		}
		tmp = snd_mixer_get_bit(prsw, 3);
		if (tmp != ((oright & right_bit) != 0)) {
			change = 1;
			oright &= ~right_bit;
			if (tmp)
				oright |= right_bit;
		}
		snd_cmipci_mixer_write(cm, SB_DSP4_INPUT_LEFT, oleft);
		snd_cmipci_mixer_write(cm, SB_DSP4_INPUT_RIGHT, oright);
	}
	spin_unlock_irqrestore(&cm->reg_lock, flags);
	return change;
}

static int snd_cmipci_cd_input_switch(snd_kmixer_element_t *element, int w_flag, unsigned int *prsw)
{
	return snd_cmipci_input_route(element, w_flag, prsw, 2, 1);
}

static int snd_cmipci_line_input_switch(snd_kmixer_element_t *element, int w_flag, unsigned int *prsw)
{
	return snd_cmipci_input_route(element, w_flag, prsw, 4, 3);
}

static int snd_cmipci_midi_input_switch(snd_kmixer_element_t *element, int w_flag, unsigned int *prsw)
{
	return snd_cmipci_input_route(element, w_flag, prsw, 6, 5);
}

static int snd_cmipci_mic_input_switch(snd_kmixer_element_t *element, int w_flag, unsigned int *bitmap)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, element->private_data, -ENXIO);
	unsigned long flags;
	unsigned char lreg, rreg, oleft, oright;
	int change = 0;

	spin_lock_irqsave(&cm->reg_lock, flags);
	oleft = (lreg = snd_cmipci_mixer_read(cm, SB_DSP4_INPUT_LEFT)) & 1;
	oright = (rreg = snd_cmipci_mixer_read(cm, SB_DSP4_INPUT_RIGHT)) & 1;
	if (!w_flag) {
		snd_mixer_set_bit(bitmap, 0, oleft);
		snd_mixer_set_bit(bitmap, 1, oright);
	} else {
		change = oleft != snd_mixer_get_bit(bitmap, 0) ||
		    oright != snd_mixer_get_bit(bitmap, 1);
		lreg &= ~1;
		lreg |= snd_mixer_get_bit(bitmap, 0) ? 1 : 0;
		rreg &= ~1;
		rreg |= snd_mixer_get_bit(bitmap, 1) ? 1 : 0;
		snd_cmipci_mixer_write(cm, SB_DSP4_INPUT_LEFT, lreg);
		snd_cmipci_mixer_write(cm, SB_DSP4_INPUT_RIGHT, rreg);
	}
	spin_unlock_irqrestore(&cm->reg_lock, flags);
	return change;
}

static int snd_cmipci_mic_output_switch(snd_kmixer_element_t *element, int w_flag, unsigned int *bitmap)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, element->private_data, -ENXIO);
	unsigned long flags;
	unsigned char old, oreg;
	int change = 0;

	spin_lock_irqsave(&cm->reg_lock, flags);
	old = (oreg = snd_cmipci_mixer_read(cm, SB_DSP4_OUTPUT_SW)) & 1;
	if (!w_flag) {
		snd_mixer_set_bit(bitmap, 0, old);
	} else {
		change = old != snd_mixer_get_bit(bitmap, 0);
		oreg &= ~1;
		oreg |= snd_mixer_get_bit(bitmap, 0) ? 1 : 0;
		snd_cmipci_mixer_write(cm, SB_DSP4_OUTPUT_SW, oreg);
	}
	spin_unlock_irqrestore(&cm->reg_lock, flags);
	return change;
}

static int snd_cmipci_cd_output_switch(snd_kmixer_element_t *element, int w_flag, unsigned int *bitmap)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, element->private_data, -ENXIO);
	unsigned long flags;
	unsigned char oreg, oleft, oright;
	int change = 0;

	spin_lock_irqsave(&cm->reg_lock, flags);
	oreg = snd_cmipci_mixer_read(cm, SB_DSP4_OUTPUT_SW);
	oleft = (oreg & 4) ? 1 : 0;
	oright = (oreg & 2) ? 1 : 0;
	if (!w_flag) {
		snd_mixer_set_bit(bitmap, 0, oleft);
		snd_mixer_set_bit(bitmap, 1, oright);
	} else {
		change = oleft != snd_mixer_get_bit(bitmap, 0) ||
		    oright != snd_mixer_get_bit(bitmap, 1);
		oreg &= ~6;
		oreg |= snd_mixer_get_bit(bitmap, 0) ? 4 : 0;
		oreg |= snd_mixer_get_bit(bitmap, 1) ? 2 : 0;
		snd_cmipci_mixer_write(cm, SB_DSP4_OUTPUT_SW, oreg);
	}
	spin_unlock_irqrestore(&cm->reg_lock, flags);
	return change;
}

static int snd_cmipci_line_output_switch(snd_kmixer_element_t *element, int w_flag, unsigned int *bitmap)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, element->private_data, -ENXIO);
	unsigned long flags;
	unsigned char oreg, oleft, oright;
	int change = 0;

	spin_lock_irqsave(&cm->reg_lock, flags);
	oreg = snd_cmipci_mixer_read(cm, SB_DSP4_OUTPUT_SW);
	oleft = (oreg & 0x10) ? 1 : 0;
	oright = (oreg & 8) ? 1 : 0;
	if (!w_flag) {
		snd_mixer_set_bit(bitmap, 0, oleft);
		snd_mixer_set_bit(bitmap, 1, oright);
	} else {
		change = oleft != snd_mixer_get_bit(bitmap, 0) ||
		    oright != snd_mixer_get_bit(bitmap, 1);
		oreg &= ~0x18;
		oreg |= snd_mixer_get_bit(bitmap, 0) ? 0x10 : 0;
		oreg |= snd_mixer_get_bit(bitmap, 1) ? 8 : 0;
		snd_cmipci_mixer_write(cm, SB_DSP4_OUTPUT_SW, oreg);
	}
	spin_unlock_irqrestore(&cm->reg_lock, flags);
	return change;
}

static int snd_cmipci_volume_level(snd_kmixer_element_t *element,
				  int w_flag, int *voices,
				  unsigned char max,
				  unsigned char shift,
				  unsigned char reg)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, element->private_data, -ENXIO);
	unsigned long flags;
	unsigned char lreg, rreg, oleft, oright;
	int change = 0;

	spin_lock_irqsave(&cm->reg_lock, flags);
	oleft = ((lreg = snd_cmipci_mixer_read(cm, reg + 0)) >> shift) & max;
	oright = ((rreg = snd_cmipci_mixer_read(cm, reg + 1)) >> shift) & max;
	if (!w_flag) {
		voices[0] = oleft;
		voices[1] = oright;
	} else {
		change = oleft != voices[0] || oright != voices[1];
		lreg &= ~(max << shift);
		lreg |= voices[0] << shift;
		rreg &= ~(max << shift);
		rreg |= voices[1] << shift;
		snd_cmipci_mixer_write(cm, reg + 0, lreg);
		snd_cmipci_mixer_write(cm, reg + 1, rreg);
	}
	spin_unlock_irqrestore(&cm->reg_lock, flags);
	return change;

}

static int snd_cmipci_mono_volume_level(snd_kmixer_element_t *element,
				       int w_flag, int *voices,
				       unsigned char max,
				       unsigned char shift,
				       unsigned char reg)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, element->private_data, -ENXIO);
	unsigned long flags;
	unsigned char oreg, oval;
	int change = 0;

	spin_lock_irqsave(&cm->reg_lock, flags);
	oval = ((oreg = snd_cmipci_mixer_read(cm, reg)) >> shift) & max;
	if (!w_flag) {
		voices[0] = oval;
	} else {
		change = oval != voices[0];
		oreg &= ~(max << shift);
		oreg |= voices[0] << shift;
		snd_cmipci_mixer_write(cm, reg, oreg);
	}
	spin_unlock_irqrestore(&cm->reg_lock, flags);
	return change;

}

static int snd_cmipci_master_volume_level(snd_kmixer_element_t *element, int w_flag, int *voices)
{
	return snd_cmipci_volume_level(element, w_flag, voices,
				      31, 3, SB_DSP4_MASTER_DEV);
}

static int snd_cmipci_pcm_volume_level(snd_kmixer_element_t *element, int w_flag, int *voices)
{
	return snd_cmipci_volume_level(element, w_flag, voices,
				      31, 3, SB_DSP4_PCM_DEV);
}

static int snd_cmipci_synth_volume_level(snd_kmixer_element_t *element, int w_flag, int *voices)
{
	return snd_cmipci_volume_level(element, w_flag, voices,
				      31, 3, SB_DSP4_SYNTH_DEV);
}

static int snd_cmipci_cd_volume_level(snd_kmixer_element_t *element, int w_flag, int *voices)
{
	return snd_cmipci_volume_level(element, w_flag, voices,
				      31, 3, SB_DSP4_CD_DEV);
}

static int snd_cmipci_line_volume_level(snd_kmixer_element_t *element, int w_flag, int *voices)
{
	return snd_cmipci_volume_level(element, w_flag, voices,
				      31, 3, SB_DSP4_LINE_DEV);
}

static int snd_cmipci_mic_volume_level(snd_kmixer_element_t *element, int w_flag, int *voices)
{
	return snd_cmipci_mono_volume_level(element, w_flag, voices,
					   31, 3, SB_DSP4_MIC_DEV);
}

static int snd_cmipci_speaker_volume_level(snd_kmixer_element_t *element, int w_flag, int *voices)
{
	return snd_cmipci_mono_volume_level(element, w_flag, voices,
					   3, 6, SB_DSP4_SPEAKER_DEV);
}

static int snd_cmipci_3d_surround_switch(snd_kmixer_element_t *element,
					int w_flag, int *value)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, element->private_data, -ENXIO);
	unsigned long flags;
	unsigned char old, oreg;
	int change = 0;

	spin_lock_irqsave(&cm->reg_lock, flags);
	oreg = snd_cmipci_read_b(cm, CM_REG_MIXER1);
	old =  (oreg & CM_X3DEN) ? 1 : 0;
	if (!w_flag) {
		*value = old;
	} else {
		change = old != *value;
		oreg &= ~CM_X3DEN;
		oreg |= *value ? CM_X3DEN : 0;
		snd_cmipci_write_b(cm, CM_REG_MIXER1, oreg);
	}
	spin_unlock_irqrestore(&cm->reg_lock, flags);
	return change;
}

static int snd_cmipci_mute_pcm(snd_kmixer_element_t *element,
			       int w_flag, int *value)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, element->private_data, -ENXIO);
	unsigned long flags;
	unsigned char old, oreg;
	int change = 0;

	spin_lock_irqsave(&cm->reg_lock, flags);
	oreg = snd_cmipci_read_b(cm, CM_REG_MIXER1);
	old =  (oreg & CM_WSMUTE) ? 0 : 1;
	if (!w_flag) {
		*value = old;
	} else {
		change = old != *value;
		oreg &= ~CM_WSMUTE;
		oreg |= *value ? 0 : CM_WSMUTE;
		snd_cmipci_write_b(cm, CM_REG_MIXER1, oreg);
	}
	spin_unlock_irqrestore(&cm->reg_lock, flags);
	return change;
}

static int snd_cmipci_pcm_input_switch(snd_kmixer_element_t *element,
				       int w_flag, unsigned int *bitmap)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, element->private_data, -ENXIO);
	unsigned long flags;
	unsigned char val, oleft, oright;
	int change = 0;

	spin_lock_irqsave(&cm->reg_lock, flags);
	val = snd_cmipci_read_b(cm, CM_REG_MIXER1);
	oleft = (val & CM_WAVEINL) ? 1 : 0;
	oright = (val & CM_WAVEINR) ? 1 : 0;
	if (!w_flag) {
		snd_mixer_set_bit(bitmap, 0, oleft);
		snd_mixer_set_bit(bitmap, 1, oright);
	} else {
		change = oleft != snd_mixer_get_bit(bitmap, 0) ||
		    oright != snd_mixer_get_bit(bitmap, 1);
		val &= ~(CM_WAVEINL|CM_WAVEINR);
		val |= snd_mixer_get_bit(bitmap, 0) ? CM_WAVEINL : 0;
		val |= snd_mixer_get_bit(bitmap, 1) ? CM_WAVEINR : 0;
		snd_cmipci_write_b(cm, CM_REG_MIXER1, val);
	}
	spin_unlock_irqrestore(&cm->reg_lock, flags);
	return change;
}

static int snd_cmipci_aux_input_switch(snd_kmixer_element_t *element,
				       int w_flag, unsigned int *bitmap)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, element->private_data, -ENXIO);
	unsigned long flags;
	unsigned char val, oleft, oright;
	int change = 0;

	spin_lock_irqsave(&cm->reg_lock, flags);
	val = snd_cmipci_read_b(cm, CM_REG_MIXER2);
	oleft = (val & CM_RAUXLEN) ? 1 : 0;
	oright = (val & CM_RAUXREN) ? 1 : 0;
	if (!w_flag) {
		snd_mixer_set_bit(bitmap, 0, oleft);
		snd_mixer_set_bit(bitmap, 1, oright);
	} else {
		change = oleft != snd_mixer_get_bit(bitmap, 0) ||
		    oright != snd_mixer_get_bit(bitmap, 1);
		val &= ~(CM_RAUXLEN|CM_RAUXREN);
		val |= snd_mixer_get_bit(bitmap, 0) ? CM_RAUXLEN : 0;
		val |= snd_mixer_get_bit(bitmap, 1) ? CM_RAUXREN : 0;
		snd_cmipci_write_b(cm, CM_REG_MIXER2, val);
	}
	spin_unlock_irqrestore(&cm->reg_lock, flags);
	return change;
}

static int snd_cmipci_aux_output_switch(snd_kmixer_element_t *element,
					int w_flag, unsigned int *bitmap)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, element->private_data, -ENXIO);
	unsigned long flags;
	unsigned char oreg, oleft, oright;
	int change = 0;

	spin_lock_irqsave(&cm->reg_lock, flags);
	oreg = snd_cmipci_read_b(cm, CM_REG_MIXER2);
	oleft = (oreg & CM_VAUXLM) ? 1 : 0;
	oright = (oreg & CM_VAUXRM) ? 1 : 0;
	if (!w_flag) {
		snd_mixer_set_bit(bitmap, 0, oleft);
		snd_mixer_set_bit(bitmap, 1, oright);
	} else {
		change = oleft != snd_mixer_get_bit(bitmap, 0) ||
		    oright != snd_mixer_get_bit(bitmap, 1);
		oreg &= ~(CM_VAUXLM|CM_VAUXRM);
		oreg |= snd_mixer_get_bit(bitmap, 0) ? CM_VAUXLM : 0;
		oreg |= snd_mixer_get_bit(bitmap, 1) ? CM_VAUXRM : 0;
		snd_cmipci_write_b(cm, CM_REG_MIXER2, oreg);
	}
	spin_unlock_irqrestore(&cm->reg_lock, flags);
	return change;
}

static int snd_cmipci_aux_volume_level(snd_kmixer_element_t *element,
				       int w_flag, int *voices)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, element->private_data, -ENXIO);
	unsigned long flags;
	unsigned char reg, oleft, oright;
	int change = 0;

	spin_lock_irqsave(&cm->reg_lock, flags);
	reg = snd_cmipci_read_b(cm, CM_REG_AUX_VOL);
	oleft = (reg >> 4) & 0x0f;
	oright = reg & 0x0f;
	if (!w_flag) {
		voices[0] = oleft;
		voices[1] = oright;
	} else {
		change = oleft != voices[0] || oright != voices[1];
		reg = ((voices[0] & 0x0f) << 4) | (voices[1] & 0x0f);
		snd_cmipci_write_b(cm, CM_REG_AUX_VOL, reg);
	}
	spin_unlock_irqrestore(&cm->reg_lock, flags);
	return change;

}

static int snd_cmipci_mic_gain_switch(snd_kmixer_element_t *element,
				      int w_flag, unsigned int *bitmap)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, element->private_data, -ENXIO);
	unsigned long flags;
 	unsigned char oreg, oval;
 	int change = 0;
 
 	spin_lock_irqsave(&cm->reg_lock, flags);
 	oreg = snd_cmipci_read_b(cm, CM_REG_MIXER2);
 	oval = (oreg & CM_MICGAINZ) ? 0 : 1;
 	if (!w_flag) {
		snd_mixer_set_bit(bitmap, 0, oval);
	} else {
		change = oval != snd_mixer_get_bit(bitmap, 0);
		oreg &= ~(CM_MICGAINZ);
		oreg |= snd_mixer_get_bit(bitmap, 0) ? 0 : CM_MICGAINZ;
		snd_cmipci_write_b(cm, CM_REG_MIXER2, oreg);
	}
	spin_unlock_irqrestore(&cm->reg_lock, flags);
	return change;
}

static int snd_cmipci_mic_gain_level(snd_kmixer_element_t *element,
				     int w_flag, int *voices)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, element->private_data, -ENXIO);
	unsigned long flags;
	unsigned char reg, oval;
	int change = 0;
 
	spin_lock_irqsave(&cm->reg_lock, flags);
	reg = snd_cmipci_read_b(cm, CM_REG_MIXER2);
	oval = (reg & CM_VADMIC_MASK) >> 1;
	if (!w_flag) {
		voices[0] = oval;
	} else {
		change = oval != voices[0];
		reg &= ~CM_VADMIC_MASK;
		reg |= ((voices[0] << 1 ) & CM_VADMIC_MASK);
		snd_cmipci_write_b(cm, CM_REG_MIXER2, reg);
	}
	spin_unlock_irqrestore(&cm->reg_lock, flags);
	return change;
}
 
#if 0
static int snd_cmipci_mute_synth(snd_kmixer_element_t *element,
			       int w_flag, int *value)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, element->private_data, -ENXIO);
	unsigned long flags;
	unsigned char old, oreg;
	int change = 0;

	spin_lock_irqsave(&cm->reg_lock, flags);
	oreg = snd_cmipci_read_b(cm, CM_REG_MIXER1);
	old =  (oreg & CM_FMMUTE) ? 0 : 1;
	if (!w_flag) {
		*value = old;
	} else {
		change = old != *value;
		oreg &= ~CM_FMMUTE;
		oreg |= *value ? 0: CM_FMMUTE;
		snd_cmipci_write_b(cm, CM_REG_MIXER1, oreg);
	}
	spin_unlock_irqrestore(&cm->reg_lock, flags);
	return change;
}
#endif

static int snd_cmipci_group_ctrl_stereo(snd_kmixer_group_t * group,
					snd_kmixer_file_t * file, int w_flag,
					snd_mixer_group_t * ugroup,
					snd_mixer_volume1_control_t * volume1,
					snd_kmixer_element_t * volume1_element,
					int max,
					snd_mixer_sw1_control_t * sw1_output,
					snd_kmixer_element_t * sw1_output_element,
					snd_mixer_sw1_control_t * sw1_input,
					snd_kmixer_element_t * sw1_input_element,
					snd_mixer_sw3_control_t * sw3_input,
					snd_kmixer_element_t * sw3_input_element,
					snd_mixer_sw2_control_t * sw2_output,
					snd_kmixer_element_t * sw2_output_element)
{
	int voices[2];
	unsigned int bitmap;
	int change = 0;
	
	if (!w_flag) {
		ugroup->caps = 0;
		ugroup->channels = SND_MIXER_CHN_MASK_STEREO;
		if (volume1) {
			ugroup->caps |= SND_MIXER_GRPCAP_VOLUME;
			volume1(volume1_element, 0, voices);
			ugroup->volume.names.front_left = voices[0];
			ugroup->volume.names.front_right = voices[1];
			ugroup->min = 0;
			ugroup->max = max;
		}
		if (sw1_output) {
			ugroup->caps |= SND_MIXER_GRPCAP_MUTE;
			sw1_output(sw1_output_element, 0, &bitmap);
			ugroup->mute = 0;
			if (!snd_mixer_get_bit(&bitmap, 0))
				ugroup->mute |= SND_MIXER_CHN_MASK_FRONT_LEFT;
			if (!snd_mixer_get_bit(&bitmap, 1))
				ugroup->mute |= SND_MIXER_CHN_MASK_FRONT_RIGHT;
		} else if (sw2_output) {
			int value;
			ugroup->caps |= SND_MIXER_GRPCAP_MUTE | SND_MIXER_GRPCAP_JOINTLY_MUTE;
			sw2_output(sw2_output_element, 0, &value);
			ugroup->mute = 0;
			if (!value)
				ugroup->mute |= SND_MIXER_CHN_MASK_STEREO;
		}
		if (sw1_input) {
			ugroup->caps |= SND_MIXER_GRPCAP_CAPTURE;
			sw1_input(sw1_input_element, 0, &bitmap);
			ugroup->capture = 0;
			if (snd_mixer_get_bit(&bitmap, 0))
				ugroup->capture |= SND_MIXER_CHN_MASK_FRONT_LEFT;
			if (snd_mixer_get_bit(&bitmap, 1))
				ugroup->capture |= SND_MIXER_CHN_MASK_FRONT_RIGHT;
		} else if (sw3_input) {
			ugroup->caps |= SND_MIXER_GRPCAP_CAPTURE;
			sw3_input(sw3_input_element, 0, &bitmap);
			ugroup->capture = 0;
			if (snd_mixer_get_bit(&bitmap, 0))
				ugroup->capture |= SND_MIXER_CHN_MASK_FRONT_LEFT;
			if (snd_mixer_get_bit(&bitmap, 3))
				ugroup->capture |= SND_MIXER_CHN_MASK_FRONT_RIGHT;
		}
	} else {
		if (volume1) {
			voices[0] = ugroup->volume.names.front_left & max;
			voices[1] = ugroup->volume.names.front_right & max;
			if (volume1(volume1_element, 1, voices) > 0) {
				snd_mixer_element_value_change(file, volume1_element, 0);
				change = 1;
			}
		}
		if (sw1_output) {
			bitmap = 0;
			if (!(ugroup->mute & SND_MIXER_CHN_MASK_FRONT_LEFT))
				snd_mixer_set_bit(&bitmap, 0, 1);
			if (!(ugroup->mute & SND_MIXER_CHN_MASK_FRONT_RIGHT))
				snd_mixer_set_bit(&bitmap, 1, 1);
			if (sw1_output(sw1_output_element, 1, &bitmap) > 0) {
				snd_mixer_element_value_change(file, sw1_output_element, 0);
				change = 1;
			}
		} else if (sw2_output) {
			int value = 0;
			if ((ugroup->mute & SND_MIXER_CHN_MASK_STEREO) == 0)
				value = 1;
			if (sw2_output(sw2_output_element, 1, &value) > 0) {
				snd_mixer_element_value_change(file, sw2_output_element, 0);
				change = 1;
			}
		}
		if (sw1_input) {
			bitmap = 0;
			if (ugroup->capture & SND_MIXER_CHN_MASK_FRONT_LEFT)
				snd_mixer_set_bit(&bitmap, 0, 1);
			if (ugroup->capture & SND_MIXER_CHN_MASK_FRONT_RIGHT)
				snd_mixer_set_bit(&bitmap, 1, 1);
			if (sw1_input(sw1_input_element, 1, &bitmap) > 0) {
				snd_mixer_element_value_change(file, sw1_input_element, 0);
				change = 1;
			}
		} else if (sw3_input) {
			bitmap = 0;
			if (ugroup->capture & SND_MIXER_CHN_MASK_FRONT_LEFT)
				snd_mixer_set_bit(&bitmap, 0, 1);
			if (ugroup->capture & SND_MIXER_CHN_MASK_FRONT_RIGHT)
				snd_mixer_set_bit(&bitmap, 3, 1);
			if (sw3_input(sw3_input_element, 1, &bitmap) > 0) {
				snd_mixer_element_value_change(file, sw3_input_element, 0);
				change = 1;
			}
		}
	}
	return change;
}

static int snd_cmipci_group_ctrl_mono(snd_kmixer_group_t * group,
				     snd_kmixer_file_t * file, int w_flag,
				     snd_mixer_group_t * ugroup,
				     snd_mixer_volume1_control_t * volume1,
				     snd_kmixer_element_t * volume1_element,
				     int max,
				     snd_mixer_sw1_control_t * sw1_output,
				     snd_kmixer_element_t * sw1_output_element,
				     snd_mixer_sw1_control_t * sw1_input,
				     snd_kmixer_element_t * sw1_input_element)
{
	int voice;
	unsigned int bitmap;
	int change = 0;

	if (!w_flag) {
		ugroup->caps = 0;
		ugroup->channels = SND_MIXER_CHN_MASK_STEREO;
		if (volume1) {
			ugroup->caps |= SND_MIXER_GRPCAP_VOLUME | SND_MIXER_GRPCAP_JOINTLY_VOLUME;
			volume1(volume1_element, 0, &voice);
			ugroup->volume.names.front_left = voice;
			ugroup->volume.names.front_right = voice;
			ugroup->min = 0;
			ugroup->max = max;
		}
		if (sw1_output) {
			ugroup->caps |= SND_MIXER_GRPCAP_MUTE;
			sw1_output(sw1_output_element, 0, &bitmap);
			ugroup->mute = 0;
			if (!snd_mixer_get_bit(&bitmap, 0))
				ugroup->mute |= SND_MIXER_CHN_MASK_FRONT_LEFT;
			if (!snd_mixer_get_bit(&bitmap, 1))
				ugroup->mute |= SND_MIXER_CHN_MASK_FRONT_RIGHT;
		}
		if (sw1_input) {
			ugroup->caps |= SND_MIXER_GRPCAP_CAPTURE;
			sw1_input(sw1_input_element, 0, &bitmap);
			ugroup->capture = 0;
			if (snd_mixer_get_bit(&bitmap, 0))
				ugroup->capture |= SND_MIXER_CHN_MASK_FRONT_LEFT;
			if (snd_mixer_get_bit(&bitmap, 1))
				ugroup->capture |= SND_MIXER_CHN_MASK_FRONT_RIGHT;
		}
	} else {
		if (volume1) {
			voice = ugroup->volume.names.front_left & max;
			if (volume1(volume1_element, 1, &voice) > 0) {
				snd_mixer_element_value_change(file, volume1_element, 0);
				change = 1;
			}
		}
		if (sw1_output) {
			bitmap = 0;
			if (!(ugroup->mute & SND_MIXER_CHN_MASK_FRONT_LEFT))
				snd_mixer_set_bit(&bitmap, 0, 1);
			if (!(ugroup->mute & SND_MIXER_CHN_MASK_FRONT_RIGHT))
				snd_mixer_set_bit(&bitmap, 1, 1);
			if (sw1_output(sw1_output_element, 1, &bitmap) > 0) {
				snd_mixer_element_value_change(file, sw1_output_element, 0);
				change = 1;
			}
		}
		if (sw1_input) {
			bitmap = 0;
			if (ugroup->capture & SND_MIXER_CHN_MASK_FRONT_LEFT)
				snd_mixer_set_bit(&bitmap, 0, 1);
			if (ugroup->capture & SND_MIXER_CHN_MASK_FRONT_RIGHT)
				snd_mixer_set_bit(&bitmap, 1, 1);
			if (sw1_input(sw1_input_element, 1, &bitmap) > 0) {
				snd_mixer_element_value_change(file, sw1_input_element, 0);
				change = 1;
			}
		}
	}
	return change;
}

static int snd_cmipci_group_speaker(snd_kmixer_group_t * group,
				   snd_kmixer_file_t * file, int w_flag,
				   snd_mixer_group_t * ugroup)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, group->private_data, -ENXIO);

	return snd_cmipci_group_ctrl_mono(group, file, w_flag, ugroup,
					 snd_cmipci_speaker_volume_level,
					 cm->me_vol_speaker,
					 3,
					 NULL,
					 NULL,
					 NULL,
					 NULL);
}

static int snd_cmipci_group_mic(snd_kmixer_group_t * group,
			       snd_kmixer_file_t * file, int w_flag,
			       snd_mixer_group_t * ugroup)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, group->private_data, -ENXIO);

	return snd_cmipci_group_ctrl_mono(group, file, w_flag, ugroup,
					 snd_cmipci_mic_volume_level,
					 cm->me_vol_mic,
					 31,
					 snd_cmipci_mic_output_switch,
					 cm->me_sw1_mic_output,
					 snd_cmipci_mic_input_switch,
					 cm->me_sw1_mic_input);
}

static int snd_cmipci_group_mic_gain(snd_kmixer_group_t * group,
				     snd_kmixer_file_t * file, int w_flag,
				     snd_mixer_group_t * ugroup)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, group->private_data, -ENXIO);

	return snd_cmipci_group_ctrl_mono(group, file, w_flag, ugroup,
					  snd_cmipci_mic_gain_level,
					  cm->me_mic_gain_vol,
					  7,
					  NULL, NULL,
					  NULL, NULL);
}
 
static int snd_cmipci_group_line(snd_kmixer_group_t * group,
				snd_kmixer_file_t * file, int w_flag,
				snd_mixer_group_t * ugroup)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, group->private_data, -ENXIO);
            
	return snd_cmipci_group_ctrl_stereo(group, file, w_flag, ugroup,
					    snd_cmipci_line_volume_level,
					    cm->me_vol_line,
					    31,
					    snd_cmipci_line_output_switch,
					    cm->me_sw1_line_output,
					    NULL,
					    NULL,
					    snd_cmipci_line_input_switch,
					    cm->me_sw3_line_input,
					    NULL, NULL);
}

static int snd_cmipci_group_cd(snd_kmixer_group_t * group,
			      snd_kmixer_file_t * file, int w_flag,
			      snd_mixer_group_t * ugroup)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, group->private_data, -ENXIO);

	return snd_cmipci_group_ctrl_stereo(group, file, w_flag, ugroup,
					    snd_cmipci_cd_volume_level,
					    cm->me_vol_cd,
					    31,
					    snd_cmipci_cd_output_switch,
					    cm->me_sw1_cd_output,
					    NULL,
					    NULL,
					    snd_cmipci_cd_input_switch,
					    cm->me_sw3_cd_input,
					    NULL, NULL);
}

static int snd_cmipci_group_aux(snd_kmixer_group_t * group,
				snd_kmixer_file_t * file, int w_flag,
				snd_mixer_group_t * ugroup)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, group->private_data, -ENXIO);

	return snd_cmipci_group_ctrl_stereo(group, file, w_flag, ugroup,
					    snd_cmipci_aux_volume_level,
					    cm->me_vol_aux,
					    15,
					    snd_cmipci_aux_output_switch,
					    cm->me_sw1_aux_output,
					    snd_cmipci_aux_input_switch,
					    cm->me_sw1_aux_input,
					    NULL, NULL,
					    NULL, NULL);
}

static int snd_cmipci_group_synth(snd_kmixer_group_t * group,
				 snd_kmixer_file_t * file, int w_flag,
				 snd_mixer_group_t * ugroup)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, group->private_data, -ENXIO);

	return snd_cmipci_group_ctrl_stereo(group, file, w_flag, ugroup,
					    snd_cmipci_synth_volume_level,
					    cm->me_vol_synth,
					    31,
					    NULL,
					    NULL,
					    NULL,
					    NULL,
					    snd_cmipci_midi_input_switch,
					    cm->me_sw3_synth_input,
					    NULL, NULL);
}

static int snd_cmipci_group_pcm(snd_kmixer_group_t * group,
				   snd_kmixer_file_t * file, int w_flag,
				   snd_mixer_group_t * ugroup)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, group->private_data, -ENXIO);

	return snd_cmipci_group_ctrl_stereo(group, file, w_flag, ugroup,
					    snd_cmipci_pcm_volume_level,
					    cm->me_vol_pcm,
					    31,
					    NULL,
					    NULL,
					    snd_cmipci_pcm_input_switch,
					    cm->me_sw1_pcm_input,
					    NULL,
					    NULL,
					    snd_cmipci_mute_pcm,
					    cm->me_sw2_mute_pcm);
}

static int snd_cmipci_group_master(snd_kmixer_group_t * group,
				  snd_kmixer_file_t * file, int w_flag,
				  snd_mixer_group_t * ugroup)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, group->private_data, -ENXIO);

	return snd_cmipci_group_ctrl_stereo(group, file, w_flag, ugroup,
					    snd_cmipci_master_volume_level,
					    cm->me_vol_master,
					    31,
					    NULL,
					    NULL,
					    NULL,
					    NULL,
					    NULL,
					    NULL,
					    NULL, NULL);
}


/*
 * mixer/control switches
 */

typedef struct snd_cmipci_switch_args {
	int reg;
	unsigned int mask;
	unsigned int mask_on;
	int is_byte;
} cmipci_switch_args_t;

static int snd_cmipci_get_switch(void *desc, snd_kswitch_t *kswitch, snd_switch_t *uswitch)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, kswitch->private_data, -ENXIO);
	cmipci_switch_args_t *args = (cmipci_switch_args_t*)kswitch->private_value;
	unsigned long flags;
	unsigned int val;

	spin_lock_irqsave(&cm->reg_lock, flags);
	if (args->is_byte)
		val = snd_cmipci_read_b(cm, args->reg);
	else
		val = snd_cmipci_read(cm, args->reg);
	spin_unlock_irqrestore(&cm->reg_lock, flags);

	uswitch->type = SND_SW_TYPE_BOOLEAN;
	uswitch->value.enable = (val & args->mask) == args->mask_on ? 1 : 0;
	return 0;
}

static int snd_cmipci_set_switch(void *desc, snd_kswitch_t *kswitch, snd_switch_t *uswitch)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, kswitch->private_data, -ENXIO);
	cmipci_switch_args_t *args = (cmipci_switch_args_t*)kswitch->private_value;
	unsigned long flags;
	unsigned int val, on;
	int change;

	if (uswitch->type != SND_SW_TYPE_BOOLEAN)
		return -EINVAL;
	spin_lock_irqsave(&cm->reg_lock, flags);
	if (args->is_byte)
		val = snd_cmipci_read_b(cm, args->reg);
	else
		val = snd_cmipci_read(cm, args->reg);
	if (! uswitch->value.enable)
		on = ~args->mask_on & args->mask;
	else
		on = args->mask_on;
	change = (val & args->mask) != on;
	val &= ~args->mask;
	val |= on;
	if (args->is_byte)
		snd_cmipci_write_b(cm, args->reg, val);
	else
		snd_cmipci_write(cm, args->reg, val);
	spin_unlock_irqrestore(&cm->reg_lock, flags);
	return change;
}


#define DEFINE_SWITCH_ARG(sname, xreg, xmask, xmask_on, xis_byte) \
static cmipci_switch_args_t cmipci_switch_arg_##sname = { \
  reg: xreg, \
  mask: xmask, \
  mask_on: xmask_on, \
  is_byte: xis_byte, \
}
	
#define DEFINE_BIT_SWITCH_ARG(sname, xreg, xmask, xis_byte) \
	DEFINE_SWITCH_ARG(sname, xreg, xmask, xmask, xis_byte)

#if 0 /* these will be controlled in pcm device */
DEFINE_BIT_SWITCH_ARG(spdif_in, CM_REG_FUNCTRL1, CM_SPDF_1, 0);
DEFINE_BIT_SWITCH_ARG(spdif_out, CM_REG_FUNCTRL1, CM_SPDF_0, 0);
DEFINE_BIT_SWITCH_ARG(spdo_48k, CM_REG_MISC_CTRL, CM_SPDF_AC97|CM_SPDIF48K, 0);
#endif
DEFINE_BIT_SWITCH_ARG(spdif_enable, CM_REG_LEGACY_CTRL, CM_ENSPDOUT, 0);
DEFINE_BIT_SWITCH_ARG(spdi_valid, CM_REG_MISC, CM_SPDVALID, 1);
DEFINE_BIT_SWITCH_ARG(spdif_copyright, CM_REG_LEGACY_CTRL, CM_SPDCOPYRHT, 0);
DEFINE_BIT_SWITCH_ARG(spdif_dac_out, CM_REG_LEGACY_CTRL, CM_DAC2SPDO, 0);
DEFINE_BIT_SWITCH_ARG(spdo2dac, CM_REG_FUNCTRL1, CM_SPDO2DAC, 0);
DEFINE_SWITCH_ARG(spdo_5v, CM_REG_MISC_CTRL, CM_SPDO5V, 0, 0 ); /* inverse: 0 = 5V */
DEFINE_BIT_SWITCH_ARG(spdif_loop, CM_REG_FUNCTRL1, CM_SPDFLOOP, 0);
DEFINE_BIT_SWITCH_ARG(spdi_monitor, CM_REG_MIXER1, CM_CDPLAY, 1);
DEFINE_BIT_SWITCH_ARG(spdi_phase, CM_REG_MISC, 0x04, 0 );
DEFINE_BIT_SWITCH_ARG(exchange_dac, CM_REG_MISC_CTRL, CM_XCHGDAC, 0);
DEFINE_BIT_SWITCH_ARG(fourch, CM_REG_MISC_CTRL, CM_N4SPK3D, 0);
DEFINE_BIT_SWITCH_ARG(line_rear, CM_REG_MIXER1, CM_SPK4, 1);
DEFINE_BIT_SWITCH_ARG(line_bass, CM_REG_LEGACY_CTRL, CM_LINE_AS_BASS, 0);
DEFINE_BIT_SWITCH_ARG(joystick, CM_REG_FUNCTRL1, CM_JYSTK_EN, 0);
DEFINE_SWITCH_ARG(modem, CM_REG_MISC_CTRL, CM_FLINKON|CM_FLINKOFF, CM_FLINKON, 0);


#define DEFINE_SWITCH(sname, sarg) \
{ name: sname, \
  get: snd_cmipci_get_switch, \
  set: snd_cmipci_set_switch, \
  private_value: (unsigned long)&cmipci_switch_arg_##sarg,\
}

#define numberof(ary)	(sizeof(ary) / sizeof(ary[0]))

/* both for CM8338/8738 */
static snd_kswitch_t snd_cmipci_mixer_switches[] = {
	DEFINE_SWITCH("Exchange DAC", exchange_dac),
	DEFINE_SWITCH("Analog Four Channel", fourch),
	DEFINE_SWITCH("Line-In As Rear", line_rear),
};

/* only for CM8738 */
static snd_kswitch_t snd_cmipci_8738_mixer_switches[] = {
#if 0 /* controlled in pcm device */
	DEFINE_SWITCH("SPDIF In Record", spdif_in),
	DEFINE_SWITCH("SPDIF Out", spdif_out),
	DEFINE_SWITCH("SPDIF Out 48KHz", spdo_48k),
#endif
	DEFINE_SWITCH("SPDIF Enable", spdif_enable),
	DEFINE_SWITCH("SPDIF In Valid", spdi_valid),
	DEFINE_SWITCH("SPDIF Copyright", spdif_copyright),
	DEFINE_SWITCH("SPDIF DAC To Out", spdif_dac_out),
	DEFINE_SWITCH("SPDIF Out To DAC", spdo2dac),
	DEFINE_SWITCH("SPDIF 5V", spdo_5v),
	DEFINE_SWITCH("SPDIF Loop", spdif_loop),
	DEFINE_SWITCH("SPDIF In Monitor", spdi_monitor),
	DEFINE_SWITCH("SPDIF In Phase Inverse", spdi_phase),
};

/* only for model 039 */
static snd_kswitch_t snd_cmipci_extra_mixer_switches[] = {
	DEFINE_SWITCH("Line-In As Bass", line_bass),
};

/* universal switches */
static snd_kswitch_t snd_cmipci_control_switches[] = {
	DEFINE_SWITCH(SND_CTL_SW_JOYSTICK, joystick),
	DEFINE_SWITCH("Modem", modem),
};


/*
 * create mixer interface
 */

static int __init snd_cmipci_mixer_new(cmipci_t *cm, int pcm_device)
{
	snd_kmixer_t *mixer;
	unsigned long flags;
	snd_kmixer_group_t *group;
	int i, err;
	snd_kswitch_t *sw;
	static struct snd_mixer_element_volume1_range db_range_31s[2] =
	{
		{0, 31, -6200, 0},
		{0, 31, -6200, 0}
	};
	static struct snd_mixer_element_volume1_range db_range_3m[1] =
	{
		{0, 3, -1800, 0}
	};
	static struct snd_mixer_element_volume1_range db_range_15s[2] =
	{
		{0, 15, -6200, 0},
		{0, 15, -6200, 0}
	};
	static struct snd_mixer_element_volume1_range db_range_7m[1] =
	{
		{0, 7, -6200, 0}
	};
	static snd_mixer_voice_t stereo_voices[2] =
	{
		{SND_MIXER_VOICE_LEFT, 0},
		{SND_MIXER_VOICE_RIGHT, 0}
	};

	if ((err = snd_mixer_new(cm->card, "C-Media PCI", 0, &mixer)) < 0)
		return err;
	strcpy(mixer->name, mixer->id);

	spin_lock_irqsave(&cm->reg_lock, flags);
	snd_cmipci_mixer_write(cm, 0x00, 0x00);		/* mixer reset */
	spin_unlock_irqrestore(&cm->reg_lock, flags);

	/* build input and output accumulator */
	if ((cm->me_in_accu = snd_mixer_lib_accu1(mixer, SND_MIXER_ELEMENT_INPUT_ACCU, 0, 0)) == NULL)
		goto __error;
	if ((cm->me_out_accu = snd_mixer_lib_accu1(mixer, SND_MIXER_ELEMENT_OUTPUT_ACCU, 0, 0)) == NULL)
		goto __error;
	/* build master volume control */
	if ((group = snd_mixer_lib_group_ctrl(mixer, SND_MIXER_OUT_MASTER, 0, SND_MIXER_OSS_VOLUME, snd_cmipci_group_master, cm)) == NULL)
		goto __error;
	if ((cm->me_vol_master = snd_mixer_lib_volume1(mixer, "Master Volume", 0, 2, db_range_31s, snd_cmipci_master_volume_level, cm)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, cm->me_vol_master) < 0)
		goto __error;
	if ((cm->me_sw1_3dse = snd_mixer_lib_sw2(mixer, "3D Surround", 0, snd_cmipci_3d_surround_switch, cm)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, cm->me_sw1_3dse) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, cm->me_out_accu, cm->me_sw1_3dse) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, cm->me_sw1_3dse, cm->me_vol_master) < 0)
		goto __error;
	if ((cm->me_out_master = snd_mixer_lib_io_stereo(mixer, SND_MIXER_OUT_MASTER, 0, SND_MIXER_ETYPE_OUTPUT, 0)) == NULL)
		goto __error;
	if (snd_mixer_element_route_add(mixer, cm->me_vol_master, cm->me_out_master) < 0)
		goto __error;
	/* PCM */
	if ((group = snd_mixer_lib_group_ctrl(mixer, SND_MIXER_IN_PCM, 0, SND_MIXER_OSS_PCM, snd_cmipci_group_pcm, cm)) == NULL)
		goto __error;
	if ((cm->me_playback = snd_mixer_lib_pcm1(mixer, SND_MIXER_ELEMENT_PLAYBACK, 0, SND_MIXER_ETYPE_PLAYBACK1, 1, &pcm_device)) == NULL)
		goto __error;
	if ((cm->me_vol_pcm = snd_mixer_lib_volume1(mixer, "PCM Volume", 0, 2, db_range_31s, snd_cmipci_pcm_volume_level, cm)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, cm->me_vol_pcm) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, cm->me_playback, cm->me_vol_pcm) < 0)
		goto __error;
	if ((cm->me_sw2_mute_pcm = snd_mixer_lib_sw2(mixer, "PCM Output Switch", 0, snd_cmipci_mute_pcm, cm)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, cm->me_sw2_mute_pcm) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, cm->me_vol_pcm, cm->me_sw2_mute_pcm) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, cm->me_sw2_mute_pcm, cm->me_out_accu) < 0)
		goto __error;
	if ((cm->me_sw1_pcm_input = snd_mixer_lib_sw1(mixer, "PCM Input Switch", 0, 2, snd_cmipci_pcm_input_switch, cm)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, cm->me_sw1_pcm_input) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, cm->me_vol_pcm, cm->me_sw1_pcm_input) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, cm->me_sw1_pcm_input, cm->me_in_accu) < 0)
		goto __error;
	/* Synth */
	if ((group = snd_mixer_lib_group_ctrl(mixer, SND_MIXER_IN_SYNTHESIZER, 0, SND_MIXER_OSS_SYNTH, snd_cmipci_group_synth, cm)) == NULL)
		goto __error;
	if ((cm->me_in_synth = snd_mixer_lib_io_stereo(mixer, SND_MIXER_IN_SYNTHESIZER, 0, SND_MIXER_ETYPE_INPUT, 0)) == NULL)
		goto __error;
	if ((cm->me_vol_synth = snd_mixer_lib_volume1(mixer, "Synth Volume", 0, 2, db_range_31s, snd_cmipci_synth_volume_level, cm)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, cm->me_vol_synth) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, cm->me_vol_synth, cm->me_out_accu) < 0)
		goto __error;
	if ((cm->me_sw3_synth_input = snd_mixer_lib_sw3(mixer, "Synth Input Switch", 0, SND_MIXER_SWITCH3_FULL_FEATURED, 2, stereo_voices, snd_cmipci_midi_input_switch, cm)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, cm->me_sw3_synth_input) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, cm->me_vol_synth, cm->me_sw3_synth_input) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, cm->me_sw3_synth_input, cm->me_in_accu) < 0)
		goto __error;
	/* CD */
	if ((group = snd_mixer_lib_group_ctrl(mixer, SND_MIXER_IN_CD, 0, SND_MIXER_OSS_CD, snd_cmipci_group_cd, cm)) == NULL)
		goto __error;
	if ((cm->me_in_cd = snd_mixer_lib_io_stereo(mixer, SND_MIXER_IN_CD, 0, SND_MIXER_ETYPE_INPUT, 0)) == NULL)
		goto __error;
	if ((cm->me_vol_cd = snd_mixer_lib_volume1(mixer, "CD Volume", 0, 2, db_range_31s, snd_cmipci_cd_volume_level, cm)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, cm->me_vol_cd) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, cm->me_in_cd, cm->me_vol_cd) < 0)
		goto __error;
	if ((cm->me_sw1_cd_output = snd_mixer_lib_sw1(mixer, "CD Output Switch", 0, 2, snd_cmipci_cd_output_switch, cm)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, cm->me_sw1_cd_output) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, cm->me_vol_cd, cm->me_sw1_cd_output) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, cm->me_sw1_cd_output, cm->me_out_accu) < 0)
		goto __error;
	if ((cm->me_sw3_cd_input = snd_mixer_lib_sw3(mixer, "CD Input Switch", 0, SND_MIXER_SWITCH3_FULL_FEATURED, 2, stereo_voices, snd_cmipci_cd_input_switch, cm)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, cm->me_sw3_cd_input) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, cm->me_vol_cd, cm->me_sw3_cd_input) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, cm->me_sw3_cd_input, cm->me_in_accu) < 0)
		goto __error;
	/* Line */
	if ((group = snd_mixer_lib_group_ctrl(mixer, SND_MIXER_IN_LINE, 0, SND_MIXER_OSS_LINE, snd_cmipci_group_line, cm)) == NULL)
		goto __error;
	if ((cm->me_in_line = snd_mixer_lib_io_stereo(mixer, SND_MIXER_IN_LINE, 0, SND_MIXER_ETYPE_INPUT, 0)) == NULL)
		goto __error;
	if ((cm->me_vol_line = snd_mixer_lib_volume1(mixer, "Line Volume", 0, 2, db_range_31s, snd_cmipci_line_volume_level, cm)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, cm->me_vol_line) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, cm->me_in_line, cm->me_vol_line) < 0)
		goto __error;
	if ((cm->me_sw1_line_output = snd_mixer_lib_sw1(mixer, "Line Output Switch", 0, 2, snd_cmipci_line_output_switch, cm)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, cm->me_sw1_line_output) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, cm->me_vol_line, cm->me_sw1_line_output) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, cm->me_sw1_line_output, cm->me_out_accu) < 0)
		goto __error;
	if ((cm->me_sw3_line_input = snd_mixer_lib_sw3(mixer, "Line Input Switch", 0, SND_MIXER_SWITCH3_FULL_FEATURED, 2, stereo_voices, snd_cmipci_line_input_switch, cm)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, cm->me_sw3_line_input) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, cm->me_vol_line, cm->me_sw3_line_input) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, cm->me_sw3_line_input, cm->me_in_accu) < 0)
		goto __error;
	/* MIC */
	if ((group = snd_mixer_lib_group_ctrl(mixer, SND_MIXER_IN_MIC, 0, SND_MIXER_OSS_MIC, snd_cmipci_group_mic, cm)) == NULL)
		goto __error;
	if ((cm->me_in_mic = snd_mixer_lib_io_mono(mixer, SND_MIXER_IN_MIC, 0, SND_MIXER_ETYPE_INPUT, 0)) == NULL)
		goto __error;
	if ((cm->me_vol_mic = snd_mixer_lib_volume1(mixer, "MIC Volume", 0, 1, db_range_31s, snd_cmipci_mic_volume_level, cm)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, cm->me_vol_mic) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, cm->me_in_mic, cm->me_vol_mic) < 0)
		goto __error;
	if ((cm->me_sw1_mic_output = snd_mixer_lib_sw1(mixer, "MIC Output Switch", 0, 1, snd_cmipci_mic_output_switch, cm)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, cm->me_sw1_mic_output) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, cm->me_vol_mic, cm->me_sw1_mic_output) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, cm->me_sw1_mic_output, cm->me_out_accu) < 0)
		goto __error;
	if ((cm->me_sw1_mic_input = snd_mixer_lib_sw1(mixer, "MIC Input Switch", 0, 2, snd_cmipci_mic_input_switch, cm)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, cm->me_sw1_mic_input) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, cm->me_vol_mic, cm->me_sw1_mic_input) < 0)
		goto __error;
 	/* mic gain control */
	if ((group = snd_mixer_lib_group_ctrl(mixer, SND_MIXER_GRP_MIC_GAIN, 0, SND_MIXER_OSS_IGAIN, snd_cmipci_group_mic_gain, cm)) == NULL)
		goto __error;
	if ((cm->me_mic_gain_vol = snd_mixer_lib_volume1(mixer, "Mic Gain Level", 0, 1, db_range_7m, snd_cmipci_mic_gain_level, cm)) == NULL)
 		goto __error;
	if (snd_mixer_group_element_add(mixer, group, cm->me_mic_gain_vol) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, cm->me_sw1_mic_input, cm->me_mic_gain_vol) < 0)
		goto __error;
	if ((cm->me_sw1_mic_gain = snd_mixer_lib_sw1(mixer, "Mic Gain Switch", 0, 1, snd_cmipci_mic_gain_switch, cm)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, cm->me_sw1_mic_gain) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, cm->me_mic_gain_vol, cm->me_sw1_mic_gain) < 0)
		goto __error;

	if (snd_mixer_element_route_add(mixer, cm->me_sw1_mic_gain, cm->me_in_accu) < 0)
		goto __error;

	/* Speaker */
	if ((group = snd_mixer_lib_group_ctrl(mixer, SND_MIXER_IN_SPEAKER, 0, SND_MIXER_OSS_SPEAKER, snd_cmipci_group_speaker, cm)) == NULL)
		goto __error;
	if ((cm->me_in_speaker = snd_mixer_lib_io_mono(mixer, SND_MIXER_IN_SPEAKER, 0, SND_MIXER_ETYPE_INPUT, 0)) == NULL)
		goto __error;
	if ((cm->me_vol_speaker = snd_mixer_lib_volume1(mixer, "PC Speaker Volume", 0, 1, db_range_3m, snd_cmipci_speaker_volume_level, cm)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, cm->me_vol_speaker) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, cm->me_in_speaker, cm->me_vol_speaker) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, cm->me_vol_speaker, cm->me_out_accu) < 0)
		goto __error;
	/* AUX */
	if ((group = snd_mixer_lib_group_ctrl(mixer, SND_MIXER_IN_AUX, 0, SND_MIXER_OSS_LINE1, snd_cmipci_group_aux, cm)) == NULL)
		goto __error;
	if ((cm->me_in_aux = snd_mixer_lib_io_stereo(mixer, SND_MIXER_IN_AUX, 0, SND_MIXER_ETYPE_INPUT, 0)) == NULL)
		goto __error;
	if ((cm->me_vol_aux = snd_mixer_lib_volume1(mixer, "Aux Volume", 0, 2, db_range_15s, snd_cmipci_aux_volume_level, cm)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, cm->me_vol_aux) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, cm->me_in_aux, cm->me_vol_aux) < 0)
		goto __error;
	if ((cm->me_sw1_aux_output = snd_mixer_lib_sw1(mixer, "Aux Output Switch", 0, 2, snd_cmipci_aux_output_switch, cm)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, cm->me_sw1_aux_output) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, cm->me_vol_aux, cm->me_sw1_aux_output) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, cm->me_sw1_aux_output, cm->me_out_accu) < 0)
		goto __error;
	if ((cm->me_sw1_aux_input = snd_mixer_lib_sw1(mixer, "Aux Input Switch", 0, 2, snd_cmipci_aux_input_switch, cm)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, cm->me_sw1_aux_input) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, cm->me_vol_aux, cm->me_sw1_aux_input) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, cm->me_sw1_aux_input, cm->me_in_accu) < 0)
		goto __error;
	/* Capture endpoint */
	if ((cm->me_capture = snd_mixer_lib_pcm1(mixer, SND_MIXER_ELEMENT_CAPTURE, 0, SND_MIXER_ETYPE_CAPTURE1, 1, &pcm_device)) == NULL)
		goto __error;
	if (snd_mixer_element_route_add(mixer, cm->me_in_accu, cm->me_capture) < 0)
		goto __error;

	/* add mixer switches */
	sw = snd_cmipci_mixer_switches;
	for (i = 0; i < numberof(snd_cmipci_mixer_switches); i++, sw++)
		snd_mixer_switch_new(mixer, sw, cm);

	if (cm->card->type == SND_CARD_TYPE_CMI8738) {
		sw = snd_cmipci_8738_mixer_switches;
		for (i = 0; i < numberof(snd_cmipci_8738_mixer_switches); i++, sw++)
			snd_mixer_switch_new(mixer, sw, cm);
	}
	if (cm->chip_version >= 39) {
		sw = snd_cmipci_extra_mixer_switches;
		for (i = 0; i < numberof(snd_cmipci_extra_mixer_switches); i++, sw++)
			snd_mixer_switch_new(mixer, sw, cm);
	}

	cm->kmixer = mixer;
	return 0;

      __error:
	snd_device_free(cm->card, mixer);
	return -ENOMEM;
}


/*
 * universal switches
 */
static void snd_cmipci_switches_new(cmipci_t *cm)
{
	int i;
	snd_kswitch_t *sw;

	sw = snd_cmipci_control_switches;
	for (i = 0; i < numberof(snd_cmipci_control_switches); i++, sw++)
		snd_control_switch_new(cm->card, sw, cm);
}


/*
 * proc interface: dump register values
 */

static void snd_cmipci_proc_read(snd_info_buffer_t *buffer, void *private_data)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, private_data, );
	int i;
	
	snd_iprintf(buffer, "C-Media PCI\n\n");
	for (i = 0; i < 0x40; i++) {
		int v = inb(cm->iobase + i);
		if (i % 4 == 0)
			snd_iprintf(buffer, "%02x: ", i);
		snd_iprintf(buffer, "%02x", v);
		if (i % 4 == 3)
			snd_iprintf(buffer, "\n");
		else
			snd_iprintf(buffer, " ");
	}
}

static void __init snd_cmipci_proc_init(cmipci_t *cm)
{
	snd_info_entry_t *entry;

	if ((entry = snd_info_create_entry(cm->card, "cmipci")) != NULL) {
		entry->private_data = cm;
		entry->mode = S_IFREG | S_IRUGO | S_IWUSR;
		entry->t.text.read_size = 256;
		entry->t.text.read = snd_cmipci_proc_read;
		if (snd_info_register(entry) < 0) {
			snd_info_free_entry(entry);
			entry = NULL;
		}
	}
	cm->proc_entry = entry;
}

static void snd_cmipci_proc_done(cmipci_t *cm)
{
	if (cm->proc_entry) {
		snd_info_unregister(cm->proc_entry);
		cm->proc_entry = NULL;
	}
}


#ifndef PCI_VENDOR_ID_CMEDIA
#define PCI_VENDOR_ID_CMEDIA         0x13F6
#endif
#ifndef PCI_DEVICE_ID_CMEDIA_CM8338A
#define PCI_DEVICE_ID_CMEDIA_CM8338A 0x0100
#endif
#ifndef PCI_DEVICE_ID_CMEDIA_CM8338B
#define PCI_DEVICE_ID_CMEDIA_CM8338B 0x0101
#endif
#ifndef PCI_DEVICE_ID_CMEDIA_CM8738
#define PCI_DEVICE_ID_CMEDIA_CM8738  0x0111
#endif
#ifndef PCI_DEVICE_ID_CMEDIA_CM8738B
#define PCI_DEVICE_ID_CMEDIA_CM8738B 0x0112
#endif

static struct pci_device_id snd_cmipci_ids[] __devinitdata = {
	{PCI_VENDOR_ID_CMEDIA, PCI_DEVICE_ID_CMEDIA_CM8338A, PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0},
	{PCI_VENDOR_ID_CMEDIA, PCI_DEVICE_ID_CMEDIA_CM8338B, PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0},
	{PCI_VENDOR_ID_CMEDIA, PCI_DEVICE_ID_CMEDIA_CM8738, PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0},
	{PCI_VENDOR_ID_CMEDIA, PCI_DEVICE_ID_CMEDIA_CM8738B, PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0},
	{0,},
};


/*
 * check chip version and capabilities
 */
static void query_chip(cmipci_t *cm)
{
	unsigned int detect;

	/* check reg 0Ch, bit 24-31 */
	detect = snd_cmipci_read(cm, CM_REG_INT_HLDCLR) & CM_CHIP_MASK2;
	if (! detect) {
		/* check reg 08h, bit 24-28 */
		detect = snd_cmipci_read(cm, CM_REG_CHFORMAT) & CM_CHIP_MASK1;
		if (! detect) {
			cm->chip_version = 33;
			cm->max_channels = 4;
			cm->can_ac3_sw = 1;
			cm->has_dual_dac = 1;
		} else {
			cm->chip_version = 37;
			cm->max_channels = 4;
			cm->can_ac3_hw = 1;
			cm->has_dual_dac = 1;
		}
	} else {
		/* check reg 0Ch, bit 26 */
		if (detect & CM_CHIP_039) {
			cm->chip_version = 39;
			if (detect & CM_CHIP_039_6CH)
				cm->max_channels  = 6;
			else
				cm->max_channels = 4;
			cm->can_ac3_hw = 1;
			cm->has_dual_dac = 1;
			cm->can_multi_ch = 1;
		} else {
			cm->chip_version = 55; /* 4 or 6 channels */
			cm->max_channels  = 6;
			cm->can_ac3_hw = 1;
			cm->has_dual_dac = 1;
			cm->can_multi_ch = 1;
		}
	}
}


static int snd_cmipci_free(cmipci_t *cm)
{
	int i;

	snd_cmipci_proc_done(cm);

	if (cm->irqptr) {
		snd_cmipci_write(cm, CM_REG_INT_HLDCLR, 0);  /* disable ints */
		snd_cmipci_write(cm, CM_REG_FUNCTRL0, 0); /* disable channels */

		/* reset mixer */
		snd_cmipci_mixer_write(cm, 0, 0);

		synchronize_irq();
	}

	for (i = 0; i < 3; i++) {
		if (cm->hw_info[i])
			snd_kfree(cm->hw_info[i]);
	}

	snd_magic_kfree(cm);
	return 0;
}

static int snd_cmipci_dev_free(void *private_data)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, private_data, -ENXIO);
	return snd_cmipci_free(cm);
}

static int __init snd_cmipci_create(snd_card_t *card,
				    struct pci_dev *pci,
				    int playback_size, int capture_size,
				    int enable_midi,
				    unsigned long iomidi,
				    int midi_irq,
				    int enable_synth,
				    unsigned long iosynth,
				    cmipci_t **rcmipci)
{
	cmipci_t *cm;
	int err;
	static snd_device_ops_t ops = {
		dev_free: snd_cmipci_dev_free,
	};
	unsigned int val = 0;
	int pcm_index;
	
	*rcmipci = NULL;

	if ((err = pci_enable_device(pci)) < 0)
		return err;

	cm = snd_magic_kcalloc(cmipci_t, 0, GFP_KERNEL);
	if (cm == NULL)
		return -ENOMEM;

	spin_lock_init(&cm->reg_lock);
	init_MUTEX(&cm->open_mutex);
	cm->card = card;
	cm->pci = pci;
	cm->iobase = pci_resource_start(pci, 0);

	/* reserve resources */
	if (snd_register_ioport(card, cm->iobase,
				CM_EXTENT_CODEC, "C-Media PCI", NULL) < 0) {
		err = -EBUSY;
		goto __error;
	}
	if ((err = snd_register_interrupt(card,
					  "C-Media PCI", pci->irq,
					  SND_IRQ_TYPE_PCI, snd_cmipci_interrupt,
					  cm, NULL, &cm->irqptr)) < 0)
		goto __error;
	if ((err = snd_register_dma_channel(card,
					    "C-Media PCI - DAC1 frame", 0,
					    SND_DMA_TYPE_PCI, playback_size,
					    NULL, &cm->channel[0].dmaptr)) < 0)
		goto __error;
	if ((err = snd_register_dma_channel(card,
					    "C-Media PCI - ADC frame", 1,
					    SND_DMA_TYPE_PCI, capture_size,
					    NULL, &cm->channel[1].dmaptr)) < 0)
		goto __error;

	if (enable_midi) {
		if (snd_register_ioport(card, iomidi, CM_EXTENT_MIDI,
					"C-Media PCI MIDI", NULL) < 0) {
			snd_printk("cmipci: MIDI port is busy\n");
			enable_midi = 0;
		}
		if (midi_irq > 0) {
			if (snd_register_interrupt(card,
						   "C-Media MIDI", midi_irq,
						   SND_IRQ_TYPE_PCI,
						   snd_cmipci_midi_interrupt,
						   cm, NULL, &cm->midi_irqptr) < 0) {
				snd_printk("cmipci: MIDI irq is busy\n");
				enable_midi = 0;
			}
		}
	}

	if (enable_synth) {
		if (snd_register_ioport(card, iosynth, CM_EXTENT_SYNTH,
					"C-Media PCI FM", NULL) < 0) {
			snd_printk("cmipci: synth port is busy\n");
			enable_synth = 0;
		}
	}

	pci_set_master(cm->pci);

	/*
	 * check chip version, max channels and capabilities
	 */

	cm->chip_version = 0;
	cm->max_channels = 2;

	query_chip(cm);

	cm->ctrl = CM_CHADC1;	/* default FUNCNTRL0 */

	/* initialize codec registers */
	snd_cmipci_write(cm, CM_REG_INT_HLDCLR, 0);   /* disable ints */
	snd_cmipci_write(cm, CM_REG_FUNCTRL0, 0); 	/* disable channels */
	snd_cmipci_write(cm, CM_REG_FUNCTRL1, 0);

	snd_cmipci_write(cm, CM_REG_CHFORMAT, 0x00200000);
	snd_cmipci_clear_bit(cm, CM_REG_MISC_CTRL, CM_ENDBDAC);

	/* set MPU address */
	if (enable_midi) {
		switch (iomidi) {
		case 0x320: val = CM_VMPU_320; break;
		case 0x310: val = CM_VMPU_310; break;
		case 0x300: val = CM_VMPU_300; break;
		case 0x330: val = CM_VMPU_330; break;
		default:
			enable_midi = 0; break;
		}
		if (enable_midi) {
			snd_cmipci_write(cm, CM_REG_LEGACY_CTRL, val);
			/* enable UART */
			snd_cmipci_set_bit(cm, CM_REG_FUNCTRL1, CM_UART_EN);
		}
	}

	/* set FM address */
	if (enable_synth) {
		val = snd_cmipci_read(cm, CM_REG_LEGACY_CTRL) & ~CM_FMSEL_MASK;
		switch (iosynth) {
		case 0x3E8: val |= CM_FMSEL_3E8; break;
		case 0x3E0: val |= CM_FMSEL_3E0; break;
		case 0x3C8: val |= CM_FMSEL_3C8; break;
		case 0x388: val |= CM_FMSEL_388; break;
		default:
			enable_synth = 0; break;
		}
		if (enable_synth) {
			snd_cmipci_write(cm, CM_REG_LEGACY_CTRL, val);
			/* enable FM */
			snd_cmipci_set_bit(cm, CM_REG_MISC_CTRL, CM_FM_EN);
		}
	}

	/* reset mixer */
	snd_cmipci_mixer_write(cm, 0, 0);

	snd_cmipci_proc_init(cm);

	/* create mixer interface */
	if ((err = snd_cmipci_mixer_new(cm, 0)) < 0)
		goto __error;

	/* create pcm devices */
	pcm_index = 0;
	if ((err = snd_cmipci_pcm_new(cm, pcm_index)) < 0)
		goto __error;
	pcm_index++;
	if (cm->has_dual_dac) {
		if ((err = snd_cmipci_pcm2_new(cm, pcm_index)) < 0)
			goto __error;
		pcm_index++;
	}
	if (cm->can_ac3_hw || cm->can_ac3_sw) {
		if ((err = snd_cmipci_pcm_spdif_new(cm, pcm_index)) < 0)
			goto __error;
	}

	/* midi uart */
	if (enable_midi) {
		err = snd_mpu401_uart_new(card, 0, MPU401_HW_CMIPCI,
					  iomidi, midi_irq > 0 ? midi_irq : pci->irq,
					  &cm->rmidi);
		if (err < 0)
			goto __error;
	}

	/* create switches */
	snd_cmipci_switches_new(cm);

	if ((err = snd_device_new(card, SND_DEV_LOWLEVEL, cm, 0, &ops, NULL)) < 0)
		goto __error;

	*rcmipci = cm;
	return 0;

 __error:
	snd_cmipci_free(cm);
	return err;
}

/*
 */

MODULE_DEVICE_TABLE(pci, snd_cmipci_ids);

static void snd_cmipci_use_inc(snd_card_t *card)
{
	MOD_INC_USE_COUNT;
}

static void snd_cmipci_use_dec(snd_card_t *card)
{
	MOD_DEC_USE_COUNT;
}

static int __init snd_cmipci_probe(struct pci_dev *pci,
				   const struct pci_device_id *id)
{
	static int dev = 0;
	snd_card_t *card;
	cmipci_t *cm = NULL;
	int err;

	for (; dev < SND_CARDS; dev++) {
		if (! snd_enable[dev]) {
			dev++;
			return -ENOENT;
		}
		break;
	}
	if (dev >= SND_CARDS)
		return -ENODEV;

	card = snd_card_new(snd_index[dev], snd_id[dev],
			    snd_cmipci_use_inc, snd_cmipci_use_dec);
	if (card == NULL)
		return -ENOMEM;

	if (pci->device == PCI_DEVICE_ID_CMEDIA_CM8738)
		card->type = SND_CARD_TYPE_CMI8738;
	else
		card->type = SND_CARD_TYPE_CMI8338;

	strcpy(card->abbreviation, "C-Media PCI");
	sprintf(card->shortname, "C-Media PCI %s",
		card->type == SND_CARD_TYPE_CMI8738 ? "CM8738" : "CM8338");

	if ((err = snd_cmipci_create(card, pci,
				     snd_dac_frame_size[dev],
				     snd_adc_frame_size[dev],
				     snd_enable_midi[dev],
				     snd_mpu_port[dev],
				     snd_mpu_irq[dev],
				     snd_enable_fm[dev],
				     snd_fm_port[dev],
				     &cm)) < 0) {
		snd_card_free(card);
		return err;
	}

	sprintf(card->longname, "%s (model %d) at 0x%lx, irq %i",
		card->shortname,
		cm->chip_version,
		pci_resource_start(cm->pci, 0),
		pci->irq);

	snd_printd("%s is detected\n", card->longname);

	if ((err = snd_card_register(card)) < 0) {
		snd_card_free(card);
		return err;
	}

	PCI_SET_DRIVER_DATA(pci, card);
	dev++;
	return 0;
}

static void __exit snd_cmipci_remove(struct pci_dev *pci)
{
	snd_card_free(PCI_GET_DRIVER_DATA(pci));
	PCI_SET_DRIVER_DATA(pci, NULL);
}


static struct pci_driver driver = {
	name: "C-Media PCI",
	id_table: snd_cmipci_ids,
	probe: snd_cmipci_probe,
	remove: snd_cmipci_remove,
};
	
static int __init alsa_card_cmipci_init(void)
{
	int err;

	if ((err = pci_module_init(&driver)) < 0) {
#ifdef MODULE
		snd_printk("C-Media PCI soundcard not found or device busy\n");
#endif
		return err;
	}
	return 0;
}

static void __exit alsa_card_cmipci_exit(void)
{
	pci_unregister_driver(&driver);
}

module_init(alsa_card_cmipci_init)
module_exit(alsa_card_cmipci_exit)
