/*
 *  Copyright (c) by Jaroslav Kysela <perex@suse.cz>
 *  Routines for control of SoundBlaster cards - MIDI interface
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * --
 *
 * Sun May  9 22:54:38 BST 1999 George David Morrison <gdm@gedamo.demon.co.uk>
 *   Fixed typo in snd_sb8dsp_midi_new_device which prevented midi from 
 *   working.
 */

#include "../../include/driver.h"
#include "../../include/sb.h"

/*

 */

void snd_sb8dsp_midi_interrupt(snd_rawmidi_t * rmidi)
{
	unsigned long flags;
	sbdsp_t *codec;
	int max = 64;
	char byte;

	codec = snd_magic_cast(sbdsp_t, rmidi->private_data, );
	if (codec == NULL) {
		inb(SBP(codec, READ));	/* ack interrupt */
		return;
	}
	while (max-- > 0) {
		spin_lock_irqsave(&codec->midi_input_lock, flags);
		if (inb(SBP(codec, DATA_AVAIL)) & 0x80) {
			byte = inb(SBP(codec, READ));
			spin_unlock_irqrestore(&codec->midi_input_lock, flags);
			snd_rawmidi_receive(rmidi, &byte, 1);
		} else {
			spin_unlock_irqrestore(&codec->midi_input_lock, flags);
		}
	}
}

/*

 */

static int snd_sb8dsp_midi_input_open(snd_rawmidi_t * rmidi)
{
	unsigned long flags;
	sbdsp_t *codec;

	codec = snd_magic_cast(sbdsp_t, rmidi->private_data, -ENXIO);
	spin_lock_irqsave(&codec->open8_lock, flags);
	if (codec->open8) {
		spin_unlock_irqrestore(&codec->open8_lock, flags);
		return -EAGAIN;
	}
	codec->open8 |= SB_OPEN_MIDI_INPUT;
	if (!(codec->open8 & SB_OPEN_MIDI_OUTPUT)) {
		spin_unlock_irqrestore(&codec->open8_lock, flags);
		snd_sb8dsp_reset(codec);		/* reset DSP */
	} else {
		spin_unlock_irqrestore(&codec->open8_lock, flags);
	}
	return 0;
}

static int snd_sb8dsp_midi_output_open(snd_rawmidi_t * rmidi)
{
	unsigned long flags;
	sbdsp_t *codec;

	codec = snd_magic_cast(sbdsp_t, rmidi->private_data, -ENXIO);
	spin_lock_irqsave(&codec->open8_lock, flags);
	if (codec->open8) {
		spin_unlock_irqrestore(&codec->open8_lock, flags);
		return -EAGAIN;
	}
	codec->open8 |= SB_OPEN_MIDI_OUTPUT;
	if (!(codec->open8 & SB_OPEN_MIDI_INPUT)) {
		spin_unlock_irqrestore(&codec->open8_lock, flags);
		snd_sb8dsp_reset(codec);		/* reset DSP */
	} else {
		spin_unlock_irqrestore(&codec->open8_lock, flags);
	}
	return 0;
}

static int snd_sb8dsp_midi_input_close(snd_rawmidi_t * rmidi)
{
	unsigned long flags;
	sbdsp_t *codec;

	codec = snd_magic_cast(sbdsp_t, rmidi->private_data, -ENXIO);
	spin_lock_irqsave(&codec->open8_lock, flags);
	codec->open8 &= ~(SB_OPEN_MIDI_INPUT | SB_OPEN_MIDI_TRIGGER);
	if (!(codec->open8 & SB_OPEN_MIDI_OUTPUT)) {
		spin_unlock_irqrestore(&codec->open8_lock, flags);
		snd_sb8dsp_reset(codec);		/* reset DSP */
	} else {
		spin_unlock_irqrestore(&codec->open8_lock, flags);
	}
	return 0;
}

static int snd_sb8dsp_midi_output_close(snd_rawmidi_t * rmidi)
{
	unsigned long flags;
	sbdsp_t *codec;

	codec = snd_magic_cast(sbdsp_t, rmidi->private_data, -ENXIO);
	spin_lock_irqsave(&codec->open8_lock, flags);
	codec->open8 &= ~SB_OPEN_MIDI_OUTPUT;
	if (!(codec->open8 & SB_OPEN_MIDI_INPUT)) {
		spin_unlock_irqrestore(&codec->open8_lock, flags);
		snd_sb8dsp_reset(codec);		/* reset DSP */
	} else {
		spin_unlock_irqrestore(&codec->open8_lock, flags);
	}
	return 0;
}

static void snd_sb8dsp_midi_input_trigger(snd_rawmidi_t * rmidi, int up)
{
	unsigned long flags;
	sbdsp_t *codec;

	codec = snd_magic_cast(sbdsp_t, rmidi->private_data, );
	spin_lock_irqsave(&codec->open8_lock, flags);
	if (up) {
		if (!(codec->open8 & SB_OPEN_MIDI_TRIGGER)) {
			snd_sb8dsp_command(codec, SB_DSP_MIDI_INPUT_IRQ);
			codec->open8 |= SB_OPEN_MIDI_TRIGGER;
		}
	} else {
		if (codec->open8 & SB_OPEN_MIDI_TRIGGER) {
			snd_sb8dsp_command(codec, SB_DSP_MIDI_INPUT_IRQ);
			codec->open8 &= ~SB_OPEN_MIDI_TRIGGER;
		}
	}
	spin_unlock_irqrestore(&codec->open8_lock, flags);
}

static void snd_sb8dsp_midi_output_write(snd_rawmidi_t * rmidi)
{
	unsigned long flags;
	sbdsp_t *codec;
	char byte;
	int max = 32;

	/* how big is Tx FIFO? */
	codec = snd_magic_cast(sbdsp_t, rmidi->private_data, );
	while (max-- > 0) {
		spin_lock_irqsave(&codec->open8_lock, flags);
		if (snd_rawmidi_transmit(rmidi, &byte, 1) != 1) {
			codec->open8 &= ~SB_OPEN_MIDI_TRIGGER;
			del_timer(&codec->midi_timer);
			spin_unlock_irqrestore(&codec->open8_lock, flags);
			return;
		}
		snd_sb8dsp_command(codec, SB_DSP_MIDI_OUTPUT);
		snd_sb8dsp_command(codec, byte);
		spin_unlock_irqrestore(&codec->open8_lock, flags);
	}
}

static void snd_sb8dsp_midi_output_timer(unsigned long data)
{
	snd_rawmidi_t * rmidi = (snd_rawmidi_t *) data;
	sbdsp_t * codec = snd_magic_cast(sbdsp_t, rmidi->private_data, );
	unsigned long flags;

	spin_lock_irqsave(&codec->open8_lock, flags);
	codec->midi_timer.expires = 1 + jiffies;
	add_timer(&codec->midi_timer);
	spin_unlock_irqrestore(&codec->open8_lock, flags);	
	snd_sb8dsp_midi_output_write(rmidi);
}

static void snd_sb8dsp_midi_output_trigger(snd_rawmidi_t * rmidi, int up)
{
	unsigned long flags;
	sbdsp_t *codec;

	codec = snd_magic_cast(sbdsp_t, rmidi->private_data, );
	spin_lock_irqsave(&codec->open8_lock, flags);
	if (up) {
		if (!(codec->open8 & SB_OPEN_MIDI_TRIGGER)) {
			codec->midi_timer.function = snd_sb8dsp_midi_output_timer;
			codec->midi_timer.data = (unsigned long) rmidi;
			codec->midi_timer.expires = 1 + jiffies;
			add_timer(&codec->midi_timer);
			codec->open8 |= SB_OPEN_MIDI_TRIGGER;
		}
	} else {
		if (codec->open8 & SB_OPEN_MIDI_TRIGGER) {
			codec->open8 &= ~SB_OPEN_MIDI_TRIGGER;
		}
	}
	spin_unlock_irqrestore(&codec->open8_lock, flags);

	if (up)
		snd_sb8dsp_midi_output_write(rmidi);
}

/*

 */

static struct snd_stru_rawmidi_channel_hw snd_sb8dsp_midi_output =
{
	open:           snd_sb8dsp_midi_output_open,
	close:          snd_sb8dsp_midi_output_close,
	trigger:	snd_sb8dsp_midi_output_trigger,
};

static struct snd_stru_rawmidi_channel_hw snd_sb8dsp_midi_input =
{
	open:           snd_sb8dsp_midi_input_open,
	close:          snd_sb8dsp_midi_input_close,
	trigger:        snd_sb8dsp_midi_input_trigger,
};

int snd_sb8dsp_midi_new(sbdsp_t * codec, int device, snd_rawmidi_t ** rrawmidi)
{
	snd_rawmidi_t *rmidi;
	int err;

	*rrawmidi = NULL;
	if ((err = snd_rawmidi_new(codec->card, "SB8 MIDI", device, &rmidi)) < 0)
		return err;
	strcpy(rmidi->name, "SB8 MIDI");
	memcpy(&rmidi->chn[SND_RAWMIDI_CHANNEL_OUTPUT].hw, &snd_sb8dsp_midi_output, sizeof(snd_sb8dsp_midi_output));
	memcpy(&rmidi->chn[SND_RAWMIDI_CHANNEL_INPUT].hw, &snd_sb8dsp_midi_input, sizeof(snd_sb8dsp_midi_input));
	rmidi->info_flags |= SND_RAWMIDI_INFO_OUTPUT | SND_RAWMIDI_INFO_INPUT | SND_RAWMIDI_INFO_DUPLEX;
	rmidi->private_data = codec;
	*rrawmidi = rmidi;
	return 0;
}
