﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/rekognition/RekognitionRequest.h>
#include <aws/rekognition/Rekognition_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Rekognition {
namespace Model {

/**
 */
class PutProjectPolicyRequest : public RekognitionRequest {
 public:
  AWS_REKOGNITION_API PutProjectPolicyRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "PutProjectPolicy"; }

  AWS_REKOGNITION_API Aws::String SerializePayload() const override;

  AWS_REKOGNITION_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the project that the project policy is
   * attached to.</p>
   */
  inline const Aws::String& GetProjectArn() const { return m_projectArn; }
  inline bool ProjectArnHasBeenSet() const { return m_projectArnHasBeenSet; }
  template <typename ProjectArnT = Aws::String>
  void SetProjectArn(ProjectArnT&& value) {
    m_projectArnHasBeenSet = true;
    m_projectArn = std::forward<ProjectArnT>(value);
  }
  template <typename ProjectArnT = Aws::String>
  PutProjectPolicyRequest& WithProjectArn(ProjectArnT&& value) {
    SetProjectArn(std::forward<ProjectArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A name for the policy.</p>
   */
  inline const Aws::String& GetPolicyName() const { return m_policyName; }
  inline bool PolicyNameHasBeenSet() const { return m_policyNameHasBeenSet; }
  template <typename PolicyNameT = Aws::String>
  void SetPolicyName(PolicyNameT&& value) {
    m_policyNameHasBeenSet = true;
    m_policyName = std::forward<PolicyNameT>(value);
  }
  template <typename PolicyNameT = Aws::String>
  PutProjectPolicyRequest& WithPolicyName(PolicyNameT&& value) {
    SetPolicyName(std::forward<PolicyNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The revision ID for the Project Policy. Each time you modify a policy, Amazon
   * Rekognition Custom Labels generates and assigns a new
   * <code>PolicyRevisionId</code> and then deletes the previous version of the
   * policy.</p>
   */
  inline const Aws::String& GetPolicyRevisionId() const { return m_policyRevisionId; }
  inline bool PolicyRevisionIdHasBeenSet() const { return m_policyRevisionIdHasBeenSet; }
  template <typename PolicyRevisionIdT = Aws::String>
  void SetPolicyRevisionId(PolicyRevisionIdT&& value) {
    m_policyRevisionIdHasBeenSet = true;
    m_policyRevisionId = std::forward<PolicyRevisionIdT>(value);
  }
  template <typename PolicyRevisionIdT = Aws::String>
  PutProjectPolicyRequest& WithPolicyRevisionId(PolicyRevisionIdT&& value) {
    SetPolicyRevisionId(std::forward<PolicyRevisionIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A resource policy to add to the model. The policy is a JSON structure that
   * contains one or more statements that define the policy. The policy must follow
   * the IAM syntax. For more information about the contents of a JSON policy
   * document, see <a
   * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies.html">IAM
   * JSON policy reference</a>. </p>
   */
  inline const Aws::String& GetPolicyDocument() const { return m_policyDocument; }
  inline bool PolicyDocumentHasBeenSet() const { return m_policyDocumentHasBeenSet; }
  template <typename PolicyDocumentT = Aws::String>
  void SetPolicyDocument(PolicyDocumentT&& value) {
    m_policyDocumentHasBeenSet = true;
    m_policyDocument = std::forward<PolicyDocumentT>(value);
  }
  template <typename PolicyDocumentT = Aws::String>
  PutProjectPolicyRequest& WithPolicyDocument(PolicyDocumentT&& value) {
    SetPolicyDocument(std::forward<PolicyDocumentT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_projectArn;

  Aws::String m_policyName;

  Aws::String m_policyRevisionId;

  Aws::String m_policyDocument;
  bool m_projectArnHasBeenSet = false;
  bool m_policyNameHasBeenSet = false;
  bool m_policyRevisionIdHasBeenSet = false;
  bool m_policyDocumentHasBeenSet = false;
};

}  // namespace Model
}  // namespace Rekognition
}  // namespace Aws
