﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/cleanrooms/CleanRoomsRequest.h>
#include <aws/cleanrooms/CleanRooms_EXPORTS.h>
#include <aws/cleanrooms/model/ChangeInput.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace CleanRooms {
namespace Model {

/**
 */
class CreateCollaborationChangeRequestRequest : public CleanRoomsRequest {
 public:
  AWS_CLEANROOMS_API CreateCollaborationChangeRequestRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateCollaborationChangeRequest"; }

  AWS_CLEANROOMS_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The identifier of the collaboration that the change request is made
   * against.</p>
   */
  inline const Aws::String& GetCollaborationIdentifier() const { return m_collaborationIdentifier; }
  inline bool CollaborationIdentifierHasBeenSet() const { return m_collaborationIdentifierHasBeenSet; }
  template <typename CollaborationIdentifierT = Aws::String>
  void SetCollaborationIdentifier(CollaborationIdentifierT&& value) {
    m_collaborationIdentifierHasBeenSet = true;
    m_collaborationIdentifier = std::forward<CollaborationIdentifierT>(value);
  }
  template <typename CollaborationIdentifierT = Aws::String>
  CreateCollaborationChangeRequestRequest& WithCollaborationIdentifier(CollaborationIdentifierT&& value) {
    SetCollaborationIdentifier(std::forward<CollaborationIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The list of changes to apply to the collaboration. Each change specifies the
   * type of modification and the details of what should be changed.</p>
   */
  inline const Aws::Vector<ChangeInput>& GetChanges() const { return m_changes; }
  inline bool ChangesHasBeenSet() const { return m_changesHasBeenSet; }
  template <typename ChangesT = Aws::Vector<ChangeInput>>
  void SetChanges(ChangesT&& value) {
    m_changesHasBeenSet = true;
    m_changes = std::forward<ChangesT>(value);
  }
  template <typename ChangesT = Aws::Vector<ChangeInput>>
  CreateCollaborationChangeRequestRequest& WithChanges(ChangesT&& value) {
    SetChanges(std::forward<ChangesT>(value));
    return *this;
  }
  template <typename ChangesT = ChangeInput>
  CreateCollaborationChangeRequestRequest& AddChanges(ChangesT&& value) {
    m_changesHasBeenSet = true;
    m_changes.emplace_back(std::forward<ChangesT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_collaborationIdentifier;

  Aws::Vector<ChangeInput> m_changes;
  bool m_collaborationIdentifierHasBeenSet = false;
  bool m_changesHasBeenSet = false;
};

}  // namespace Model
}  // namespace CleanRooms
}  // namespace Aws
