
// Licensed to the Apache Software Foundation (ASF) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The ASF licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

const { generate_udf_test, generate_udaf_test } = unit_test_utils;

// using defaults

const theta_1 = `FROM_BASE64('AQQDPwEazJMDEIFfUcrcGW6ylF+DQ0nLOjDZ/9ze6gyQ')`;

generate_udaf_test("theta_sketch_agg_string", {
  input_columns: [`str`],
  input_rows: `SELECT * FROM UNNEST(['a', 'b', 'c']) AS str`,
  expected_output: theta_1
});

const theta_2 = `FROM_BASE64('AQQDPwEazJMDEIFfUcrcGW6aAe192ejCXiH3k2yKaRSQ')`;

generate_udaf_test("theta_sketch_agg_string", {
  input_columns: [`str`],
  input_rows: `SELECT * FROM UNNEST(['c', 'd', 'e']) AS str`,
  expected_output: theta_2
});

const theta_union_1 = `FROM_BASE64('AQQDPgEazJMFIQK+o5W4Mt5oB7X3Z6MJcYknIFaWEI3+GlXsNvTgWyADqD2ToYTc')`;

generate_udf_test("theta_sketch_union", [{
  inputs: [ theta_1, theta_2 ],
  expected_output: theta_union_1
}]);

generate_udf_test("theta_sketch_get_estimate", [{
  inputs: [ theta_union_1 ],
  expected_output: 5
}]);

generate_udf_test("theta_sketch_to_string", [{
  inputs: [ theta_union_1 ],
  expected_output: `'''### Theta sketch summary:
   num retained entries : 5
   seed hash            : 37836
   empty?               : false
   ordered?             : true
   estimation mode?     : false
   theta (fraction)     : 1
   theta (raw 64-bit)   : 9223372036854775807
   estimate             : 5
   lower bound 95% conf : 5
   upper bound 95% conf : 5
### End sketch summary
'''`
}]);

const theta_intersection = `FROM_BASE64('AQMDAAAazJO3DG7lqK9ACA==')`;

generate_udf_test("theta_sketch_intersection", [{
  inputs: [ theta_1, theta_2 ],
  expected_output: theta_intersection
}]);

generate_udf_test("theta_sketch_get_estimate", [{
  inputs: [ theta_intersection ],
  expected_output: 1
}]);

const theta_a_not_b = `FROM_BASE64('AQQDPwEazJMCacuPE2yA/wsYbP/ub3UGSA==')`;

generate_udf_test("theta_sketch_a_not_b", [{
  inputs: [ theta_1, theta_2 ],
  expected_output: theta_a_not_b
}]);

generate_udf_test("theta_sketch_get_estimate", [{
  inputs: [ theta_a_not_b ],
  expected_output: 2
}]);

generate_udf_test("theta_sketch_jaccard_similarity", [{
  inputs: [ theta_1, theta_2 ],
  expected_output: `STRUCT(0.2 AS lower_bound, 0.2 AS estimate, 0.2 AS upper_bound)`
}]);

const theta_3 = `FROM_BASE64('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')`;

generate_udaf_test("theta_sketch_agg_int64", {
  input_columns: [`value`],
  input_rows: `SELECT * FROM UNNEST(GENERATE_ARRAY(1, 10000, 1)) AS value`,
  expected_output: theta_3
});

const theta_4 = `FROM_BASE64('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')`;

generate_udaf_test("theta_sketch_agg_int64", {
  input_columns: [`value`],
  input_rows: `SELECT * FROM UNNEST(GENERATE_ARRAY(100000, 110000, 1)) AS value`,
  expected_output: theta_4
});

const theta_union_2 = `FROM_BASE64('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')`;

generate_udaf_test("theta_sketch_agg_union", {
  input_columns: [`sketch`],
  input_rows: `SELECT * FROM UNNEST([${theta_3}, ${theta_4}]) AS sketch`,
  expected_output: theta_union_2
});

generate_udf_test("theta_sketch_get_estimate_and_bounds", [{
  inputs: [ theta_union_2, 3 ],
  expected_output: `STRUCT(19736.541348415347 AS estimate, 18927.112205958525 AS lower_bound, 20580.437426810073 AS upper_bound)`
}]);

generate_udf_test("theta_sketch_get_theta", [{
  inputs: [ theta_union_2 ],
  expected_output: 0.20753382913916013
}]);

generate_udf_test("theta_sketch_get_num_retained", [{
  inputs: [ theta_union_2 ],
  expected_output: 4096
}]);


// using full signatures

const theta_8_111_09_1 = `FROM_BASE64('AgQDPgEajNMAAAAAADMzcwNbX0hyljVFUBHLpzFb/p08wnWFIBcXdIA=')`;

generate_udaf_test("theta_sketch_agg_string_lgk_seed_p", {
  input_columns: [`str`, 'STRUCT(8 AS lgk, 111 AS seed, 0.9 AS p) NOT AGGREGATE'],
  input_rows: `SELECT * FROM UNNEST(['a', 'b', 'c']) AS str`,
  expected_output: theta_8_111_09_1
});

const theta_8_111_09_2 = `FROM_BASE64('AgQDPwEajNMAAAAAADMzcwMtr6Q5SxqiqB2ct1zLfRY+Ix9R7oe55fg=')`;

generate_udaf_test("theta_sketch_agg_string_lgk_seed_p", {
  input_columns: [`str`, 'STRUCT(8 AS lgk, 111 AS seed, 0.9 AS p) NOT AGGREGATE'],
  input_rows: `SELECT * FROM UNNEST(['c', 'd', 'e']) AS str`,
  expected_output: theta_8_111_09_2
});

const theta_union_8_111_1 = `FROM_BASE64('AgQDPgEajNMAAAAAADMzcwVbX0hyljVFUBHLpzFb/p0xkpzZB0fW7yy/YrBjPQIWXpIeNbbo6qQ=')`;

generate_udf_test("theta_sketch_union_lgk_seed", [{
  inputs: [ theta_8_111_09_1, theta_8_111_09_2, 8, 111 ],
  expected_output: theta_union_8_111_1
}]);

generate_udf_test("theta_sketch_get_estimate_seed", [{
  inputs: [ theta_union_8_111_1, 111 ],
  expected_output: 5.5555557027275215
}]);

generate_udf_test("theta_sketch_to_string_seed", [{
  inputs: [ theta_union_8_111_1, 111 ],
  expected_output: `'''### Theta sketch summary:
   num retained entries : 5
   seed hash            : 54156
   empty?               : false
   ordered?             : true
   estimation mode?     : true
   theta (fraction)     : 0.9
   theta (raw 64-bit)   : 8301034613266972672
   estimate             : 5.55556
   lower bound 95% conf : 5
   upper bound 95% conf : 9
### End sketch summary
'''`
}]);

const theta_intersection_111 = `FROM_BASE64('AgQDPQEajNMAAAAAADMzcwG2vpDlLGqKoA==')`;

generate_udf_test("theta_sketch_intersection_seed", [{
  inputs: [ theta_8_111_09_1, theta_8_111_09_2, 111 ],
  expected_output: theta_intersection_111
}]);

generate_udf_test("theta_sketch_get_estimate_seed", [{
  inputs: [ theta_intersection_111, 111 ],
  expected_output: 1.1111111405455043
}]);

const theta_a_not_b_111 = `FROM_BASE64('AgQDPgEajNMAAAAAADMzcwJf0jI+7TTsnzCdYUgFxd0g')`;

generate_udf_test("theta_sketch_a_not_b_seed", [{
  inputs: [ theta_8_111_09_1, theta_8_111_09_2, 111 ],
  expected_output: theta_a_not_b_111
}]);

generate_udf_test("theta_sketch_get_estimate_seed", [{
  inputs: [ theta_a_not_b_111, 111 ],
  expected_output: 2.2222222810910086
}]);

generate_udf_test("theta_sketch_jaccard_similarity_seed", [{
  inputs: [ theta_8_111_09_1, theta_8_111_09_2, 111 ],
  expected_output: `STRUCT(0.05868247546115801 AS lower_bound, 0.2 AS estimate, 0.4517325934817119 AS upper_bound)`
}]);

const theta_8_111_09_3 = `FROM_BASE64('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')`;

generate_udaf_test("theta_sketch_agg_int64_lgk_seed_p", {
  input_columns: [`value`, 'STRUCT(8 AS lgk, 111 AS seed, 0.9 AS p) NOT AGGREGATE'],
  input_rows: `SELECT * FROM UNNEST(GENERATE_ARRAY(1, 10000, 1)) AS value`,
  expected_output: theta_8_111_09_3
});

const theta_8_111_09_4 = `FROM_BASE64('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')`;

generate_udaf_test("theta_sketch_agg_int64_lgk_seed_p", {
  input_columns: [`value`, 'STRUCT(8 AS lgk, 111 AS seed, 0.9 AS p) NOT AGGREGATE'],
  input_rows: `SELECT * FROM UNNEST(GENERATE_ARRAY(100000, 110000, 1)) AS value`,
  expected_output: theta_8_111_09_4
});

const theta_union_8_111_09_2 = `FROM_BASE64('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')`;

generate_udaf_test("theta_sketch_agg_union_lgk_seed", {
  input_columns: [`sketch`, 'STRUCT(8 AS lgk, 111 AS seed) NOT AGGREGATE'],
  input_rows: `SELECT * FROM UNNEST([${theta_8_111_09_3}, ${theta_8_111_09_4}]) AS sketch`,
  expected_output: theta_union_8_111_09_2
});

generate_udf_test("theta_sketch_get_estimate_and_bounds_seed", [{
  inputs: [ theta_union_8_111_09_2, 3, 111 ],
  expected_output: `STRUCT(22034.160662067967 AS estimate, 18252.303584500878 AS lower_bound, 26589.643724271038 AS upper_bound)`
}]);

generate_udf_test("theta_sketch_get_theta_seed", [{
  inputs: [ theta_union_8_111_09_2, 111 ],
  expected_output: 0.011618323199426725
}]);

generate_udf_test("theta_sketch_get_num_retained_seed", [{
  inputs: [ theta_union_8_111_09_2, 111 ],
  expected_output: 256
}]);
