#!/usr/bin/env bash
#
# ignite3 Bash Completion
# =======================
#
# Bash completion support for the `ignite3` command,
# generated by [picocli](https://picocli.info/) version 4.7.5.
#
# Installation
# ------------
#
# 1. Source all completion scripts in your .bash_profile
#
#   cd $YOUR_APP_HOME/bin
#   for f in $(find . -name "*_completion"); do line=". $(pwd)/$f"; grep "$line" ~/.bash_profile || echo "$line" >> ~/.bash_profile; done
#
# 2. Open a new bash console, and type `ignite3 [TAB][TAB]`
#
# 1a. Alternatively, if you have [bash-completion](https://github.com/scop/bash-completion) installed:
#     Place this file in a `bash-completion.d` folder:
#
#   * /etc/bash-completion.d
#   * /usr/local/etc/bash-completion.d
#   * ~/bash-completion.d
#
# Documentation
# -------------
# The script is called by bash whenever [TAB] or [TAB][TAB] is pressed after
# 'ignite3 (..)'. By reading entered command line parameters,
# it determines possible bash completions and writes them to the COMPREPLY variable.
# Bash then completes the user input if only one entry is listed in the variable or
# shows the options if more than one is listed in COMPREPLY.
#
# References
# ----------
# [1] http://stackoverflow.com/a/12495480/1440785
# [2] http://tiswww.case.edu/php/chet/bash/FAQ
# [3] https://www.gnu.org/software/bash/manual/html_node/The-Shopt-Builtin.html
# [4] http://zsh.sourceforge.net/Doc/Release/Options.html#index-COMPLETE_005fALIASES
# [5] https://stackoverflow.com/questions/17042057/bash-check-element-in-array-for-elements-in-another-array/17042655#17042655
# [6] https://www.gnu.org/software/bash/manual/html_node/Programmable-Completion.html#Programmable-Completion
# [7] https://stackoverflow.com/questions/3249432/can-a-bash-tab-completion-script-be-used-in-zsh/27853970#27853970
#

if [ -n "$BASH_VERSION" ]; then
  # Enable programmable completion facilities when using bash (see [3])
  shopt -s progcomp
elif [ -n "$ZSH_VERSION" ]; then
  # Make alias a distinct command for completion purposes when using zsh (see [4])
  setopt COMPLETE_ALIASES
  alias compopt=complete

  # Enable bash completion in zsh (see [7])
  # Only initialize completions module once to avoid unregistering existing completions.
  if ! type compdef > /dev/null; then
    autoload -U +X compinit && compinit
  fi
  autoload -U +X bashcompinit && bashcompinit
fi

# CompWordsContainsArray takes an array and then checks
# if all elements of this array are in the global COMP_WORDS array.
#
# Returns zero (no error) if all elements of the array are in the COMP_WORDS array,
# otherwise returns 1 (error).
function CompWordsContainsArray() {
  declare -a localArray
  localArray=("$@")
  local findme
  for findme in "${localArray[@]}"; do
    if ElementNotInCompWords "$findme"; then return 1; fi
  done
  return 0
}
function ElementNotInCompWords() {
  local findme="$1"
  local element
  for element in "${COMP_WORDS[@]}"; do
    if [[ "$findme" = "$element" ]]; then return 1; fi
  done
  return 0
}

# The `currentPositionalIndex` function calculates the index of the current positional parameter.
#
# currentPositionalIndex takes three parameters:
# the command name,
# a space-separated string with the names of options that take a parameter, and
# a space-separated string with the names of boolean options (that don't take any params).
# When done, this function echos the current positional index to std_out.
#
# Example usage:
# local currIndex=$(currentPositionalIndex "mysubcommand" "$ARG_OPTS" "$FLAG_OPTS")
function currentPositionalIndex() {
  local commandName="$1"
  local optionsWithArgs="$2"
  local booleanOptions="$3"
  local previousWord
  local result=0

  for i in $(seq $((COMP_CWORD - 1)) -1 0); do
    previousWord=${COMP_WORDS[i]}
    if [ "${previousWord}" = "$commandName" ]; then
      break
    fi
    if [[ "${optionsWithArgs}" =~ ${previousWord} ]]; then
      ((result-=2)) # Arg option and its value not counted as positional param
    elif [[ "${booleanOptions}" =~ ${previousWord} ]]; then
      ((result-=1)) # Flag option itself not counted as positional param
    fi
    ((result++))
  done
  echo "$result"
}

# compReplyArray generates a list of completion suggestions based on an array, ensuring all values are properly escaped.
#
# compReplyArray takes a single parameter: the array of options to be displayed
#
# The output is echoed to std_out, one option per line.
#
# Example usage:
# local options=("foo", "bar", "baz")
# local IFS=$'\n'
# COMPREPLY=($(compReplyArray "${options[@]}"))
function compReplyArray() {
  declare -a options
  options=("$@")
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local i
  local quoted
  local optionList=()

  for (( i=0; i<${#options[@]}; i++ )); do
    # Double escape, since we want escaped values, but compgen -W expands the argument
    printf -v quoted %q "${options[i]}"
    quoted=\'${quoted//\'/\'\\\'\'}\'

    optionList[i]=$quoted
  done

  # We also have to add another round of escaping to $curr_word.
  curr_word=${curr_word//\\/\\\\}
  curr_word=${curr_word//\'/\\\'}

  # Actually generate completions.
  local IFS=$'\n'
  echo -e "$(compgen -W "${optionList[*]}" -- "$curr_word")"
}

# Bash completion entry point function.
# _complete_ignite3 finds which commands and subcommands have been specified
# on the command line and delegates to the appropriate function
# to generate possible options and subcommands for the last specified subcommand.
function _complete_ignite3() {
  # Edge case: if command line has no space after subcommand, then don't assume this subcommand is selected (remkop/picocli#1468).
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} sql" ];    then _picocli_ignite3; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} help" ];    then _picocli_ignite3; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} cli" ];    then _picocli_ignite3; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} connect" ];    then _picocli_ignite3; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} node" ];    then _picocli_ignite3; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} cluster" ];    then _picocli_ignite3; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} recovery" ];    then _picocli_ignite3; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} cli config" ];    then _picocli_ignite3_cli; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} cli config get" ];    then _picocli_ignite3_cli_config; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} cli config set" ];    then _picocli_ignite3_cli_config; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} cli config remove" ];    then _picocli_ignite3_cli_config; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} cli config show" ];    then _picocli_ignite3_cli_config; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} cli config profile" ];    then _picocli_ignite3_cli_config; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} cli config profile create" ];    then _picocli_ignite3_cli_config_profile; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} cli config profile activate" ];    then _picocli_ignite3_cli_config_profile; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} cli config profile list" ];    then _picocli_ignite3_cli_config_profile; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} cli config profile show" ];    then _picocli_ignite3_cli_config_profile; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} node config" ];    then _picocli_ignite3_node; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} node status" ];    then _picocli_ignite3_node; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} node version" ];    then _picocli_ignite3_node; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} node metric" ];    then _picocli_ignite3_node; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} node unit" ];    then _picocli_ignite3_node; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} node config show" ];    then _picocli_ignite3_node_config; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} node config update" ];    then _picocli_ignite3_node_config; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} node metric list" ];    then _picocli_ignite3_node_metric; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} node metric source" ];    then _picocli_ignite3_node_metric; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} node metric source enable" ];    then _picocli_ignite3_node_metric_source; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} node metric source disable" ];    then _picocli_ignite3_node_metric_source; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} node metric source list" ];    then _picocli_ignite3_node_metric_source; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} node unit list" ];    then _picocli_ignite3_node_unit; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} cluster config" ];    then _picocli_ignite3_cluster; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} cluster init" ];    then _picocli_ignite3_cluster; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} cluster status" ];    then _picocli_ignite3_cluster; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} cluster show" ];    then _picocli_ignite3_cluster; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} cluster topology" ];    then _picocli_ignite3_cluster; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} cluster unit" ];    then _picocli_ignite3_cluster; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} cluster metric" ];    then _picocli_ignite3_cluster; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} cluster config show" ];    then _picocli_ignite3_cluster_config; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} cluster config update" ];    then _picocli_ignite3_cluster_config; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} cluster topology physical" ];    then _picocli_ignite3_cluster_topology; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} cluster topology logical" ];    then _picocli_ignite3_cluster_topology; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} cluster unit deploy" ];    then _picocli_ignite3_cluster_unit; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} cluster unit undeploy" ];    then _picocli_ignite3_cluster_unit; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} cluster unit list" ];    then _picocli_ignite3_cluster_unit; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} cluster metric source" ];    then _picocli_ignite3_cluster_metric; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} cluster metric source enable" ];    then _picocli_ignite3_cluster_metric_source; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} cluster metric source disable" ];    then _picocli_ignite3_cluster_metric_source; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} cluster metric source list" ];    then _picocli_ignite3_cluster_metric_source; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} recovery partitions" ];    then _picocli_ignite3_recovery; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} recovery cluster" ];    then _picocli_ignite3_recovery; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} recovery partitions restart" ];    then _picocli_ignite3_recovery_partitions; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} recovery partitions reset" ];    then _picocli_ignite3_recovery_partitions; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} recovery partitions states" ];    then _picocli_ignite3_recovery_partitions; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} recovery cluster reset" ];    then _picocli_ignite3_recovery_cluster; return $?; fi
  if [ "${COMP_LINE}" = "${COMP_WORDS[0]} recovery cluster migrate" ];    then _picocli_ignite3_recovery_cluster; return $?; fi

  # Find the longest sequence of subcommands and call the bash function for that subcommand.
  local cmds0=(sql)
  local cmds1=(help)
  local cmds2=(cli)
  local cmds3=(connect)
  local cmds4=(node)
  local cmds5=(cluster)
  local cmds6=(recovery)
  local cmds7=(cli config)
  local cmds8=(cli config get)
  local cmds9=(cli config set)
  local cmds10=(cli config remove)
  local cmds11=(cli config show)
  local cmds12=(cli config profile)
  local cmds13=(cli config profile create)
  local cmds14=(cli config profile activate)
  local cmds15=(cli config profile list)
  local cmds16=(cli config profile show)
  local cmds17=(node config)
  local cmds18=(node status)
  local cmds19=(node version)
  local cmds20=(node metric)
  local cmds21=(node unit)
  local cmds22=(node config show)
  local cmds23=(node config update)
  local cmds24=(node metric list)
  local cmds25=(node metric source)
  local cmds26=(node metric source enable)
  local cmds27=(node metric source disable)
  local cmds28=(node metric source list)
  local cmds29=(node unit list)
  local cmds30=(cluster config)
  local cmds31=(cluster init)
  local cmds32=(cluster status)
  local cmds33=(cluster show)
  local cmds34=(cluster topology)
  local cmds35=(cluster unit)
  local cmds36=(cluster metric)
  local cmds37=(cluster config show)
  local cmds38=(cluster config update)
  local cmds39=(cluster topology physical)
  local cmds40=(cluster topology logical)
  local cmds41=(cluster unit deploy)
  local cmds42=(cluster unit undeploy)
  local cmds43=(cluster unit list)
  local cmds44=(cluster metric source)
  local cmds45=(cluster metric source enable)
  local cmds46=(cluster metric source disable)
  local cmds47=(cluster metric source list)
  local cmds48=(recovery partitions)
  local cmds49=(recovery cluster)
  local cmds50=(recovery partitions restart)
  local cmds51=(recovery partitions reset)
  local cmds52=(recovery partitions states)
  local cmds53=(recovery cluster reset)
  local cmds54=(recovery cluster migrate)

  if CompWordsContainsArray "${cmds54[@]}"; then _picocli_ignite3_recovery_cluster_migrate; return $?; fi
  if CompWordsContainsArray "${cmds53[@]}"; then _picocli_ignite3_recovery_cluster_reset; return $?; fi
  if CompWordsContainsArray "${cmds52[@]}"; then _picocli_ignite3_recovery_partitions_states; return $?; fi
  if CompWordsContainsArray "${cmds51[@]}"; then _picocli_ignite3_recovery_partitions_reset; return $?; fi
  if CompWordsContainsArray "${cmds50[@]}"; then _picocli_ignite3_recovery_partitions_restart; return $?; fi
  if CompWordsContainsArray "${cmds49[@]}"; then _picocli_ignite3_recovery_cluster; return $?; fi
  if CompWordsContainsArray "${cmds48[@]}"; then _picocli_ignite3_recovery_partitions; return $?; fi
  if CompWordsContainsArray "${cmds47[@]}"; then _picocli_ignite3_cluster_metric_source_list; return $?; fi
  if CompWordsContainsArray "${cmds46[@]}"; then _picocli_ignite3_cluster_metric_source_disable; return $?; fi
  if CompWordsContainsArray "${cmds45[@]}"; then _picocli_ignite3_cluster_metric_source_enable; return $?; fi
  if CompWordsContainsArray "${cmds44[@]}"; then _picocli_ignite3_cluster_metric_source; return $?; fi
  if CompWordsContainsArray "${cmds43[@]}"; then _picocli_ignite3_cluster_unit_list; return $?; fi
  if CompWordsContainsArray "${cmds42[@]}"; then _picocli_ignite3_cluster_unit_undeploy; return $?; fi
  if CompWordsContainsArray "${cmds41[@]}"; then _picocli_ignite3_cluster_unit_deploy; return $?; fi
  if CompWordsContainsArray "${cmds40[@]}"; then _picocli_ignite3_cluster_topology_logical; return $?; fi
  if CompWordsContainsArray "${cmds39[@]}"; then _picocli_ignite3_cluster_topology_physical; return $?; fi
  if CompWordsContainsArray "${cmds38[@]}"; then _picocli_ignite3_cluster_config_update; return $?; fi
  if CompWordsContainsArray "${cmds37[@]}"; then _picocli_ignite3_cluster_config_show; return $?; fi
  if CompWordsContainsArray "${cmds36[@]}"; then _picocli_ignite3_cluster_metric; return $?; fi
  if CompWordsContainsArray "${cmds35[@]}"; then _picocli_ignite3_cluster_unit; return $?; fi
  if CompWordsContainsArray "${cmds34[@]}"; then _picocli_ignite3_cluster_topology; return $?; fi
  if CompWordsContainsArray "${cmds33[@]}"; then _picocli_ignite3_cluster_show; return $?; fi
  if CompWordsContainsArray "${cmds32[@]}"; then _picocli_ignite3_cluster_status; return $?; fi
  if CompWordsContainsArray "${cmds31[@]}"; then _picocli_ignite3_cluster_init; return $?; fi
  if CompWordsContainsArray "${cmds30[@]}"; then _picocli_ignite3_cluster_config; return $?; fi
  if CompWordsContainsArray "${cmds29[@]}"; then _picocli_ignite3_node_unit_list; return $?; fi
  if CompWordsContainsArray "${cmds28[@]}"; then _picocli_ignite3_node_metric_source_list; return $?; fi
  if CompWordsContainsArray "${cmds27[@]}"; then _picocli_ignite3_node_metric_source_disable; return $?; fi
  if CompWordsContainsArray "${cmds26[@]}"; then _picocli_ignite3_node_metric_source_enable; return $?; fi
  if CompWordsContainsArray "${cmds25[@]}"; then _picocli_ignite3_node_metric_source; return $?; fi
  if CompWordsContainsArray "${cmds24[@]}"; then _picocli_ignite3_node_metric_list; return $?; fi
  if CompWordsContainsArray "${cmds23[@]}"; then _picocli_ignite3_node_config_update; return $?; fi
  if CompWordsContainsArray "${cmds22[@]}"; then _picocli_ignite3_node_config_show; return $?; fi
  if CompWordsContainsArray "${cmds21[@]}"; then _picocli_ignite3_node_unit; return $?; fi
  if CompWordsContainsArray "${cmds20[@]}"; then _picocli_ignite3_node_metric; return $?; fi
  if CompWordsContainsArray "${cmds19[@]}"; then _picocli_ignite3_node_version; return $?; fi
  if CompWordsContainsArray "${cmds18[@]}"; then _picocli_ignite3_node_status; return $?; fi
  if CompWordsContainsArray "${cmds17[@]}"; then _picocli_ignite3_node_config; return $?; fi
  if CompWordsContainsArray "${cmds16[@]}"; then _picocli_ignite3_cli_config_profile_show; return $?; fi
  if CompWordsContainsArray "${cmds15[@]}"; then _picocli_ignite3_cli_config_profile_list; return $?; fi
  if CompWordsContainsArray "${cmds14[@]}"; then _picocli_ignite3_cli_config_profile_activate; return $?; fi
  if CompWordsContainsArray "${cmds13[@]}"; then _picocli_ignite3_cli_config_profile_create; return $?; fi
  if CompWordsContainsArray "${cmds12[@]}"; then _picocli_ignite3_cli_config_profile; return $?; fi
  if CompWordsContainsArray "${cmds11[@]}"; then _picocli_ignite3_cli_config_show; return $?; fi
  if CompWordsContainsArray "${cmds10[@]}"; then _picocli_ignite3_cli_config_remove; return $?; fi
  if CompWordsContainsArray "${cmds9[@]}"; then _picocli_ignite3_cli_config_set; return $?; fi
  if CompWordsContainsArray "${cmds8[@]}"; then _picocli_ignite3_cli_config_get; return $?; fi
  if CompWordsContainsArray "${cmds7[@]}"; then _picocli_ignite3_cli_config; return $?; fi
  if CompWordsContainsArray "${cmds6[@]}"; then _picocli_ignite3_recovery; return $?; fi
  if CompWordsContainsArray "${cmds5[@]}"; then _picocli_ignite3_cluster; return $?; fi
  if CompWordsContainsArray "${cmds4[@]}"; then _picocli_ignite3_node; return $?; fi
  if CompWordsContainsArray "${cmds3[@]}"; then _picocli_ignite3_connect; return $?; fi
  if CompWordsContainsArray "${cmds2[@]}"; then _picocli_ignite3_cli; return $?; fi
  if CompWordsContainsArray "${cmds1[@]}"; then _picocli_ignite3_help; return $?; fi
  if CompWordsContainsArray "${cmds0[@]}"; then _picocli_ignite3_sql; return $?; fi

  # No subcommands were specified; generate completions for the top-level command.
  _picocli_ignite3; return $?;
}

# Generates completions for the options and subcommands of the `ignite3` command.
function _picocli_ignite3() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands="sql help cli connect node cluster recovery"
  local flag_opts="--help -h --version"
  local arg_opts="--verbose -v"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `sql` subcommand.
function _picocli_ignite3_sql() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands=""
  local flag_opts="--help -h --plain"
  local arg_opts="--verbose -v --jdbc-url --file"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
    --jdbc-url)
      return
      ;;
    --file)
      local IFS=$'\n'
      type compopt &>/dev/null && compopt -o filenames
      COMPREPLY=( $( compgen -f -- "${curr_word}" ) ) # files
      return $?
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `help` subcommand.
function _picocli_ignite3_help() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}

  local commands="sql cli connect node cluster recovery"
  local flag_opts="-h --help"
  local arg_opts=""

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `cli` subcommand.
function _picocli_ignite3_cli() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands="config"
  local flag_opts="--help -h"
  local arg_opts="--verbose -v"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `connect` subcommand.
function _picocli_ignite3_connect() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands=""
  local flag_opts="--help -h"
  local arg_opts="--verbose -v --username -u --password -p"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
    --username|-u)
      return
      ;;
    --password|-p)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `node` subcommand.
function _picocli_ignite3_node() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands="config status version metric unit"
  local flag_opts="--help -h"
  local arg_opts="--verbose -v"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `cluster` subcommand.
function _picocli_ignite3_cluster() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands="config init status show topology unit metric"
  local flag_opts="--help -h"
  local arg_opts="--verbose -v"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `recovery` subcommand.
function _picocli_ignite3_recovery() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands="partitions cluster"
  local flag_opts="--help -h"
  local arg_opts="--verbose -v"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `config` subcommand.
function _picocli_ignite3_cli_config() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands="get set remove show profile"
  local flag_opts="--help -h"
  local arg_opts="--verbose -v"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `get` subcommand.
function _picocli_ignite3_cli_config_get() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands=""
  local flag_opts="--help -h"
  local arg_opts="--verbose -v --profile"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
    --profile)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `set` subcommand.
function _picocli_ignite3_cli_config_set() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands=""
  local flag_opts="--help -h"
  local arg_opts="--verbose -v --profile"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
    --profile)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `remove` subcommand.
function _picocli_ignite3_cli_config_remove() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands=""
  local flag_opts="--help -h"
  local arg_opts="--verbose -v --profile"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
    --profile)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `show` subcommand.
function _picocli_ignite3_cli_config_show() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands=""
  local flag_opts="--help -h"
  local arg_opts="--verbose -v --profile"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
    --profile)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `profile` subcommand.
function _picocli_ignite3_cli_config_profile() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands="create activate list show"
  local flag_opts="--help -h"
  local arg_opts="--verbose -v"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `create` subcommand.
function _picocli_ignite3_cli_config_profile_create() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands=""
  local flag_opts="--help -h --activate"
  local arg_opts="--verbose -v --copy-from"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
    --copy-from)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `activate` subcommand.
function _picocli_ignite3_cli_config_profile_activate() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands=""
  local flag_opts="--help -h"
  local arg_opts="--verbose -v"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `list` subcommand.
function _picocli_ignite3_cli_config_profile_list() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands=""
  local flag_opts="--help -h"
  local arg_opts="--verbose -v"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `show` subcommand.
function _picocli_ignite3_cli_config_profile_show() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands=""
  local flag_opts="--help -h"
  local arg_opts="--verbose -v"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `config` subcommand.
function _picocli_ignite3_node_config() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands="show update"
  local flag_opts="--help -h"
  local arg_opts="--verbose -v"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `status` subcommand.
function _picocli_ignite3_node_status() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands=""
  local flag_opts="--help -h"
  local arg_opts="--verbose -v --url --profile"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
    --url)
      return
      ;;
    --profile)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `version` subcommand.
function _picocli_ignite3_node_version() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands=""
  local flag_opts="--help -h"
  local arg_opts="--verbose -v --url --profile"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
    --url)
      return
      ;;
    --profile)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `metric` subcommand.
function _picocli_ignite3_node_metric() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands="list source"
  local flag_opts="--help -h"
  local arg_opts="--verbose -v"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `unit` subcommand.
function _picocli_ignite3_node_unit() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands="list"
  local flag_opts="--help -h"
  local arg_opts="--verbose -v"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `show` subcommand.
function _picocli_ignite3_node_config_show() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands=""
  local flag_opts="--help -h"
  local arg_opts="--verbose -v --url --profile --format"
  local format_option_args=("JSON" "HOCON") # --format values

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
    --url)
      return
      ;;
    --profile)
      return
      ;;
    --format)
      local IFS=$'\n'
      COMPREPLY=( $( compReplyArray "${format_option_args[@]}" ) )
      return $?
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `update` subcommand.
function _picocli_ignite3_node_config_update() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands=""
  local flag_opts="--help -h"
  local arg_opts="--verbose -v --url --profile"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
    --url)
      return
      ;;
    --profile)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `list` subcommand.
function _picocli_ignite3_node_metric_list() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands=""
  local flag_opts="--help -h --plain"
  local arg_opts="--verbose -v --url --profile"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
    --url)
      return
      ;;
    --profile)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `source` subcommand.
function _picocli_ignite3_node_metric_source() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands="enable disable list"
  local flag_opts="--help -h"
  local arg_opts="--verbose -v"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `enable` subcommand.
function _picocli_ignite3_node_metric_source_enable() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands=""
  local flag_opts="--help -h"
  local arg_opts="--verbose -v --url --profile"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
    --url)
      return
      ;;
    --profile)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `disable` subcommand.
function _picocli_ignite3_node_metric_source_disable() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands=""
  local flag_opts="--help -h"
  local arg_opts="--verbose -v --url --profile"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
    --url)
      return
      ;;
    --profile)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `list` subcommand.
function _picocli_ignite3_node_metric_source_list() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands=""
  local flag_opts="--help -h --plain"
  local arg_opts="--verbose -v --url --profile"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
    --url)
      return
      ;;
    --profile)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `list` subcommand.
function _picocli_ignite3_node_unit_list() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands=""
  local flag_opts="--help -h --plain"
  local arg_opts="--verbose -v --version --status --url --profile"
  local statuses_option_args=("UPLOADING" "DEPLOYED" "OBSOLETE" "REMOVING") # --status values

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
    --version)
      return
      ;;
    --status)
      local IFS=$'\n'
      COMPREPLY=( $( compReplyArray "${statuses_option_args[@]}" ) )
      return $?
      ;;
    --url)
      return
      ;;
    --profile)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `config` subcommand.
function _picocli_ignite3_cluster_config() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands="show update"
  local flag_opts="--help -h"
  local arg_opts="--verbose -v"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `init` subcommand.
function _picocli_ignite3_cluster_init() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands=""
  local flag_opts="--help -h"
  local arg_opts="--verbose -v --config --config-files --name --metastorage-group --cluster-management-group --url --profile"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
    --config)
      return
      ;;
    --config-files)
      local IFS=$'\n'
      type compopt &>/dev/null && compopt -o filenames
      COMPREPLY=( $( compgen -f -- "${curr_word}" ) ) # files
      return $?
      ;;
    --name)
      return
      ;;
    --metastorage-group)
      return
      ;;
    --cluster-management-group)
      return
      ;;
    --url)
      return
      ;;
    --profile)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `status` subcommand.
function _picocli_ignite3_cluster_status() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands=""
  local flag_opts="--help -h"
  local arg_opts="--verbose -v --url --profile"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
    --url)
      return
      ;;
    --profile)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `show` subcommand.
function _picocli_ignite3_cluster_show() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands=""
  local flag_opts="--help -h"
  local arg_opts="--verbose -v --url --profile"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
    --url)
      return
      ;;
    --profile)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `topology` subcommand.
function _picocli_ignite3_cluster_topology() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands="physical logical"
  local flag_opts="--help -h"
  local arg_opts="--verbose -v"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `unit` subcommand.
function _picocli_ignite3_cluster_unit() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands="deploy undeploy list"
  local flag_opts="--help -h"
  local arg_opts="--verbose -v"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `metric` subcommand.
function _picocli_ignite3_cluster_metric() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands="source"
  local flag_opts="--help -h"
  local arg_opts="--verbose -v"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `show` subcommand.
function _picocli_ignite3_cluster_config_show() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands=""
  local flag_opts="--help -h"
  local arg_opts="--verbose -v --url --profile --format"
  local format_option_args=("JSON" "HOCON") # --format values

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
    --url)
      return
      ;;
    --profile)
      return
      ;;
    --format)
      local IFS=$'\n'
      COMPREPLY=( $( compReplyArray "${format_option_args[@]}" ) )
      return $?
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `update` subcommand.
function _picocli_ignite3_cluster_config_update() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands=""
  local flag_opts="--help -h"
  local arg_opts="--verbose -v --url --profile"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
    --url)
      return
      ;;
    --profile)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `physical` subcommand.
function _picocli_ignite3_cluster_topology_physical() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands=""
  local flag_opts="--help -h --plain"
  local arg_opts="--verbose -v --url --profile"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
    --url)
      return
      ;;
    --profile)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `logical` subcommand.
function _picocli_ignite3_cluster_topology_logical() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands=""
  local flag_opts="--help -h --plain"
  local arg_opts="--verbose -v --url --profile"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
    --url)
      return
      ;;
    --profile)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `deploy` subcommand.
function _picocli_ignite3_cluster_unit_deploy() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands=""
  local flag_opts="--help -h"
  local arg_opts="--verbose -v --url --profile --version --path --nodes"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
    --url)
      return
      ;;
    --profile)
      return
      ;;
    --version)
      return
      ;;
    --path)
      local IFS=$'\n'
      type compopt &>/dev/null && compopt -o filenames
      COMPREPLY=( $( compgen -f -- "${curr_word}" ) ) # files
      return $?
      ;;
    --nodes)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `undeploy` subcommand.
function _picocli_ignite3_cluster_unit_undeploy() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands=""
  local flag_opts="--help -h"
  local arg_opts="--verbose -v --url --profile --version"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
    --url)
      return
      ;;
    --profile)
      return
      ;;
    --version)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `list` subcommand.
function _picocli_ignite3_cluster_unit_list() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands=""
  local flag_opts="--help -h --plain"
  local arg_opts="--verbose -v --version --status --url --profile"
  local statuses_option_args=("UPLOADING" "DEPLOYED" "OBSOLETE" "REMOVING") # --status values

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
    --version)
      return
      ;;
    --status)
      local IFS=$'\n'
      COMPREPLY=( $( compReplyArray "${statuses_option_args[@]}" ) )
      return $?
      ;;
    --url)
      return
      ;;
    --profile)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `source` subcommand.
function _picocli_ignite3_cluster_metric_source() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands="enable disable list"
  local flag_opts="--help -h"
  local arg_opts="--verbose -v"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `enable` subcommand.
function _picocli_ignite3_cluster_metric_source_enable() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands=""
  local flag_opts="--help -h"
  local arg_opts="--verbose -v --url --profile"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
    --url)
      return
      ;;
    --profile)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `disable` subcommand.
function _picocli_ignite3_cluster_metric_source_disable() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands=""
  local flag_opts="--help -h"
  local arg_opts="--verbose -v --url --profile"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
    --url)
      return
      ;;
    --profile)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `list` subcommand.
function _picocli_ignite3_cluster_metric_source_list() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands=""
  local flag_opts="--help -h --plain"
  local arg_opts="--verbose -v --url --profile"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
    --url)
      return
      ;;
    --profile)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `partitions` subcommand.
function _picocli_ignite3_recovery_partitions() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands="restart reset states"
  local flag_opts="--help -h"
  local arg_opts="--verbose -v"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `cluster` subcommand.
function _picocli_ignite3_recovery_cluster() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands="reset migrate"
  local flag_opts="--help -h"
  local arg_opts="--verbose -v"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `restart` subcommand.
function _picocli_ignite3_recovery_partitions_restart() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands=""
  local flag_opts="--help -h"
  local arg_opts="--verbose -v --url --profile --partitions --nodes --zone --table"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
    --url)
      return
      ;;
    --profile)
      return
      ;;
    --partitions)
      return
      ;;
    --nodes)
      return
      ;;
    --zone)
      return
      ;;
    --table)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `reset` subcommand.
function _picocli_ignite3_recovery_partitions_reset() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands=""
  local flag_opts="--help -h"
  local arg_opts="--verbose -v --url --profile --partitions --zone --table"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
    --url)
      return
      ;;
    --profile)
      return
      ;;
    --partitions)
      return
      ;;
    --zone)
      return
      ;;
    --table)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `states` subcommand.
function _picocli_ignite3_recovery_partitions_states() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands=""
  local flag_opts="--help -h --global --local --plain"
  local arg_opts="--verbose -v --url --profile --nodes --partitions --zones"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
    --url)
      return
      ;;
    --profile)
      return
      ;;
    --nodes)
      return
      ;;
    --partitions)
      return
      ;;
    --zones)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `reset` subcommand.
function _picocli_ignite3_recovery_cluster_reset() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands=""
  local flag_opts="--help -h"
  local arg_opts="--verbose -v --url --profile --cluster-management-group --metastorage-replication-factor"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
    --url)
      return
      ;;
    --profile)
      return
      ;;
    --cluster-management-group)
      return
      ;;
    --metastorage-replication-factor)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Generates completions for the options and subcommands of the `migrate` subcommand.
function _picocli_ignite3_recovery_cluster_migrate() {
  # Get completion data
  local curr_word=${COMP_WORDS[COMP_CWORD]}
  local prev_word=${COMP_WORDS[COMP_CWORD-1]}

  local commands=""
  local flag_opts="--help -h"
  local arg_opts="--verbose -v --old-cluster-url --new-cluster-url"

  type compopt &>/dev/null && compopt +o default

  case ${prev_word} in
    --verbose|-v)
      return
      ;;
    --old-cluster-url)
      return
      ;;
    --new-cluster-url)
      return
      ;;
  esac

  if [[ "${curr_word}" == -* ]]; then
    COMPREPLY=( $(compgen -W "${flag_opts} ${arg_opts}" -- "${curr_word}") )
  else
    local positionals=""
    local IFS=$'\n'
    COMPREPLY=( $(compgen -W "${commands// /$'\n'}${IFS}${positionals}" -- "${curr_word}") )
  fi
}

# Define a completion specification (a compspec) for the
# `ignite3`, `ignite3.sh`, and `ignite3.bash` commands.
# Uses the bash `complete` builtin (see [6]) to specify that shell function
# `_complete_ignite3` is responsible for generating possible completions for the
# current word on the command line.
# The `-o default` option means that if the function generated no matches, the
# default Bash completions and the Readline default filename completions are performed.
complete -F _complete_ignite3 -o default ignite3 ignite3.sh ignite3.bash
