﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/s3tables/S3Tables_EXPORTS.h>
#include <aws/s3tables/model/TableBucketType.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace S3Tables {
namespace Model {

/**
 * <p>Contains details about a table bucket.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/s3tables-2018-05-10/TableBucketSummary">AWS
 * API Reference</a></p>
 */
class TableBucketSummary {
 public:
  AWS_S3TABLES_API TableBucketSummary() = default;
  AWS_S3TABLES_API TableBucketSummary(Aws::Utils::Json::JsonView jsonValue);
  AWS_S3TABLES_API TableBucketSummary& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_S3TABLES_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the table bucket.</p>
   */
  inline const Aws::String& GetArn() const { return m_arn; }
  inline bool ArnHasBeenSet() const { return m_arnHasBeenSet; }
  template <typename ArnT = Aws::String>
  void SetArn(ArnT&& value) {
    m_arnHasBeenSet = true;
    m_arn = std::forward<ArnT>(value);
  }
  template <typename ArnT = Aws::String>
  TableBucketSummary& WithArn(ArnT&& value) {
    SetArn(std::forward<ArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the table bucket.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  TableBucketSummary& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the account that owns the table bucket.</p>
   */
  inline const Aws::String& GetOwnerAccountId() const { return m_ownerAccountId; }
  inline bool OwnerAccountIdHasBeenSet() const { return m_ownerAccountIdHasBeenSet; }
  template <typename OwnerAccountIdT = Aws::String>
  void SetOwnerAccountId(OwnerAccountIdT&& value) {
    m_ownerAccountIdHasBeenSet = true;
    m_ownerAccountId = std::forward<OwnerAccountIdT>(value);
  }
  template <typename OwnerAccountIdT = Aws::String>
  TableBucketSummary& WithOwnerAccountId(OwnerAccountIdT&& value) {
    SetOwnerAccountId(std::forward<OwnerAccountIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date and time the table bucket was created at.</p>
   */
  inline const Aws::Utils::DateTime& GetCreatedAt() const { return m_createdAt; }
  inline bool CreatedAtHasBeenSet() const { return m_createdAtHasBeenSet; }
  template <typename CreatedAtT = Aws::Utils::DateTime>
  void SetCreatedAt(CreatedAtT&& value) {
    m_createdAtHasBeenSet = true;
    m_createdAt = std::forward<CreatedAtT>(value);
  }
  template <typename CreatedAtT = Aws::Utils::DateTime>
  TableBucketSummary& WithCreatedAt(CreatedAtT&& value) {
    SetCreatedAt(std::forward<CreatedAtT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The system-assigned unique identifier for the table bucket.</p>
   */
  inline const Aws::String& GetTableBucketId() const { return m_tableBucketId; }
  inline bool TableBucketIdHasBeenSet() const { return m_tableBucketIdHasBeenSet; }
  template <typename TableBucketIdT = Aws::String>
  void SetTableBucketId(TableBucketIdT&& value) {
    m_tableBucketIdHasBeenSet = true;
    m_tableBucketId = std::forward<TableBucketIdT>(value);
  }
  template <typename TableBucketIdT = Aws::String>
  TableBucketSummary& WithTableBucketId(TableBucketIdT&& value) {
    SetTableBucketId(std::forward<TableBucketIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of the table bucket.</p>
   */
  inline TableBucketType GetType() const { return m_type; }
  inline bool TypeHasBeenSet() const { return m_typeHasBeenSet; }
  inline void SetType(TableBucketType value) {
    m_typeHasBeenSet = true;
    m_type = value;
  }
  inline TableBucketSummary& WithType(TableBucketType value) {
    SetType(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_arn;

  Aws::String m_name;

  Aws::String m_ownerAccountId;

  Aws::Utils::DateTime m_createdAt{};

  Aws::String m_tableBucketId;

  TableBucketType m_type{TableBucketType::NOT_SET};
  bool m_arnHasBeenSet = false;
  bool m_nameHasBeenSet = false;
  bool m_ownerAccountIdHasBeenSet = false;
  bool m_createdAtHasBeenSet = false;
  bool m_tableBucketIdHasBeenSet = false;
  bool m_typeHasBeenSet = false;
};

}  // namespace Model
}  // namespace S3Tables
}  // namespace Aws
