﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/athena/AthenaRequest.h>
#include <aws/athena/Athena_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Athena {
namespace Model {

/**
 */
class StartCalculationExecutionRequest : public AthenaRequest {
 public:
  AWS_ATHENA_API StartCalculationExecutionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "StartCalculationExecution"; }

  AWS_ATHENA_API Aws::String SerializePayload() const override;

  AWS_ATHENA_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The session ID.</p>
   */
  inline const Aws::String& GetSessionId() const { return m_sessionId; }
  inline bool SessionIdHasBeenSet() const { return m_sessionIdHasBeenSet; }
  template <typename SessionIdT = Aws::String>
  void SetSessionId(SessionIdT&& value) {
    m_sessionIdHasBeenSet = true;
    m_sessionId = std::forward<SessionIdT>(value);
  }
  template <typename SessionIdT = Aws::String>
  StartCalculationExecutionRequest& WithSessionId(SessionIdT&& value) {
    SetSessionId(std::forward<SessionIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A description of the calculation.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  StartCalculationExecutionRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A string that contains the code of the calculation. Use this parameter
   * instead of <a>CalculationConfiguration$CodeBlock</a>, which is deprecated.</p>
   */
  inline const Aws::String& GetCodeBlock() const { return m_codeBlock; }
  inline bool CodeBlockHasBeenSet() const { return m_codeBlockHasBeenSet; }
  template <typename CodeBlockT = Aws::String>
  void SetCodeBlock(CodeBlockT&& value) {
    m_codeBlockHasBeenSet = true;
    m_codeBlock = std::forward<CodeBlockT>(value);
  }
  template <typename CodeBlockT = Aws::String>
  StartCalculationExecutionRequest& WithCodeBlock(CodeBlockT&& value) {
    SetCodeBlock(std::forward<CodeBlockT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A unique case-sensitive string used to ensure the request to create the
   * calculation is idempotent (executes only once). If another
   * <code>StartCalculationExecutionRequest</code> is received, the same response is
   * returned and another calculation is not created. If a parameter has changed, an
   * error is returned.</p>  <p>This token is listed as not required
   * because Amazon Web Services SDKs (for example the Amazon Web Services SDK for
   * Java) auto-generate the token for users. If you are not using the Amazon Web
   * Services SDK or the Amazon Web Services CLI, you must provide this token or the
   * action will fail.</p>
   */
  inline const Aws::String& GetClientRequestToken() const { return m_clientRequestToken; }
  inline bool ClientRequestTokenHasBeenSet() const { return m_clientRequestTokenHasBeenSet; }
  template <typename ClientRequestTokenT = Aws::String>
  void SetClientRequestToken(ClientRequestTokenT&& value) {
    m_clientRequestTokenHasBeenSet = true;
    m_clientRequestToken = std::forward<ClientRequestTokenT>(value);
  }
  template <typename ClientRequestTokenT = Aws::String>
  StartCalculationExecutionRequest& WithClientRequestToken(ClientRequestTokenT&& value) {
    SetClientRequestToken(std::forward<ClientRequestTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_sessionId;

  Aws::String m_description;

  Aws::String m_codeBlock;

  Aws::String m_clientRequestToken;
  bool m_sessionIdHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_codeBlockHasBeenSet = false;
  bool m_clientRequestTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace Athena
}  // namespace Aws
