﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/cost-optimization-hub/CostOptimizationHub_EXPORTS.h>
#include <aws/cost-optimization-hub/model/ActionType.h>
#include <aws/cost-optimization-hub/model/ImplementationEffort.h>
#include <aws/cost-optimization-hub/model/ResourceType.h>
#include <aws/cost-optimization-hub/model/Tag.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace CostOptimizationHub {
namespace Model {

/**
 * <p>Describes a filter that returns a more specific list of recommendations.
 * Filters recommendations by different dimensions.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/cost-optimization-hub-2022-07-26/Filter">AWS
 * API Reference</a></p>
 */
class Filter {
 public:
  AWS_COSTOPTIMIZATIONHUB_API Filter() = default;
  AWS_COSTOPTIMIZATIONHUB_API Filter(Aws::Utils::Json::JsonView jsonValue);
  AWS_COSTOPTIMIZATIONHUB_API Filter& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_COSTOPTIMIZATIONHUB_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>Whether or not implementing the recommendation requires a restart.</p>
   */
  inline bool GetRestartNeeded() const { return m_restartNeeded; }
  inline bool RestartNeededHasBeenSet() const { return m_restartNeededHasBeenSet; }
  inline void SetRestartNeeded(bool value) {
    m_restartNeededHasBeenSet = true;
    m_restartNeeded = value;
  }
  inline Filter& WithRestartNeeded(bool value) {
    SetRestartNeeded(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Whether or not implementing the recommendation can be rolled back.</p>
   */
  inline bool GetRollbackPossible() const { return m_rollbackPossible; }
  inline bool RollbackPossibleHasBeenSet() const { return m_rollbackPossibleHasBeenSet; }
  inline void SetRollbackPossible(bool value) {
    m_rollbackPossibleHasBeenSet = true;
    m_rollbackPossible = value;
  }
  inline Filter& WithRollbackPossible(bool value) {
    SetRollbackPossible(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The effort required to implement the recommendation.</p>
   */
  inline const Aws::Vector<ImplementationEffort>& GetImplementationEfforts() const { return m_implementationEfforts; }
  inline bool ImplementationEffortsHasBeenSet() const { return m_implementationEffortsHasBeenSet; }
  template <typename ImplementationEffortsT = Aws::Vector<ImplementationEffort>>
  void SetImplementationEfforts(ImplementationEffortsT&& value) {
    m_implementationEffortsHasBeenSet = true;
    m_implementationEfforts = std::forward<ImplementationEffortsT>(value);
  }
  template <typename ImplementationEffortsT = Aws::Vector<ImplementationEffort>>
  Filter& WithImplementationEfforts(ImplementationEffortsT&& value) {
    SetImplementationEfforts(std::forward<ImplementationEffortsT>(value));
    return *this;
  }
  inline Filter& AddImplementationEfforts(ImplementationEffort value) {
    m_implementationEffortsHasBeenSet = true;
    m_implementationEfforts.push_back(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The account to which the recommendation applies.</p>
   */
  inline const Aws::Vector<Aws::String>& GetAccountIds() const { return m_accountIds; }
  inline bool AccountIdsHasBeenSet() const { return m_accountIdsHasBeenSet; }
  template <typename AccountIdsT = Aws::Vector<Aws::String>>
  void SetAccountIds(AccountIdsT&& value) {
    m_accountIdsHasBeenSet = true;
    m_accountIds = std::forward<AccountIdsT>(value);
  }
  template <typename AccountIdsT = Aws::Vector<Aws::String>>
  Filter& WithAccountIds(AccountIdsT&& value) {
    SetAccountIds(std::forward<AccountIdsT>(value));
    return *this;
  }
  template <typename AccountIdsT = Aws::String>
  Filter& AddAccountIds(AccountIdsT&& value) {
    m_accountIdsHasBeenSet = true;
    m_accountIds.emplace_back(std::forward<AccountIdsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Web Services Region of the resource.</p>
   */
  inline const Aws::Vector<Aws::String>& GetRegions() const { return m_regions; }
  inline bool RegionsHasBeenSet() const { return m_regionsHasBeenSet; }
  template <typename RegionsT = Aws::Vector<Aws::String>>
  void SetRegions(RegionsT&& value) {
    m_regionsHasBeenSet = true;
    m_regions = std::forward<RegionsT>(value);
  }
  template <typename RegionsT = Aws::Vector<Aws::String>>
  Filter& WithRegions(RegionsT&& value) {
    SetRegions(std::forward<RegionsT>(value));
    return *this;
  }
  template <typename RegionsT = Aws::String>
  Filter& AddRegions(RegionsT&& value) {
    m_regionsHasBeenSet = true;
    m_regions.emplace_back(std::forward<RegionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The resource type of the recommendation.</p>
   */
  inline const Aws::Vector<ResourceType>& GetResourceTypes() const { return m_resourceTypes; }
  inline bool ResourceTypesHasBeenSet() const { return m_resourceTypesHasBeenSet; }
  template <typename ResourceTypesT = Aws::Vector<ResourceType>>
  void SetResourceTypes(ResourceTypesT&& value) {
    m_resourceTypesHasBeenSet = true;
    m_resourceTypes = std::forward<ResourceTypesT>(value);
  }
  template <typename ResourceTypesT = Aws::Vector<ResourceType>>
  Filter& WithResourceTypes(ResourceTypesT&& value) {
    SetResourceTypes(std::forward<ResourceTypesT>(value));
    return *this;
  }
  inline Filter& AddResourceTypes(ResourceType value) {
    m_resourceTypesHasBeenSet = true;
    m_resourceTypes.push_back(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of action you can take by adopting the recommendation.</p>
   */
  inline const Aws::Vector<ActionType>& GetActionTypes() const { return m_actionTypes; }
  inline bool ActionTypesHasBeenSet() const { return m_actionTypesHasBeenSet; }
  template <typename ActionTypesT = Aws::Vector<ActionType>>
  void SetActionTypes(ActionTypesT&& value) {
    m_actionTypesHasBeenSet = true;
    m_actionTypes = std::forward<ActionTypesT>(value);
  }
  template <typename ActionTypesT = Aws::Vector<ActionType>>
  Filter& WithActionTypes(ActionTypesT&& value) {
    SetActionTypes(std::forward<ActionTypesT>(value));
    return *this;
  }
  inline Filter& AddActionTypes(ActionType value) {
    m_actionTypesHasBeenSet = true;
    m_actionTypes.push_back(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of tags assigned to the recommendation.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  Filter& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  Filter& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The resource ID of the recommendation.</p>
   */
  inline const Aws::Vector<Aws::String>& GetResourceIds() const { return m_resourceIds; }
  inline bool ResourceIdsHasBeenSet() const { return m_resourceIdsHasBeenSet; }
  template <typename ResourceIdsT = Aws::Vector<Aws::String>>
  void SetResourceIds(ResourceIdsT&& value) {
    m_resourceIdsHasBeenSet = true;
    m_resourceIds = std::forward<ResourceIdsT>(value);
  }
  template <typename ResourceIdsT = Aws::Vector<Aws::String>>
  Filter& WithResourceIds(ResourceIdsT&& value) {
    SetResourceIds(std::forward<ResourceIdsT>(value));
    return *this;
  }
  template <typename ResourceIdsT = Aws::String>
  Filter& AddResourceIds(ResourceIdsT&& value) {
    m_resourceIdsHasBeenSet = true;
    m_resourceIds.emplace_back(std::forward<ResourceIdsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the recommendation.</p>
   */
  inline const Aws::Vector<Aws::String>& GetResourceArns() const { return m_resourceArns; }
  inline bool ResourceArnsHasBeenSet() const { return m_resourceArnsHasBeenSet; }
  template <typename ResourceArnsT = Aws::Vector<Aws::String>>
  void SetResourceArns(ResourceArnsT&& value) {
    m_resourceArnsHasBeenSet = true;
    m_resourceArns = std::forward<ResourceArnsT>(value);
  }
  template <typename ResourceArnsT = Aws::Vector<Aws::String>>
  Filter& WithResourceArns(ResourceArnsT&& value) {
    SetResourceArns(std::forward<ResourceArnsT>(value));
    return *this;
  }
  template <typename ResourceArnsT = Aws::String>
  Filter& AddResourceArns(ResourceArnsT&& value) {
    m_resourceArnsHasBeenSet = true;
    m_resourceArns.emplace_back(std::forward<ResourceArnsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The IDs for the recommendations.</p>
   */
  inline const Aws::Vector<Aws::String>& GetRecommendationIds() const { return m_recommendationIds; }
  inline bool RecommendationIdsHasBeenSet() const { return m_recommendationIdsHasBeenSet; }
  template <typename RecommendationIdsT = Aws::Vector<Aws::String>>
  void SetRecommendationIds(RecommendationIdsT&& value) {
    m_recommendationIdsHasBeenSet = true;
    m_recommendationIds = std::forward<RecommendationIdsT>(value);
  }
  template <typename RecommendationIdsT = Aws::Vector<Aws::String>>
  Filter& WithRecommendationIds(RecommendationIdsT&& value) {
    SetRecommendationIds(std::forward<RecommendationIdsT>(value));
    return *this;
  }
  template <typename RecommendationIdsT = Aws::String>
  Filter& AddRecommendationIds(RecommendationIdsT&& value) {
    m_recommendationIdsHasBeenSet = true;
    m_recommendationIds.emplace_back(std::forward<RecommendationIdsT>(value));
    return *this;
  }
  ///@}
 private:
  bool m_restartNeeded{false};

  bool m_rollbackPossible{false};

  Aws::Vector<ImplementationEffort> m_implementationEfforts;

  Aws::Vector<Aws::String> m_accountIds;

  Aws::Vector<Aws::String> m_regions;

  Aws::Vector<ResourceType> m_resourceTypes;

  Aws::Vector<ActionType> m_actionTypes;

  Aws::Vector<Tag> m_tags;

  Aws::Vector<Aws::String> m_resourceIds;

  Aws::Vector<Aws::String> m_resourceArns;

  Aws::Vector<Aws::String> m_recommendationIds;
  bool m_restartNeededHasBeenSet = false;
  bool m_rollbackPossibleHasBeenSet = false;
  bool m_implementationEffortsHasBeenSet = false;
  bool m_accountIdsHasBeenSet = false;
  bool m_regionsHasBeenSet = false;
  bool m_resourceTypesHasBeenSet = false;
  bool m_actionTypesHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
  bool m_resourceIdsHasBeenSet = false;
  bool m_resourceArnsHasBeenSet = false;
  bool m_recommendationIdsHasBeenSet = false;
};

}  // namespace Model
}  // namespace CostOptimizationHub
}  // namespace Aws
