﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/docdb/DocDBRequest.h>
#include <aws/docdb/DocDB_EXPORTS.h>
#include <aws/docdb/model/Parameter.h>

#include <utility>

namespace Aws {
namespace DocDB {
namespace Model {

/**
 * <p>Represents the input to <a>ResetDBClusterParameterGroup</a>.</p><p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/docdb-2014-10-31/ResetDBClusterParameterGroupMessage">AWS
 * API Reference</a></p>
 */
class ResetDBClusterParameterGroupRequest : public DocDBRequest {
 public:
  AWS_DOCDB_API ResetDBClusterParameterGroupRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ResetDBClusterParameterGroup"; }

  AWS_DOCDB_API Aws::String SerializePayload() const override;

 protected:
  AWS_DOCDB_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The name of the cluster parameter group to reset.</p>
   */
  inline const Aws::String& GetDBClusterParameterGroupName() const { return m_dBClusterParameterGroupName; }
  inline bool DBClusterParameterGroupNameHasBeenSet() const { return m_dBClusterParameterGroupNameHasBeenSet; }
  template <typename DBClusterParameterGroupNameT = Aws::String>
  void SetDBClusterParameterGroupName(DBClusterParameterGroupNameT&& value) {
    m_dBClusterParameterGroupNameHasBeenSet = true;
    m_dBClusterParameterGroupName = std::forward<DBClusterParameterGroupNameT>(value);
  }
  template <typename DBClusterParameterGroupNameT = Aws::String>
  ResetDBClusterParameterGroupRequest& WithDBClusterParameterGroupName(DBClusterParameterGroupNameT&& value) {
    SetDBClusterParameterGroupName(std::forward<DBClusterParameterGroupNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A value that is set to <code>true</code> to reset all parameters in the
   * cluster parameter group to their default values, and <code>false</code>
   * otherwise. You can't use this parameter if there is a list of parameter names
   * specified for the <code>Parameters</code> parameter.</p>
   */
  inline bool GetResetAllParameters() const { return m_resetAllParameters; }
  inline bool ResetAllParametersHasBeenSet() const { return m_resetAllParametersHasBeenSet; }
  inline void SetResetAllParameters(bool value) {
    m_resetAllParametersHasBeenSet = true;
    m_resetAllParameters = value;
  }
  inline ResetDBClusterParameterGroupRequest& WithResetAllParameters(bool value) {
    SetResetAllParameters(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of parameter names in the cluster parameter group to reset to the
   * default values. You can't use this parameter if the
   * <code>ResetAllParameters</code> parameter is set to <code>true</code>.</p>
   */
  inline const Aws::Vector<Parameter>& GetParameters() const { return m_parameters; }
  inline bool ParametersHasBeenSet() const { return m_parametersHasBeenSet; }
  template <typename ParametersT = Aws::Vector<Parameter>>
  void SetParameters(ParametersT&& value) {
    m_parametersHasBeenSet = true;
    m_parameters = std::forward<ParametersT>(value);
  }
  template <typename ParametersT = Aws::Vector<Parameter>>
  ResetDBClusterParameterGroupRequest& WithParameters(ParametersT&& value) {
    SetParameters(std::forward<ParametersT>(value));
    return *this;
  }
  template <typename ParametersT = Parameter>
  ResetDBClusterParameterGroupRequest& AddParameters(ParametersT&& value) {
    m_parametersHasBeenSet = true;
    m_parameters.emplace_back(std::forward<ParametersT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_dBClusterParameterGroupName;

  bool m_resetAllParameters{false};

  Aws::Vector<Parameter> m_parameters;
  bool m_dBClusterParameterGroupNameHasBeenSet = false;
  bool m_resetAllParametersHasBeenSet = false;
  bool m_parametersHasBeenSet = false;
};

}  // namespace Model
}  // namespace DocDB
}  // namespace Aws
