﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/elasticbeanstalk/ElasticBeanstalk_EXPORTS.h>
#include <aws/elasticbeanstalk/model/EnvironmentHealth.h>
#include <aws/elasticbeanstalk/model/EnvironmentHealthStatus.h>
#include <aws/elasticbeanstalk/model/EnvironmentLink.h>
#include <aws/elasticbeanstalk/model/EnvironmentResourcesDescription.h>
#include <aws/elasticbeanstalk/model/EnvironmentStatus.h>
#include <aws/elasticbeanstalk/model/EnvironmentTier.h>
#include <aws/elasticbeanstalk/model/ResponseMetadata.h>

#include <utility>

namespace Aws {
template <typename RESULT_TYPE>
class AmazonWebServiceResult;

namespace Utils {
namespace Xml {
class XmlDocument;
}  // namespace Xml
}  // namespace Utils
namespace ElasticBeanstalk {
namespace Model {
/**
 * <p>Describes the properties of an environment.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/elasticbeanstalk-2010-12-01/EnvironmentDescription">AWS
 * API Reference</a></p>
 */
class TerminateEnvironmentResult {
 public:
  AWS_ELASTICBEANSTALK_API TerminateEnvironmentResult() = default;
  AWS_ELASTICBEANSTALK_API TerminateEnvironmentResult(const Aws::AmazonWebServiceResult<Aws::Utils::Xml::XmlDocument>& result);
  AWS_ELASTICBEANSTALK_API TerminateEnvironmentResult& operator=(const Aws::AmazonWebServiceResult<Aws::Utils::Xml::XmlDocument>& result);

  ///@{
  /**
   * <p>The name of this environment.</p>
   */
  inline const Aws::String& GetEnvironmentName() const { return m_environmentName; }
  template <typename EnvironmentNameT = Aws::String>
  void SetEnvironmentName(EnvironmentNameT&& value) {
    m_environmentNameHasBeenSet = true;
    m_environmentName = std::forward<EnvironmentNameT>(value);
  }
  template <typename EnvironmentNameT = Aws::String>
  TerminateEnvironmentResult& WithEnvironmentName(EnvironmentNameT&& value) {
    SetEnvironmentName(std::forward<EnvironmentNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of this environment.</p>
   */
  inline const Aws::String& GetEnvironmentId() const { return m_environmentId; }
  template <typename EnvironmentIdT = Aws::String>
  void SetEnvironmentId(EnvironmentIdT&& value) {
    m_environmentIdHasBeenSet = true;
    m_environmentId = std::forward<EnvironmentIdT>(value);
  }
  template <typename EnvironmentIdT = Aws::String>
  TerminateEnvironmentResult& WithEnvironmentId(EnvironmentIdT&& value) {
    SetEnvironmentId(std::forward<EnvironmentIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the application associated with this environment.</p>
   */
  inline const Aws::String& GetApplicationName() const { return m_applicationName; }
  template <typename ApplicationNameT = Aws::String>
  void SetApplicationName(ApplicationNameT&& value) {
    m_applicationNameHasBeenSet = true;
    m_applicationName = std::forward<ApplicationNameT>(value);
  }
  template <typename ApplicationNameT = Aws::String>
  TerminateEnvironmentResult& WithApplicationName(ApplicationNameT&& value) {
    SetApplicationName(std::forward<ApplicationNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The application version deployed in this environment.</p>
   */
  inline const Aws::String& GetVersionLabel() const { return m_versionLabel; }
  template <typename VersionLabelT = Aws::String>
  void SetVersionLabel(VersionLabelT&& value) {
    m_versionLabelHasBeenSet = true;
    m_versionLabel = std::forward<VersionLabelT>(value);
  }
  template <typename VersionLabelT = Aws::String>
  TerminateEnvironmentResult& WithVersionLabel(VersionLabelT&& value) {
    SetVersionLabel(std::forward<VersionLabelT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The name of the <code>SolutionStack</code> deployed with this environment.
   * </p>
   */
  inline const Aws::String& GetSolutionStackName() const { return m_solutionStackName; }
  template <typename SolutionStackNameT = Aws::String>
  void SetSolutionStackName(SolutionStackNameT&& value) {
    m_solutionStackNameHasBeenSet = true;
    m_solutionStackName = std::forward<SolutionStackNameT>(value);
  }
  template <typename SolutionStackNameT = Aws::String>
  TerminateEnvironmentResult& WithSolutionStackName(SolutionStackNameT&& value) {
    SetSolutionStackName(std::forward<SolutionStackNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the platform version.</p>
   */
  inline const Aws::String& GetPlatformArn() const { return m_platformArn; }
  template <typename PlatformArnT = Aws::String>
  void SetPlatformArn(PlatformArnT&& value) {
    m_platformArnHasBeenSet = true;
    m_platformArn = std::forward<PlatformArnT>(value);
  }
  template <typename PlatformArnT = Aws::String>
  TerminateEnvironmentResult& WithPlatformArn(PlatformArnT&& value) {
    SetPlatformArn(std::forward<PlatformArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the configuration template used to originally launch this
   * environment.</p>
   */
  inline const Aws::String& GetTemplateName() const { return m_templateName; }
  template <typename TemplateNameT = Aws::String>
  void SetTemplateName(TemplateNameT&& value) {
    m_templateNameHasBeenSet = true;
    m_templateName = std::forward<TemplateNameT>(value);
  }
  template <typename TemplateNameT = Aws::String>
  TerminateEnvironmentResult& WithTemplateName(TemplateNameT&& value) {
    SetTemplateName(std::forward<TemplateNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Describes this environment.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  TerminateEnvironmentResult& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>For load-balanced, autoscaling environments, the URL to the LoadBalancer. For
   * single-instance environments, the IP address of the instance.</p>
   */
  inline const Aws::String& GetEndpointURL() const { return m_endpointURL; }
  template <typename EndpointURLT = Aws::String>
  void SetEndpointURL(EndpointURLT&& value) {
    m_endpointURLHasBeenSet = true;
    m_endpointURL = std::forward<EndpointURLT>(value);
  }
  template <typename EndpointURLT = Aws::String>
  TerminateEnvironmentResult& WithEndpointURL(EndpointURLT&& value) {
    SetEndpointURL(std::forward<EndpointURLT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The URL to the CNAME for this environment.</p>
   */
  inline const Aws::String& GetCNAME() const { return m_cNAME; }
  template <typename CNAMET = Aws::String>
  void SetCNAME(CNAMET&& value) {
    m_cNAMEHasBeenSet = true;
    m_cNAME = std::forward<CNAMET>(value);
  }
  template <typename CNAMET = Aws::String>
  TerminateEnvironmentResult& WithCNAME(CNAMET&& value) {
    SetCNAME(std::forward<CNAMET>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The creation date for this environment.</p>
   */
  inline const Aws::Utils::DateTime& GetDateCreated() const { return m_dateCreated; }
  template <typename DateCreatedT = Aws::Utils::DateTime>
  void SetDateCreated(DateCreatedT&& value) {
    m_dateCreatedHasBeenSet = true;
    m_dateCreated = std::forward<DateCreatedT>(value);
  }
  template <typename DateCreatedT = Aws::Utils::DateTime>
  TerminateEnvironmentResult& WithDateCreated(DateCreatedT&& value) {
    SetDateCreated(std::forward<DateCreatedT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The last modified date for this environment.</p>
   */
  inline const Aws::Utils::DateTime& GetDateUpdated() const { return m_dateUpdated; }
  template <typename DateUpdatedT = Aws::Utils::DateTime>
  void SetDateUpdated(DateUpdatedT&& value) {
    m_dateUpdatedHasBeenSet = true;
    m_dateUpdated = std::forward<DateUpdatedT>(value);
  }
  template <typename DateUpdatedT = Aws::Utils::DateTime>
  TerminateEnvironmentResult& WithDateUpdated(DateUpdatedT&& value) {
    SetDateUpdated(std::forward<DateUpdatedT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The current operational status of the environment:</p> <ul> <li> <p>
   * <code>Launching</code>: Environment is in the process of initial deployment.</p>
   * </li> <li> <p> <code>Updating</code>: Environment is in the process of updating
   * its configuration settings or application version.</p> </li> <li> <p>
   * <code>Ready</code>: Environment is available to have an action performed on it,
   * such as update or terminate.</p> </li> <li> <p> <code>Terminating</code>:
   * Environment is in the shut-down process.</p> </li> <li> <p>
   * <code>Terminated</code>: Environment is not running.</p> </li> </ul>
   */
  inline EnvironmentStatus GetStatus() const { return m_status; }
  inline void SetStatus(EnvironmentStatus value) {
    m_statusHasBeenSet = true;
    m_status = value;
  }
  inline TerminateEnvironmentResult& WithStatus(EnvironmentStatus value) {
    SetStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates if there is an in-progress environment configuration update or
   * application version deployment that you can cancel.</p> <p> <code>true:</code>
   * There is an update in progress. </p> <p> <code>false:</code> There are no
   * updates currently in progress. </p>
   */
  inline bool GetAbortableOperationInProgress() const { return m_abortableOperationInProgress; }
  inline void SetAbortableOperationInProgress(bool value) {
    m_abortableOperationInProgressHasBeenSet = true;
    m_abortableOperationInProgress = value;
  }
  inline TerminateEnvironmentResult& WithAbortableOperationInProgress(bool value) {
    SetAbortableOperationInProgress(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Describes the health status of the environment. AWS Elastic Beanstalk
   * indicates the failure levels for a running environment:</p> <ul> <li> <p>
   * <code>Red</code>: Indicates the environment is not responsive. Occurs when three
   * or more consecutive failures occur for an environment.</p> </li> <li> <p>
   * <code>Yellow</code>: Indicates that something is wrong. Occurs when two
   * consecutive failures occur for an environment.</p> </li> <li> <p>
   * <code>Green</code>: Indicates the environment is healthy and fully
   * functional.</p> </li> <li> <p> <code>Grey</code>: Default health for a new
   * environment. The environment is not fully launched and health checks have not
   * started or health checks are suspended during an <code>UpdateEnvironment</code>
   * or <code>RestartEnvironment</code> request.</p> </li> </ul> <p> Default:
   * <code>Grey</code> </p>
   */
  inline EnvironmentHealth GetHealth() const { return m_health; }
  inline void SetHealth(EnvironmentHealth value) {
    m_healthHasBeenSet = true;
    m_health = value;
  }
  inline TerminateEnvironmentResult& WithHealth(EnvironmentHealth value) {
    SetHealth(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Returns the health status of the application running in your environment. For
   * more information, see <a
   * href="https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/health-enhanced-status.html">Health
   * Colors and Statuses</a>.</p>
   */
  inline EnvironmentHealthStatus GetHealthStatus() const { return m_healthStatus; }
  inline void SetHealthStatus(EnvironmentHealthStatus value) {
    m_healthStatusHasBeenSet = true;
    m_healthStatus = value;
  }
  inline TerminateEnvironmentResult& WithHealthStatus(EnvironmentHealthStatus value) {
    SetHealthStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description of the AWS resources used by this environment.</p>
   */
  inline const EnvironmentResourcesDescription& GetResources() const { return m_resources; }
  template <typename ResourcesT = EnvironmentResourcesDescription>
  void SetResources(ResourcesT&& value) {
    m_resourcesHasBeenSet = true;
    m_resources = std::forward<ResourcesT>(value);
  }
  template <typename ResourcesT = EnvironmentResourcesDescription>
  TerminateEnvironmentResult& WithResources(ResourcesT&& value) {
    SetResources(std::forward<ResourcesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Describes the current tier of this environment.</p>
   */
  inline const EnvironmentTier& GetTier() const { return m_tier; }
  template <typename TierT = EnvironmentTier>
  void SetTier(TierT&& value) {
    m_tierHasBeenSet = true;
    m_tier = std::forward<TierT>(value);
  }
  template <typename TierT = EnvironmentTier>
  TerminateEnvironmentResult& WithTier(TierT&& value) {
    SetTier(std::forward<TierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of links to other environments in the same group.</p>
   */
  inline const Aws::Vector<EnvironmentLink>& GetEnvironmentLinks() const { return m_environmentLinks; }
  template <typename EnvironmentLinksT = Aws::Vector<EnvironmentLink>>
  void SetEnvironmentLinks(EnvironmentLinksT&& value) {
    m_environmentLinksHasBeenSet = true;
    m_environmentLinks = std::forward<EnvironmentLinksT>(value);
  }
  template <typename EnvironmentLinksT = Aws::Vector<EnvironmentLink>>
  TerminateEnvironmentResult& WithEnvironmentLinks(EnvironmentLinksT&& value) {
    SetEnvironmentLinks(std::forward<EnvironmentLinksT>(value));
    return *this;
  }
  template <typename EnvironmentLinksT = EnvironmentLink>
  TerminateEnvironmentResult& AddEnvironmentLinks(EnvironmentLinksT&& value) {
    m_environmentLinksHasBeenSet = true;
    m_environmentLinks.emplace_back(std::forward<EnvironmentLinksT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The environment's Amazon Resource Name (ARN), which can be used in other API
   * requests that require an ARN.</p>
   */
  inline const Aws::String& GetEnvironmentArn() const { return m_environmentArn; }
  template <typename EnvironmentArnT = Aws::String>
  void SetEnvironmentArn(EnvironmentArnT&& value) {
    m_environmentArnHasBeenSet = true;
    m_environmentArn = std::forward<EnvironmentArnT>(value);
  }
  template <typename EnvironmentArnT = Aws::String>
  TerminateEnvironmentResult& WithEnvironmentArn(EnvironmentArnT&& value) {
    SetEnvironmentArn(std::forward<EnvironmentArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the environment's operations role. For more
   * information, see <a
   * href="https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/iam-operationsrole.html">Operations
   * roles</a> in the <i>AWS Elastic Beanstalk Developer Guide</i>.</p>
   */
  inline const Aws::String& GetOperationsRole() const { return m_operationsRole; }
  template <typename OperationsRoleT = Aws::String>
  void SetOperationsRole(OperationsRoleT&& value) {
    m_operationsRoleHasBeenSet = true;
    m_operationsRole = std::forward<OperationsRoleT>(value);
  }
  template <typename OperationsRoleT = Aws::String>
  TerminateEnvironmentResult& WithOperationsRole(OperationsRoleT&& value) {
    SetOperationsRole(std::forward<OperationsRoleT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const ResponseMetadata& GetResponseMetadata() const { return m_responseMetadata; }
  template <typename ResponseMetadataT = ResponseMetadata>
  void SetResponseMetadata(ResponseMetadataT&& value) {
    m_responseMetadataHasBeenSet = true;
    m_responseMetadata = std::forward<ResponseMetadataT>(value);
  }
  template <typename ResponseMetadataT = ResponseMetadata>
  TerminateEnvironmentResult& WithResponseMetadata(ResponseMetadataT&& value) {
    SetResponseMetadata(std::forward<ResponseMetadataT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_environmentName;

  Aws::String m_environmentId;

  Aws::String m_applicationName;

  Aws::String m_versionLabel;

  Aws::String m_solutionStackName;

  Aws::String m_platformArn;

  Aws::String m_templateName;

  Aws::String m_description;

  Aws::String m_endpointURL;

  Aws::String m_cNAME;

  Aws::Utils::DateTime m_dateCreated{};

  Aws::Utils::DateTime m_dateUpdated{};

  EnvironmentStatus m_status{EnvironmentStatus::NOT_SET};

  bool m_abortableOperationInProgress{false};

  EnvironmentHealth m_health{EnvironmentHealth::NOT_SET};

  EnvironmentHealthStatus m_healthStatus{EnvironmentHealthStatus::NOT_SET};

  EnvironmentResourcesDescription m_resources;

  EnvironmentTier m_tier;

  Aws::Vector<EnvironmentLink> m_environmentLinks;

  Aws::String m_environmentArn;

  Aws::String m_operationsRole;

  ResponseMetadata m_responseMetadata;
  bool m_environmentNameHasBeenSet = false;
  bool m_environmentIdHasBeenSet = false;
  bool m_applicationNameHasBeenSet = false;
  bool m_versionLabelHasBeenSet = false;
  bool m_solutionStackNameHasBeenSet = false;
  bool m_platformArnHasBeenSet = false;
  bool m_templateNameHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_endpointURLHasBeenSet = false;
  bool m_cNAMEHasBeenSet = false;
  bool m_dateCreatedHasBeenSet = false;
  bool m_dateUpdatedHasBeenSet = false;
  bool m_statusHasBeenSet = false;
  bool m_abortableOperationInProgressHasBeenSet = false;
  bool m_healthHasBeenSet = false;
  bool m_healthStatusHasBeenSet = false;
  bool m_resourcesHasBeenSet = false;
  bool m_tierHasBeenSet = false;
  bool m_environmentLinksHasBeenSet = false;
  bool m_environmentArnHasBeenSet = false;
  bool m_operationsRoleHasBeenSet = false;
  bool m_responseMetadataHasBeenSet = false;
};

}  // namespace Model
}  // namespace ElasticBeanstalk
}  // namespace Aws
