﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/iot/IoTRequest.h>
#include <aws/iot/IoT_EXPORTS.h>
#include <aws/iot/model/ResourceIdentifier.h>

#include <utility>

namespace Aws {
namespace IoT {
namespace Model {

/**
 */
class CreateAuditSuppressionRequest : public IoTRequest {
 public:
  AWS_IOT_API CreateAuditSuppressionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateAuditSuppression"; }

  AWS_IOT_API Aws::String SerializePayload() const override;

  ///@{

  inline const Aws::String& GetCheckName() const { return m_checkName; }
  inline bool CheckNameHasBeenSet() const { return m_checkNameHasBeenSet; }
  template <typename CheckNameT = Aws::String>
  void SetCheckName(CheckNameT&& value) {
    m_checkNameHasBeenSet = true;
    m_checkName = std::forward<CheckNameT>(value);
  }
  template <typename CheckNameT = Aws::String>
  CreateAuditSuppressionRequest& WithCheckName(CheckNameT&& value) {
    SetCheckName(std::forward<CheckNameT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const ResourceIdentifier& GetResourceIdentifier() const { return m_resourceIdentifier; }
  inline bool ResourceIdentifierHasBeenSet() const { return m_resourceIdentifierHasBeenSet; }
  template <typename ResourceIdentifierT = ResourceIdentifier>
  void SetResourceIdentifier(ResourceIdentifierT&& value) {
    m_resourceIdentifierHasBeenSet = true;
    m_resourceIdentifier = std::forward<ResourceIdentifierT>(value);
  }
  template <typename ResourceIdentifierT = ResourceIdentifier>
  CreateAuditSuppressionRequest& WithResourceIdentifier(ResourceIdentifierT&& value) {
    SetResourceIdentifier(std::forward<ResourceIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The epoch timestamp in seconds at which this suppression expires. </p>
   */
  inline const Aws::Utils::DateTime& GetExpirationDate() const { return m_expirationDate; }
  inline bool ExpirationDateHasBeenSet() const { return m_expirationDateHasBeenSet; }
  template <typename ExpirationDateT = Aws::Utils::DateTime>
  void SetExpirationDate(ExpirationDateT&& value) {
    m_expirationDateHasBeenSet = true;
    m_expirationDate = std::forward<ExpirationDateT>(value);
  }
  template <typename ExpirationDateT = Aws::Utils::DateTime>
  CreateAuditSuppressionRequest& WithExpirationDate(ExpirationDateT&& value) {
    SetExpirationDate(std::forward<ExpirationDateT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> Indicates whether a suppression should exist indefinitely or not. </p>
   */
  inline bool GetSuppressIndefinitely() const { return m_suppressIndefinitely; }
  inline bool SuppressIndefinitelyHasBeenSet() const { return m_suppressIndefinitelyHasBeenSet; }
  inline void SetSuppressIndefinitely(bool value) {
    m_suppressIndefinitelyHasBeenSet = true;
    m_suppressIndefinitely = value;
  }
  inline CreateAuditSuppressionRequest& WithSuppressIndefinitely(bool value) {
    SetSuppressIndefinitely(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The description of the audit suppression. </p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  CreateAuditSuppressionRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> Each audit supression must have a unique client request token. If you try to
   * create a new audit suppression with the same token as one that already exists,
   * an exception occurs. If you omit this value, Amazon Web Services SDKs will
   * automatically generate a unique client request.</p>
   */
  inline const Aws::String& GetClientRequestToken() const { return m_clientRequestToken; }
  inline bool ClientRequestTokenHasBeenSet() const { return m_clientRequestTokenHasBeenSet; }
  template <typename ClientRequestTokenT = Aws::String>
  void SetClientRequestToken(ClientRequestTokenT&& value) {
    m_clientRequestTokenHasBeenSet = true;
    m_clientRequestToken = std::forward<ClientRequestTokenT>(value);
  }
  template <typename ClientRequestTokenT = Aws::String>
  CreateAuditSuppressionRequest& WithClientRequestToken(ClientRequestTokenT&& value) {
    SetClientRequestToken(std::forward<ClientRequestTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_checkName;

  ResourceIdentifier m_resourceIdentifier;

  Aws::Utils::DateTime m_expirationDate{};

  bool m_suppressIndefinitely{false};

  Aws::String m_description;

  Aws::String m_clientRequestToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_checkNameHasBeenSet = false;
  bool m_resourceIdentifierHasBeenSet = false;
  bool m_expirationDateHasBeenSet = false;
  bool m_suppressIndefinitelyHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_clientRequestTokenHasBeenSet = true;
};

}  // namespace Model
}  // namespace IoT
}  // namespace Aws
