﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/iotsitewise/IoTSiteWise_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace IoTSiteWise {
namespace Model {

/**
 * <p>Contains information for an asset property value entry that is associated
 * with the <a
 * href="https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_BatchGetAssetPropertyValue.html">BatchGetAssetPropertyValue</a>
 * API.</p> <p>To identify an asset property, you must specify one of the
 * following:</p> <ul> <li> <p>The <code>assetId</code> and <code>propertyId</code>
 * of an asset property.</p> </li> <li> <p>A <code>propertyAlias</code>, which is a
 * data stream alias (for example,
 * <code>/company/windfarm/3/turbine/7/temperature</code>). To define an asset
 * property's alias, see <a
 * href="https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_UpdateAssetProperty.html">UpdateAssetProperty</a>.</p>
 * </li> </ul><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/iotsitewise-2019-12-02/BatchGetAssetPropertyValueEntry">AWS
 * API Reference</a></p>
 */
class BatchGetAssetPropertyValueEntry {
 public:
  AWS_IOTSITEWISE_API BatchGetAssetPropertyValueEntry() = default;
  AWS_IOTSITEWISE_API BatchGetAssetPropertyValueEntry(Aws::Utils::Json::JsonView jsonValue);
  AWS_IOTSITEWISE_API BatchGetAssetPropertyValueEntry& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_IOTSITEWISE_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The ID of the entry.</p>
   */
  inline const Aws::String& GetEntryId() const { return m_entryId; }
  inline bool EntryIdHasBeenSet() const { return m_entryIdHasBeenSet; }
  template <typename EntryIdT = Aws::String>
  void SetEntryId(EntryIdT&& value) {
    m_entryIdHasBeenSet = true;
    m_entryId = std::forward<EntryIdT>(value);
  }
  template <typename EntryIdT = Aws::String>
  BatchGetAssetPropertyValueEntry& WithEntryId(EntryIdT&& value) {
    SetEntryId(std::forward<EntryIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the asset in which the asset property was created.</p>
   */
  inline const Aws::String& GetAssetId() const { return m_assetId; }
  inline bool AssetIdHasBeenSet() const { return m_assetIdHasBeenSet; }
  template <typename AssetIdT = Aws::String>
  void SetAssetId(AssetIdT&& value) {
    m_assetIdHasBeenSet = true;
    m_assetId = std::forward<AssetIdT>(value);
  }
  template <typename AssetIdT = Aws::String>
  BatchGetAssetPropertyValueEntry& WithAssetId(AssetIdT&& value) {
    SetAssetId(std::forward<AssetIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the asset property, in UUID format.</p>
   */
  inline const Aws::String& GetPropertyId() const { return m_propertyId; }
  inline bool PropertyIdHasBeenSet() const { return m_propertyIdHasBeenSet; }
  template <typename PropertyIdT = Aws::String>
  void SetPropertyId(PropertyIdT&& value) {
    m_propertyIdHasBeenSet = true;
    m_propertyId = std::forward<PropertyIdT>(value);
  }
  template <typename PropertyIdT = Aws::String>
  BatchGetAssetPropertyValueEntry& WithPropertyId(PropertyIdT&& value) {
    SetPropertyId(std::forward<PropertyIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The alias that identifies the property, such as an OPC-UA server data stream
   * path (for example, <code>/company/windfarm/3/turbine/7/temperature</code>). For
   * more information, see <a
   * href="https://docs.aws.amazon.com/iot-sitewise/latest/userguide/connect-data-streams.html">Mapping
   * industrial data streams to asset properties</a> in the <i>IoT SiteWise User
   * Guide</i>.</p>
   */
  inline const Aws::String& GetPropertyAlias() const { return m_propertyAlias; }
  inline bool PropertyAliasHasBeenSet() const { return m_propertyAliasHasBeenSet; }
  template <typename PropertyAliasT = Aws::String>
  void SetPropertyAlias(PropertyAliasT&& value) {
    m_propertyAliasHasBeenSet = true;
    m_propertyAlias = std::forward<PropertyAliasT>(value);
  }
  template <typename PropertyAliasT = Aws::String>
  BatchGetAssetPropertyValueEntry& WithPropertyAlias(PropertyAliasT&& value) {
    SetPropertyAlias(std::forward<PropertyAliasT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_entryId;

  Aws::String m_assetId;

  Aws::String m_propertyId;

  Aws::String m_propertyAlias;
  bool m_entryIdHasBeenSet = false;
  bool m_assetIdHasBeenSet = false;
  bool m_propertyIdHasBeenSet = false;
  bool m_propertyAliasHasBeenSet = false;
};

}  // namespace Model
}  // namespace IoTSiteWise
}  // namespace Aws
