﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/iam/IAMRequest.h>
#include <aws/iam/IAM_EXPORTS.h>
#include <aws/iam/model/Tag.h>

#include <utility>

namespace Aws {
namespace IAM {
namespace Model {

/**
 */
class CreateUserRequest : public IAMRequest {
 public:
  AWS_IAM_API CreateUserRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateUser"; }

  AWS_IAM_API Aws::String SerializePayload() const override;

 protected:
  AWS_IAM_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p> The path for the user name. For more information about paths, see <a
   * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/Using_Identifiers.html">IAM
   * identifiers</a> in the <i>IAM User Guide</i>.</p> <p>This parameter is optional.
   * If it is not included, it defaults to a slash (/).</p> <p>This parameter allows
   * (through its <a href="http://wikipedia.org/wiki/regex">regex pattern</a>) a
   * string of characters consisting of either a forward slash (/) by itself or a
   * string that must begin and end with forward slashes. In addition, it can contain
   * any ASCII character from the ! (<code>\u0021</code>) through the DEL character
   * (<code>\u007F</code>), including most punctuation characters, digits, and upper
   * and lowercased letters.</p>
   */
  inline const Aws::String& GetPath() const { return m_path; }
  inline bool PathHasBeenSet() const { return m_pathHasBeenSet; }
  template <typename PathT = Aws::String>
  void SetPath(PathT&& value) {
    m_pathHasBeenSet = true;
    m_path = std::forward<PathT>(value);
  }
  template <typename PathT = Aws::String>
  CreateUserRequest& WithPath(PathT&& value) {
    SetPath(std::forward<PathT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the user to create.</p> <p>IAM user, group, role, and policy
   * names must be unique within the account. Names are not distinguished by case.
   * For example, you cannot create resources named both "MyResource" and
   * "myresource".</p>
   */
  inline const Aws::String& GetUserName() const { return m_userName; }
  inline bool UserNameHasBeenSet() const { return m_userNameHasBeenSet; }
  template <typename UserNameT = Aws::String>
  void SetUserName(UserNameT&& value) {
    m_userNameHasBeenSet = true;
    m_userName = std::forward<UserNameT>(value);
  }
  template <typename UserNameT = Aws::String>
  CreateUserRequest& WithUserName(UserNameT&& value) {
    SetUserName(std::forward<UserNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the managed policy that is used to set the permissions boundary
   * for the user.</p> <p>A permissions boundary policy defines the maximum
   * permissions that identity-based policies can grant to an entity, but does not
   * grant permissions. Permissions boundaries do not define the maximum permissions
   * that a resource-based policy can grant to an entity. To learn more, see <a
   * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_boundaries.html">Permissions
   * boundaries for IAM entities</a> in the <i>IAM User Guide</i>.</p> <p>For more
   * information about policy types, see <a
   * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies.html#access_policy-types">Policy
   * types </a> in the <i>IAM User Guide</i>.</p>
   */
  inline const Aws::String& GetPermissionsBoundary() const { return m_permissionsBoundary; }
  inline bool PermissionsBoundaryHasBeenSet() const { return m_permissionsBoundaryHasBeenSet; }
  template <typename PermissionsBoundaryT = Aws::String>
  void SetPermissionsBoundary(PermissionsBoundaryT&& value) {
    m_permissionsBoundaryHasBeenSet = true;
    m_permissionsBoundary = std::forward<PermissionsBoundaryT>(value);
  }
  template <typename PermissionsBoundaryT = Aws::String>
  CreateUserRequest& WithPermissionsBoundary(PermissionsBoundaryT&& value) {
    SetPermissionsBoundary(std::forward<PermissionsBoundaryT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of tags that you want to attach to the new user. Each tag consists of
   * a key name and an associated value. For more information about tagging, see <a
   * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_tags.html">Tagging IAM
   * resources</a> in the <i>IAM User Guide</i>.</p>  <p>If any one of the tags
   * is invalid or if you exceed the allowed maximum number of tags, then the entire
   * request fails and the resource is not created.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateUserRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateUserRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_path;

  Aws::String m_userName;

  Aws::String m_permissionsBoundary;

  Aws::Vector<Tag> m_tags;
  bool m_pathHasBeenSet = false;
  bool m_userNameHasBeenSet = false;
  bool m_permissionsBoundaryHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace IAM
}  // namespace Aws
