UNIT Formel;

{$N+,E+}

{$IFDEF DOK}
{-------------------------------------------------------------------------}
   Die UNIT Formel ist ein Interpreter fuer mathematische Ausdruecke
   mit max. 2 Variablen.
   Operatoren: +, -, *, /, ^
   Funktionen: sin, cos, tan, cot, arcsin, arccos, arctan, arccot
               sinh, cosh, tanh, coth, arsinh, arcosh, artanh, arcoth
               exp, ln, lgt, sqrt, int, frac, abs
   Konstanten: Integer- und Realzahlen in Gleitpunkt oder Exponentformat,
               (z.B. -5, 461.5, -43.1e-12)
               vordefinierte Konstanten pi und e
   Variablen:  x, y bzw. XChar, YChar
   Klammerungen wie in BASIC oder PASCAL,
   Schachtelungen beliebig tief.

   Ursprung und Idee       : c't 12/89 Seite 250 ff.
   Compiler                : Turbo Pascal 5.x
   verwendete Units        : ExSystem
{-------------------------------------------------------------------------}
{$ENDIF DOK}

INTERFACE

CONST
  Version='89-12-05';
  VersionsNr='1.10';
  XChar: Char ='x';   { die Variablen Bezeichener}
  YChar: Char ='y';
TYPE
{
  Alle Berechnungen werden in EXTENDED durchgefuehrt, weil dieses
  Format die Genauigkeit aller 'kleineren' Gleitkommaformate abdeckt.
}
  FormelPtr = ^Knoten;
  Knoten = RECORD
             Operator: CHAR;
             Operand : ^EXTENDED;
             LChild,
             RChild  : FormelPtr;
           END;

CONST
  { Moegliche Fehler-Nummern bei der Analyse eines Formelstrings. }
  { Klartext dazu liefert die Function FormelFehlerText.          }
  KeinFehler           = 0;
  KlammerZuErwartet    = 1;
  KlammerAufErwartet   = 2;
  UngueltigeZahl       = 3;
  UnerwartetesEnde     = 4;
  UnerwartetesZeichen  = 5;

{$F+}
FUNCTION  BaueFormel (VAR FormelString: STRING;
                      VAR FehlerPos, FehlerNr: INTEGER): FormelPtr;
FUNCTION  Berechne (VAR p: FormelPtr; x, y: EXTENDED): EXTENDED;
PROCEDURE LoescheFormel (VAR p: FormelPtr);
FUNCTION  FormelFehlerText (FehlerNr: INTEGER): STRING;
PROCEDURE ZeigeBaumStruktur (p: FormelPtr; tiefe: INTEGER);

{-------------------------------------------------------------------------}
                              IMPLEMENTATION
{-------------------------------------------------------------------------}

USES EXSYSTEM;

CONST
  FirstUnFktToken = '@';
  LastUnFktToken  = 'V';
  {
    Bemerkung zu den Tokens (Kurzzeichen fuer Funktionen usw.):
    Der Funktionsumfang ist erweiterbar, indem LastUnFktToken groesser
    gemacht wird. Die Procedure InitUnaereFkt ist dann zu erweitern.
    Es kann auch ein zweiter Tokenbereich fuer binaere Funktionen usw.
    definiert werden. Dann ist eine weitere Procedure InitBinFkt
    o.ae. zu definieren.
    Tokens sind im Bereich ['@'..#255] anzulegen.
    Fuer andere Zwecke sind folgende Tokens reserviert:
    '+', '-', '*', '/'   fuer arithmetische Grundoperatoren
    '^'                  fuer Potenzierung
    '&'                  fuer Vorzeichenwechsel
    #0                   fuer Operand = Realzahl
    #1                   fuer Operand = x ( bzw. XChar)
    #2                   fuer Operand = y ( bzw. YChar)
  }
TYPE
  UnFktTokens = FirstUnFktToken .. LastUnFktToken;
  UnaereFunktionsTyp = FUNCTION (x: EXTENDED): EXTENDED;
  FktStr = STRING[6];
  UnaereFktRecord = RECORD
    Name: FktStr;
    UnFkt: UnaereFunktionsTyp;
  END;

VAR
  UnaereFkt : ARRAY [UnFktTokens] OF UnaereFktRecord;
  fehler: BOOLEAN;
  aktPos, FehlerArt: INTEGER;
  aktChar: CHAR;

FUNCTION Summe (VAR F: STRING): FormelPtr; FORWARD;

FUNCTION FormelFehlerText (FehlerNr: INTEGER): STRING;
{========================}
{$IFDEF DOK}
  FUNCTION FormelFehlerText (FehlerNr: INTEGER): STRING;
  Liefert eine Fehlerbeschreibung zur von BaueBaum erzeugten FehlerNr.
{$ENDIF DOK}
CONST
  FehlerText : ARRAY[0..5] OF STRING[31] = (
    { 0 } 'Kein Fehler',
    { 1 } 'schlieende Klammer erwartet',
    { 2 } 'ffnende Klammer erwartet',
    { 3 } 'ungltige Zahl',
    { 4 } 'unerwartetes Ende',
    { 5 } 'unerwartetes Zeichen');
BEGIN
  IF (FehlerNr >= 0) AND (FehlerNr <= 5) THEN
    FormelFehlerText := FehlerText[FehlerNr]
  ELSE
    FormelFehlerText := '';
END;


{-------------------------------------------------------------------------}
{      Neudefinition der math. Systemfunktionen, damit Adresse            }
{      in InitUnaereFkt zugewiesen werden kann.                           }
{-------------------------------------------------------------------------}

FUNCTION arctan(x:EXTENDED):EXTENDED; BEGIN arctan := System.arctan(x); END;
FUNCTION sqrt(x:EXTENDED):EXTENDED; BEGIN sqrt := System.sqrt(x); END;
FUNCTION frac(x:EXTENDED):EXTENDED; BEGIN frac := System.frac(x); END;
FUNCTION sin(x:EXTENDED):EXTENDED; BEGIN sin := System.sin(x); END;
FUNCTION cos(x:EXTENDED):EXTENDED; BEGIN cos := System.cos(x); END;
FUNCTION exp(x:EXTENDED):EXTENDED; BEGIN exp := System.exp(x); END;
FUNCTION int(x:EXTENDED):EXTENDED; BEGIN int := System.int(x); END;
FUNCTION abs(x:EXTENDED):EXTENDED; BEGIN abs := System.abs(x); END;
FUNCTION ln(x:EXTENDED):EXTENDED; BEGIN ln := System.ln(x); END;


PROCEDURE InitUnaereFkt;
{----------------------}
BEGIN
  WITH UnaereFkt['@'] DO BEGIN Name := 'arcsin'; UnFkt := arcsin; END;
  WITH UnaereFkt['A'] DO BEGIN Name := 'arccos'; UnFkt := arccos; END;
  WITH UnaereFkt['B'] DO BEGIN Name := 'arctan'; UnFkt := arctan; END;
  WITH UnaereFkt['C'] DO BEGIN Name := 'arccot'; UnFkt := arccot; END;
  WITH UnaereFkt['D'] DO BEGIN Name := 'arsinh'; UnFkt := arsinh; END;
  WITH UnaereFkt['E'] DO BEGIN Name := 'arcosh'; UnFkt := arcosh; END;
  WITH UnaereFkt['F'] DO BEGIN Name := 'artanh'; UnFkt := artanh; END;
  WITH UnaereFkt['G'] DO BEGIN Name := 'arcoth'; UnFkt := arcoth; END;
  WITH UnaereFkt['H'] DO BEGIN Name := 'sinh'; UnFkt := sinh; END;
  WITH UnaereFkt['I'] DO BEGIN Name := 'cosh'; UnFkt := cosh; END;
  WITH UnaereFkt['J'] DO BEGIN Name := 'tanh'; UnFkt := tanh; END;
  WITH UnaereFkt['K'] DO BEGIN Name := 'coth'; UnFkt := coth; END;
  WITH UnaereFkt['L'] DO BEGIN Name := 'sqrt'; UnFkt := sqrt; END;
  WITH UnaereFkt['M'] DO BEGIN Name := 'frac'; UnFkt := frac; END;
  WITH UnaereFkt['N'] DO BEGIN Name := 'sin'; UnFkt := sin; END;
  WITH UnaereFkt['O'] DO BEGIN Name := 'cos'; UnFkt := cos; END;
  WITH UnaereFkt['P'] DO BEGIN Name := 'tan'; UnFkt := tan; END;
  WITH UnaereFkt['Q'] DO BEGIN Name := 'cot'; UnFkt := cot; END;
  WITH UnaereFkt['R'] DO BEGIN Name := 'lgt'; UnFkt := lgt; END;
  WITH UnaereFkt['S'] DO BEGIN Name := 'exp'; UnFkt := exp; END;
  WITH UnaereFkt['T'] DO BEGIN Name := 'int'; UnFkt := int; END;
  WITH UnaereFkt['U'] DO BEGIN Name := 'abs'; UnFkt := abs; END;
  WITH UnaereFkt['V'] DO BEGIN Name := 'ln'; UnFkt := ln; END;
END;


PROCEDURE ZeigeBaumStruktur (p: FormelPtr; tiefe: INTEGER);
{==========================}
{$IFDEF DOK}
  PROCEDURE ZeigeBaumStruktur (p: FormelPtr; tiefe: INTEGER);
  Hilfsprozedur zum Austesten. Schreibt einen Binaerbaum, der mit
  BaueBaum erzeugt wurde, auf die Standardausgabe (Bildschirm).
  'tiefe' wird normalerweise auf 0 gesetzt (=Darstellung von der
  Baumwurzel ausgehend).
  Der dargestellte Baum wird von rechts nach links und von unten nach
  oben von FUNCTION Berechne abgearbeitet.
{$ENDIF DOK}
FUNCTION UnFunktName (Token: CHAR): STRING;
BEGIN
  IF (Token >= FirstUnFktToken) AND (Token <= LastUnFktToken) THEN
    UnFunktName := UnaereFkt[Token].Name
  ELSE
    UnFunktName := '';
END;

BEGIN
  IF p <> NIL THEN BEGIN
    ZeigeBaumStruktur (p^.LChild, tiefe+1);
    Write ('':tiefe*2);
    CASE p^.Operator OF
      #0 : WriteLn (p^.Operand^:2:2);
      #1 : WriteLn ('x');
      #2 : WriteLn ('y');
      FirstUnFktToken..LastUnFktToken:
           WriteLn (UnFunktName (p^.Operator));
      ELSE WriteLn (p^.Operator);
    END;
    ZeigeBaumStruktur (p^.RChild, tiefe+1);
  END;
END;


PROCEDURE Vereinfache (VAR p: FormelPtr);
{--------------------}
VAR hilf: FormelPtr;
BEGIN
  IF p <> NIL THEN BEGIN
    Vereinfache (p^.LChild);
    Vereinfache (p^.RChild);
    { aus -(-Baum) wird Baum }
    IF ((p^.Operator = '&') AND (p^.LChild^.Operator = '&')) THEN BEGIN
      hilf := p^.LChild^.LChild;
      Dispose (p^.LChild);
      Dispose (p);
      p := hilf;
    END;
    { aus Baum1 + (-Baum2) wird Baum1 - Baum2 }
    IF ((p^.Operator = '+') AND (p^.RChild^.Operator = '&')) THEN BEGIN
      hilf := p^.RChild^.LChild;
      Dispose (p^.RChild);
      p^.RChild := hilf;
      p^.Operator := '-';
    END;
    { aus Baum1 - (-Baum2) wird Baum1 + Baum2 }
    IF ((p^.Operator = '-') AND (p^.RChild^.Operator = '&')) THEN BEGIN
      hilf := p^.RChild^.LChild;
      Dispose (p^.RChild);
      p^.RChild := hilf;
      p^.Operator := '+';
    END;
    { aus -Baum1 + Baum2 wird Baum2 - Baum1 }
    IF ((p^.Operator = '+') AND (p^.LChild^.Operator = '&')) THEN BEGIN
      hilf := p^.LChild^.LChild;
      Dispose (p^.LChild);
      p^.LChild := p^.RChild;
      p^.RChild := hilf;
      p^.Operator := '-';
    END;
  END;
END;

PROCEDURE LoescheFormel (VAR p: FormelPtr);
{======================}
{$IFDEF DOK}
  PROCEDURE LoescheFormel (VAR p: FormelPtr);
  Gegenstueck zu BaueFormel. Loescht einen Binaerbaum, der mit BaueFormel
  erzeugt wurde und gibt den Speicherplatz auf dem Heap frei.
  Sollte (wie z.B. Dispose) immer verwendet werden, wenn die Formel nicht
  mehr benoetigt wird, um den Speicher aufgeraemt zu halten.
{$ENDIF DOK}
BEGIN
  IF p <> NIL THEN BEGIN
    LoescheFormel (p^.LChild);
    LoescheFormel (p^.RChild);
    IF (p^.Operator = #0) THEN BEGIN
      Dispose (p^.Operand);
      Dispose (p);
      p := NIL;
    END ELSE BEGIN
      Dispose (p);
      p := NIL;
    END;
  END;
END;


PROCEDURE NextChar (start: INTEGER; VAR F: STRING);
{-----------------}
BEGIN
  IF start > Length(F) THEN
    aktChar := #0
  ELSE BEGIN
    REPEAT
      Inc (start);
    UNTIL (start >= Length(F)) OR (F[start] <> ' ');
    IF start <= Length(F) THEN BEGIN
      aktPos := start;
      aktChar := F[aktPos];
    END ELSE
      Inc (aktPos);
  END;
END;


PROCEDURE NeueWurzel (VAR Wurzel: FormelPtr; VAR F: STRING);
{-------------------}
{ alte Wurzel wird linker Sohn }
VAR hilf: FormelPtr;
BEGIN
  New (hilf);
  hilf^.LChild := Wurzel;
  Wurzel := hilf;
  Wurzel^.RChild := NIL;
  Wurzel^.Operator := aktChar;
  Wurzel^.Operand := NIL;
  NextChar (aktPos, F);
END;


PROCEDURE NeuerKnoten (VAR Knoten: FormelPtr);
{--------------------}
BEGIN
  New (Knoten);
  WITH Knoten^ DO BEGIN
    Operator := #0;
    Operand := NIL;
    RChild := NIL;
    LChild := NIL;
  END;
END;


FUNCTION Pos (w, F: STRING; s, l: INTEGER): BOOLEAN;
{-----------}
VAR
  index: INTEGER;
  bool: BOOLEAN;
BEGIN
  IF Length(F) < s+l-1 THEN
    Pos := FALSE
  ELSE BEGIN
    bool := TRUE;
    FOR index := s TO s+l-1 DO
      bool := bool AND (w[index-s+1] = F[index]);
    Pos := bool;
  END;
END;



FUNCTION UnaereFunktion (VAR F: STRING; VAR Token: CHAR): BOOLEAN;
{----------------------}
VAR
  LoopToken: CHAR;
  Found: BOOLEAN;
BEGIN
  Token := #0;
  LoopToken := FirstUnFktToken;
  REPEAT
    Found := Pos (UnaereFkt[LoopToken].Name, F, aktPos,
                  Length(UnaereFkt[LoopToken].Name));
    IF Found THEN Token := LoopToken;
    LoopToken := Succ(LoopToken);
  UNTIL Found OR (LoopToken > LastUnFktToken);
  IF Found THEN BEGIN
    Inc (aktPos, Length(UnaereFkt[Token].Name)-1);
    NextChar (aktPos, F);
  END;
  UnaereFunktion := Found;
END;


PROCEDURE GetNumber (VAR F: STRING; VAR posi: INTEGER; VAR Fhelp: STRING);
{------------------}
BEGIN
  Fhelp := '';
  WHILE (posi <= Length(F)) AND (F[posi] IN ['0'..'9']) DO BEGIN
    Fhelp := Fhelp + F[posi];
    Inc (posi);
  END;
  IF (posi <= Length(F)) AND (F[posi] = '.') THEN BEGIN
    Fhelp := Fhelp + F[posi];
    Inc (posi);
  END;
  WHILE (posi <= Length(F)) AND (F[posi] IN ['0'..'9']) DO BEGIN
    Fhelp := Fhelp + F[posi];
    Inc (posi);
  END;
  IF (posi <= Length(F)) AND (UpCase(F[posi]) = 'E') THEN BEGIN
    Fhelp := Fhelp + F[posi];
    Inc (posi);
    IF (posi <= Length(F)) AND (F[posi] IN ['+','-']) THEN BEGIN
      Fhelp := Fhelp + F[posi];
      Inc (posi);
    END;
    WHILE (posi <= Length(F)) AND (F[posi] IN ['0'..'9']) DO BEGIN
      Fhelp := Fhelp + F[posi];
      Inc (posi);
    END;
  END;
END;


{-------------------------------------------------------------------------}
{                             Kern-Routinen                               }
{-------------------------------------------------------------------------}

FUNCTION Faktor (VAR f: STRING): FormelPtr;
{--------------}
VAR
  Knoten: FormelPtr;
  valid : INTEGER;
  posi:   INTEGER;
  Token:  CHAR;
  Fhelp:  STRING;
BEGIN
  Knoten := NIL;
  IF aktChar = '(' THEN BEGIN
    { Rekursion bei geoeffneter Klammer }
    NextChar (aktPos, F);
    Knoten := Summe (F);
    IF (aktChar <> ')') AND NOT Fehler THEN BEGIN
      Fehler := TRUE;
      FehlerArt := KlammerZuErwartet;
    END;
    IF NOT Fehler THEN NextChar (aktPos, F);
    Faktor := Knoten;
  END ELSE IF UnaereFunktion (F, Token) THEN BEGIN
    IF aktChar <> '(' THEN BEGIN
      Fehler := TRUE;
      FehlerArt := KlammerAufErwartet;
      Faktor := Knoten;
    END ELSE BEGIN
      NeuerKnoten (Knoten);
      Knoten^.Operator := Token;
      NextChar (aktPos, F);
      Knoten^.LChild := Summe (F);
    END;
    IF (aktChar <> ')') AND NOT Fehler THEN BEGIN
      Fehler := TRUE;
      FehlerArt := KlammerZuErwartet;
    END;
    IF NOT Fehler THEN NextChar (aktPos, F);
    Faktor := Knoten;
  END ELSE IF aktChar = XChar THEN BEGIN
    NeuerKnoten (Knoten);
    Knoten^.Operator := #1;
    NextChar (aktPos, F);
    Faktor := Knoten;
  END ELSE IF aktChar = YChar THEN BEGIN
    NeuerKnoten (Knoten);
    Knoten^.Operator := #2;
    NextChar (aktPos, F);
    Faktor := Knoten;
  END ELSE IF Pos ('pi', F, aktPos, 2) THEN BEGIN
    NeuerKnoten (Knoten);
    New (Knoten^.Operand);
    Knoten^.Operand^ := Pi;
    Inc (aktPos);
    NextChar (aktPos, F);
    Faktor := Knoten;
  END ELSE IF aktChar = 'e' THEN BEGIN
    NeuerKnoten (Knoten);
    New (Knoten^.Operand);
    Knoten^.Operand^ := Exp(1.0);
    NextChar (aktPos, F);
    Faktor := Knoten;
  END ELSE IF aktChar IN ['0'..'9', '.'] THEN BEGIN
    NeuerKnoten (Knoten);
    New (Knoten^.Operand);
    posi := aktPos;
    GetNumber (F, posi, Fhelp);
    Val (Fhelp, Knoten^.Operand^, valid);
    aktPos := posi-1;
    NextChar (aktPos, F);
    Fehler := valid <> 0;
    FehlerArt := UngueltigeZahl;
    Faktor := Knoten;
  END ELSE BEGIN
    Fehler := TRUE;
    IF (aktChar = ' ') OR (aktPos = Length(F) + 1) THEN
      FehlerArt := UnerwartetesEnde
    ELSE
      FehlerArt := UnerwartetesZeichen;
    Faktor := Knoten;
  END;
END;


FUNCTION VorzFaktor (VAR F: STRING): FormelPtr;
{------------------}
VAR Wurzel: FormelPtr;
BEGIN
  Wurzel := NIL;
  IF aktChar = '-' THEN BEGIN
    NeuerKnoten (Wurzel);
    Wurzel^.Operator := '&';
    NextChar (aktPos, F);
    Wurzel^.LChild := Faktor(F);
    VorzFaktor := Wurzel;
  END ELSE IF aktChar = '+' THEN BEGIN
    NextChar (aktPos, F);
    Wurzel := Faktor(F);
    VorzFaktor := Wurzel;
  END ELSE BEGIN
    Wurzel := Faktor(F);
    VorzFaktor := Wurzel;
  END;
END;


FUNCTION Potenz (VAR F: STRING): FormelPtr;
{--------------}
VAR Wurzel: FormelPtr;
BEGIN
  Wurzel := NIL;
  Wurzel := VorzFaktor (F);
  WHILE (aktChar = '^') AND NOT Fehler DO BEGIN
    NeueWurzel (Wurzel, F);
    Wurzel^.RChild := VorzFaktor (F);
    Potenz := Wurzel;
  END;
  Potenz := Wurzel;
END;


FUNCTION Produkt (VAR F: STRING): FormelPtr;
{---------------}
VAR Wurzel: FormelPtr;
BEGIN
  Wurzel := NIL;
  Wurzel := Potenz (F);
  WHILE ((aktChar = '*') OR (aktChar = '/')) AND NOT Fehler DO BEGIN
    NeueWurzel (Wurzel, F);
    Wurzel^.RChild := Potenz (F);
  END;
  Produkt := Wurzel;
END;


FUNCTION Summe (VAR F: STRING): FormelPtr;
{-------------}
VAR Wurzel: FormelPtr;
BEGIN
  Wurzel := NIL;
  Wurzel := Produkt (F);
  WHILE ((aktChar = '+') OR (aktChar = '-')) AND NOT Fehler DO BEGIN
    NeueWurzel (Wurzel, F);
    Wurzel^.RChild := Produkt (F);
  END;
  Summe := Wurzel;
END;


FUNCTION BaueFormel (VAR FormelString: STRING;
{==================} VAR FehlerPos, FehlerNr: INTEGER): FormelPtr;
{$IFDEF DOK}
  FUNCTION BaueFormel (VAR FormelString: STRING;
                       VAR FehlerPos, FehlerNr: INTEGER): FormelPtr;

  Der uebergebene String FormelString wird von BaueFormel analysiert
  und dabei in Kleinbuchstaben umgewandelt.
  Ist die Formel syntaktisch korrekt, so wird ein Binaerbaum erzeugt,
  welcher der Formelberechnung dient. Zurueckgegeben wird der Pointer
  auf die Wurzel des Binaerbaums. FehlerNr und FehlerPos = 0.

  Ist ein Fehler in der Formel, wird kein Binaerbaum erzeugt und NIL
  zurueckgegeben. FehlerPos liefert die Position des Fehlers in
  FormelString, FehlerNr die Fehlernummer (Werte siehe Interfaceteil).
{$ENDIF DOK}
VAR Wurzel: FormelPtr;
BEGIN
  FormelString := LoString(FormelString);
  aktPos := 0;
  NextChar (aktPos, FormelString);
  Fehler := FALSE;
  Wurzel := NIL;
  Wurzel := Summe (FormelString);
  { Fehler, falls Stringende nicht erreicht, Leerzeichen am Ende erlaubt }
  IF (aktPos <= Length(FormelString)) AND (FormelString[aktPos] <> ' ')
    AND NOT Fehler THEN BEGIN
    Fehler := TRUE;
    FehlerArt := UnerwartetesZeichen;
  END;
  IF NOT Fehler THEN BEGIN
    FehlerPos := 0;
    FehlerNr := keinFehler;
    Vereinfache (Wurzel);
  END ELSE BEGIN
    { Fehler aufgetreten }
    LoescheFormel (Wurzel);
    FehlerPos := aktPos;
    FehlerNr := FehlerArt;
  END;
  BaueFormel := Wurzel;
END;


FUNCTION Berechne (VAR p: FormelPtr; x, y: EXTENDED): EXTENDED;
{================}
{$IFDEF DOK}
  FUNCTION Berechne (VAR p: FormelPtr; x, y: EXTENDED): EXTENDED;
  Berechnet fuer das Argumentpaar x, y die als Binaerbaum uebergebene
  Formel. Der Binaerbaum muss zuvor mit BaueFormel erzeugt worden sein.
{$ENDIF DOK}
  FUNCTION f (VAR p: FormelPtr): EXTENDED;
  VAR x1,x2 :EXTENDED;
  BEGIN
    CASE p^.Operator OF
      #0 : f := (p^.Operand^);
      #1 : f := x;
      #2 : f := y;
      '+': begin x1:= f(p^.LChild); x2:= f(p^.RChild); f:=x1+x2; end;
      '-': begin x1:= f(p^.LChild); x2:= f(p^.RChild); f:=x1-x2; end;
      '*': begin x1:= f(p^.LChild); x2:= f(p^.RChild); f:=x1*x2; end;
      '/': begin x1:= f(p^.LChild); x2:= f(p^.RChild); f:=x1/x2; end;
      '^': f := (pwr(f(p^.LChild), f(p^.RChild)));
      '&': f := (-f(p^.LChild));
      FirstUnFktToken..LastUnFktToken:
           f := (UnaereFkt[p^.Operator].UnFkt(f(p^.LChild)));
      ELSE Write ('Fehler im Baum');
    END;
  END;

BEGIN
  Berechne := f(p);
END;


BEGIN
  InitUnaereFkt;
END.
