/*********************************************************************
 * Copyright (c) 2017, 2021 Red Hat Inc. and others.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *  Red Hat Inc. - Initial implementation
 *******************************************************************************/
package org.eclipse.corrosion.test.actions;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.corrosion.CorrosionPlugin;
import org.eclipse.corrosion.launch.RustLaunchDelegateTools;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.jface.action.Action;
import org.eclipse.lsp4e.LSPEclipseUtils;
import org.eclipse.lsp4e.LanguageServerPlugin;
import org.eclipse.lsp4e.LanguageServiceAccessor;
import org.eclipse.lsp4j.Location;
import org.eclipse.lsp4j.Position;
import org.eclipse.lsp4j.Range;
import org.eclipse.lsp4j.SymbolInformation;
import org.eclipse.lsp4j.SymbolKind;
import org.eclipse.lsp4j.WorkspaceSymbolParams;
import org.eclipse.lsp4j.services.LanguageServer;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PlatformUI;
import org.eclipse.unittest.model.ITestCaseElement;
import org.eclipse.unittest.model.ITestElement;
import org.eclipse.unittest.model.ITestSuiteElement;

@SuppressWarnings("restriction")
public class OpenTestAction extends Action {

	private static final String SPLITTER = "::"; //$NON-NLS-1$
	private static final String EMPTY_STRING = ""; //$NON-NLS-1$

	private ITestElement fTestElement;

	public OpenTestAction(ITestSuiteElement testSuite) {
		super(ActionsMessages.OpenEditorAction_action_label);
		this.fTestElement = testSuite;
	}

	public OpenTestAction(ITestCaseElement testCase) {
		super(ActionsMessages.OpenEditorAction_action_label);
		this.fTestElement = testCase;
	}

	@Override
	public void run() {
		IProject project = getProject(fTestElement);
		if (project == null) {
			return;
		}
		List<LanguageServer> languageServers = LanguageServiceAccessor.getLanguageServers(project,
				capabilities -> Boolean.TRUE.equals(capabilities.getWorkspaceSymbolProvider().get()));
		if (languageServers != null && languageServers.isEmpty()) {
			return;
		}
		SymbolKind kind = fTestElement instanceof ITestCaseElement ? SymbolKind.Function : SymbolKind.Module;
		List<SymbolInformation> result = getQualifiedSymbols(languageServers, fTestElement.getTestName(), kind);
		if (result.isEmpty()) {
			return;
		}
		SymbolInformation symbolInformation = result.get(0);
		Location location = symbolInformation.getLocation();

		IWorkbenchPage page = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage();
		LSPEclipseUtils.openInEditor(location, page);

	}

	private static IProject getProject(ITestElement element) {
		ILaunch launch = element.getTestRunSession().getLaunch();
		ILaunchConfiguration launchConfig = launch != null ? launch.getLaunchConfiguration() : null;
		if (launchConfig == null) {
			return null;
		}

		String projectName;
		try {
			projectName = launchConfig.getAttribute(RustLaunchDelegateTools.PROJECT_ATTRIBUTE, EMPTY_STRING);
			if (!projectName.isEmpty()) {
				IProject project = ResourcesPlugin.getWorkspace().getRoot().getProject(projectName);
				return project != null && project.exists() ? project : null;
			}
		} catch (CoreException e) {
			CorrosionPlugin.logError(e);
		}

		return null;
	}

	private static List<SymbolInformation> getQualifiedSymbols(List<LanguageServer> languageServers, String path,
			SymbolKind kind) {
		List<SymbolInformation> result = new ArrayList<>();

		String[] pathArray = path.split(SPLITTER);
		if (pathArray == null || pathArray.length < 1) {
			return null;
		}

		List<SymbolInformation> parents = null;
		for (int i = 0; i < pathArray.length; i++) {
			String name = pathArray[i];
			String parent = i > 0 ? pathArray[i - 1] : null;
			SymbolKind expectedKind = i < pathArray.length - 1 ? SymbolKind.Module : kind;

			List<SymbolInformation> symbols = getSymbols(languageServers, name, parent, expectedKind);
			List<SymbolInformation> currents;
			if (parents != null) {
				List<SymbolInformation> matches = new ArrayList<>();
				for (SymbolInformation info : symbols) {
					for (SymbolInformation p : parents) {
						if (isContainerOfSymbol(p, info)) {
							matches.add(info);
							break;
						}
					}
				}
				currents = matches;
			} else {
				currents = symbols;
			}

			if (currents.isEmpty()) {
				return result; // No qualified matches found
			}

			parents = currents;
		}

		return parents;
	}

	private static List<SymbolInformation> getSymbols(List<LanguageServer> languageServers, String name, String parent,
			SymbolKind kind) {
		List<SymbolInformation> result = new ArrayList<>();

		for (LanguageServer server : languageServers) {
			WorkspaceSymbolParams params = new WorkspaceSymbolParams(name);
			CompletableFuture<List<? extends SymbolInformation>> symbols = server.getWorkspaceService().symbol(params);

			try {
				List<?> items = symbols.get(10, TimeUnit.SECONDS);
				if (items != null) {
					for (Object item : items) {
						SymbolInformation match = getMatchingItem(item, name, parent, kind);
						if (match != null) {
							result.add(match);
						}
					}
				}
			} catch (ExecutionException | TimeoutException e) {
				LanguageServerPlugin.logError(e);
			} catch (InterruptedException e) {
				LanguageServerPlugin.logError(e);
			}
		}
		return result;
	}

	private static SymbolInformation getMatchingItem(Object item, String name, String parent, SymbolKind kind) {
		if (!(item instanceof SymbolInformation)) {
			return null;
		}
		SymbolInformation info = (SymbolInformation) item;
		if (info.getKind() != kind) {
			return null;
		}
		parent = parent != null ? parent.toLowerCase() : EMPTY_STRING;
		String container = info.getContainerName() != null ? info.getContainerName().toLowerCase() : EMPTY_STRING;
		if (!parent.equals(container)) {
			return null;
		}
		return info.getName().toLowerCase().equals(name.toLowerCase()) ? info : null;
	}

	private static boolean isContainerOfSymbol(SymbolInformation container, SymbolInformation symbol) {
		if (container == null || symbol == null) {
			return false;
		}

		// Should URI to be the same?
		if (!container.getLocation().getUri().equals(symbol.getLocation().getUri())) {
			return false;
		}

		Range c = container.getLocation().getRange();
		Range s = symbol.getLocation().getRange();

		// Check if the container's start is less than or equal to the symbol's start
		Position cp = c.getStart();
		Position sp = s.getStart();

		if (cp.getLine() > sp.getLine()) {
			return false;
		}
		if (cp.getLine() == sp.getLine() && cp.getCharacter() > sp.getCharacter()) {
			return false;
		}

// The following is commented out as it looks like container end is not a module block closing brace position...
		// Check if the container's end is greater than or equal to the symbol's end
//		cp = c.getEnd();
//		sp = s.getEnd();
//
//		if (cp.getLine() < sp.getLine()) {
//			return false;
//		}
//		if (cp.getLine() == sp.getLine() && cp.getCharacter() < sp.getCharacter()) {
//			return false;
//		}

		return true;
	}

}