#
# Copyright (c) 2021, 2026 Contributors to the Eclipse Foundation
#
# This program and the accompanying materials are made
# available under the terms of the Eclipse Public License 2.0
# which is available at https://www.eclipse.org/legal/epl-2.0/
#
# SPDX-License-Identifier: EPL-2.0
#

from __future__ import annotations
import re

from pydantic import BaseModel
from typing import Dict, TYPE_CHECKING
if TYPE_CHECKING:
    from lsat_client.models.lsat_data import LsatData

class LsatBase(BaseModel):
    _field_translation = {
        "ActivityDependency": {
            "source": "action",
            "target": "action"
        },
        "Dependency": {
            "source": "task",
            "target": "task"
        },
        "Move": {
            "predecessor_move": "action",
            "successor_move": "action"
        },
        "ProductLifeCycleEventTransfer": {
            "in_task": "task",
            "out_task": "task"
        }
    }

    _resolved: bool = False

    @property
    def is_resolved(self) -> bool:
        return self._resolved

    # Resolve all links to fields ending with '_id' with a reference to the actual field
    def resolve_references(self, lsat_data: LsatData, container: LsatBase | None = None):
        self._resolved = True

        # Round 1: find reference to all fields ending with '_id' or '_ids'
        for field_name in self.__class__.model_fields:
            if field_name.endswith("_id") or field_name.endswith("_ids"):
                reference_field_name = field_name.rsplit('_', 1)[0]
                lookup_name = self._field_translation.get(self.__class__.__name__, dict()).get(reference_field_name, reference_field_name)
                lookup_name = self._make_plural(lookup_name)
                if field_name.endswith("_ids"):
                    reference_field_name = self._make_plural(reference_field_name)

                field = getattr(self, field_name)
                if lookup_name == 'actions':
                    if container.__class__.__name__ == 'Activity':
                        lookup_data: Dict = getattr(container, 'actions')
                    else:
                        activities: Dict = getattr(lsat_data, 'activities')
                        activity = activities.get(getattr(self, 'activity_id'))
                        lookup_data: Dict = getattr(activity, 'actions')
                else:
                    lookup_data: Dict = getattr(lsat_data, lookup_name)
                if isinstance(field, dict):
                    dct = getattr(self, reference_field_name)
                    if dct is None:
                        dct = dict()
                    for (key, f) in field.items():
                        value = lookup_data.get(f)
                        dct.update({key: value})
                    setattr(self, reference_field_name, dct)
                elif isinstance(field, list):
                    lst = getattr(self, reference_field_name)
                    if lst is None:
                        lst = list()
                    for f in field:
                        value = lookup_data.get(f)
                        lst.append(value)
                    setattr(self, reference_field_name, lst)
                else:
                    value = lookup_data.get(field)
                    setattr(self, reference_field_name, value)

        # Round 2: iterate over all fields and recursively 
        model_fields = self.__class__.model_fields
        for field_name in model_fields:
            if field_name + '_id' in model_fields or field_name + '_ids' in model_fields:
                continue
            field = getattr(self, field_name)
            if isinstance(field, dict):
                for f in field.values():
                    if isinstance(f, LsatBase) and not f.is_resolved:
                        f.resolve_references(lsat_data, self)
            elif isinstance(field, list):
                for f in field:
                    if isinstance(f, LsatBase) and not f.is_resolved:
                        f.resolve_references(lsat_data, self)
            elif isinstance(field, LsatBase) and not field.is_resolved:
                    field.resolve_references(lsat_data, self)

    @staticmethod
    def _make_plural(word):
        # check whether the input word is ending with s, x, z, or is 
        # ending with ah, eh, ih, oh, uh, dh, gh, kh, ph, rh, th
        if re.search('[sxz]$', word) or re.search('[^aeioudgkprt]h$', word):
            # if true, get the plural of the word by adding "es"
            return re.sub('$', 'es', word)
        # check whether the input word is not ending with ay, ey, iy, oy, uy
        elif re.search('[^aeiou]y$', word):
            # if true, get the plural of the word by removing 'y' and adding ies
            return re.sub('y$', 'ies', word)
        # else just add "s" to the word
        else:
            return word + 's'
