/*******************************************************************************
 * Copyright (c) 2009 Ecliptical Software Inc. and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Ecliptical Software Inc. - initial API and implementation
 *******************************************************************************/
package org.eclipse.emf.mint.ui;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import org.eclipse.emf.common.notify.AdapterFactory;
import org.eclipse.emf.edit.provider.ComposedAdapterFactory;
import org.eclipse.emf.edit.provider.IDisposable;
import org.eclipse.emf.mint.IItemJavaElementDescriptor;
import org.eclipse.emf.mint.IItemJavaElementSource;
import org.eclipse.emf.mint.internal.ui.actions.OpenGeneratedAction;
import org.eclipse.jface.action.ActionContributionItem;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IContributionItem;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.ui.ISources;
import org.eclipse.ui.actions.CompoundContributionItem;
import org.eclipse.ui.menus.IMenuService;
import org.eclipse.ui.menus.IWorkbenchContribution;
import org.eclipse.ui.services.IServiceLocator;

public abstract class AbstractOpenGeneratedMenu extends
		CompoundContributionItem implements IWorkbenchContribution {

	private static final IContributionItem[] NO_ITEMS = {};

	private static final int DEFAULT_MIN_CATEGORY_ITEMS = 2;

	protected IServiceLocator serviceLocator;

	private List<List<IAction>> actions;

	protected AdapterFactory adapterFactory;

	protected AbstractOpenGeneratedMenu() {
		super();
	}

	protected AbstractOpenGeneratedMenu(String id) {
		super(id);
	}

	public void initialize(IServiceLocator serviceLocator) {
		this.serviceLocator = serviceLocator;
	}

	@Override
	protected final IContributionItem[] getContributionItems() {
		if (actions == null) {
			actions = new LinkedList<List<IAction>>();
			IMenuService svc = (IMenuService) serviceLocator
					.getService(IMenuService.class);
			Object selection = svc.getCurrentState().getVariable(
					ISources.ACTIVE_MENU_SELECTION_NAME);
			if (selection instanceof ISelection) {
				Object target = getTarget((ISelection) selection);
				if (target != null) {
					AdapterFactory adapterFactory = getAdapterFactory(target);
					if (adapterFactory != null) {
						Object adapter = adapterFactory.adapt(target,
								IItemJavaElementSource.class);
						if (adapter instanceof IItemJavaElementSource)
							createActions((IItemJavaElementSource) adapter,
									target);
					}
				}
			}
		}

		if (actions.isEmpty())
			return NO_ITEMS;

		ArrayList<IContributionItem> items = new ArrayList<IContributionItem>();
		boolean needsSeparator = false;
		for (List<IAction> group : actions) {
			if (needsSeparator)
				items.add(new Separator());
			else
				needsSeparator = true;

			for (IAction action : group)
				items.add(new ActionContributionItem(action));
		}

		return items.toArray(new IContributionItem[items.size()]);
	}

	protected Object getTarget(ISelection selection) {
		if (selection instanceof IStructuredSelection)
			return ((IStructuredSelection) selection).getFirstElement();

		return null;
	}

	protected AdapterFactory getAdapterFactory(Object target) {
		if (adapterFactory == null)
			adapterFactory = createAdapterFactory(target);

		return adapterFactory;
	}

	protected AdapterFactory createAdapterFactory(Object target) {
		return new ComposedAdapterFactory(
				ComposedAdapterFactory.Descriptor.Registry.INSTANCE);
	}

	protected Comparator<? super IItemJavaElementDescriptor> createDescriptorComparator(
			Object target) {
		return new DescriptorComparator(target);
	}

	protected int getMinCategoryItems() {
		return DEFAULT_MIN_CATEGORY_ITEMS;
	}

	private void createActions(IItemJavaElementSource provider, Object target) {
		List<IItemJavaElementDescriptor> descriptors = new ArrayList<IItemJavaElementDescriptor>(
				provider.getJavaElementDescriptors(target));
		Collections.sort(descriptors, createDescriptorComparator(target));

		LinkedList<IAction> group = null;
		String category = null;
		boolean keepGroups = false;
		int categoryItemCount = 0;
		int minCategoryItems = getMinCategoryItems();
		for (IItemJavaElementDescriptor descriptor : descriptors) {
			String newCategory = descriptor.getCategory(target);
			if (group == null
					|| (category == null ? newCategory != null : !category
							.equals(newCategory))) {
				category = newCategory;
				categoryItemCount = 0;
				group = new LinkedList<IAction>();
				actions.add(group);
			}

			IAction action = createAction(descriptor, target);
			group.add(action);

			if (++categoryItemCount > minCategoryItems)
				keepGroups = true;
		}

		if (!keepGroups) {
			Iterator<List<IAction>> i = actions.iterator();
			if (i.hasNext()) {
				List<IAction> masterGroup = i.next();
				while (i.hasNext()) {
					masterGroup.addAll(i.next());
					i.remove();
				}
			}
		}
	}

	protected IAction createAction(IItemJavaElementDescriptor descriptor,
			Object target) {
		return new OpenGeneratedAction(descriptor, target);
	}

	@Override
	public void dispose() {
		if (adapterFactory instanceof IDisposable) {
			((IDisposable) adapterFactory).dispose();
			adapterFactory = null;
		}

		actions = null;
		super.dispose();
	}

	protected static class DescriptorComparator implements
			Comparator<IItemJavaElementDescriptor> {

		protected final Object target;

		public DescriptorComparator(Object target) {
			this.target = target;
		}

		public int compare(IItemJavaElementDescriptor o1,
				IItemJavaElementDescriptor o2) {
			String category1 = o1.getCategory(target);
			String category2 = o2.getCategory(target);
			if (category1 == null) {
				if (category2 != null)
					return -1;
			} else {
				if (category2 == null)
					return 1;

				int result = category1.compareTo(category2);
				if (result != 0)
					return result;
			}

			String name1 = o1.getDisplayName(target);
			String name2 = o2.getDisplayName(target);
			if (name1 == null)
				return name2 == null ? 0 : -1;

			if (name2 == null)
				return 1;

			return name1.compareTo(name2);
		}
	}
}