import { Diagnostic, TextEdit, FormattingOptions } from 'vscode-languageserver-types';
/**
 * Error codes that correspond to a given validation error. These
 * values are exposed for the purpose of allowing clients to identify
 * what kind of validation error has occurred and to then prompt
 * action that is appropriate for resolving the error to the user.
 *
 * Note that the names and values of these errors are not considered
 * to be API and may change between releases of dockerfiles-util.
 * This is because the Docker builder's errors and error messages are
 * itself not considered to be API. Thus, Dockerfiles that originally
 * would not build and throw an error may stop throwing an error
 * a future release of Docker due to newly added features. This would
 * then mean that an error (code and message) is no longer valid and
 * should thus be removed. Hence, this list of error codes is not
 * stable and as aforementioned may change between releases of
 * dockerfile-utils.
 */
export declare enum ValidationCode {
    CASING_INSTRUCTION = 0,
    CASING_DIRECTIVE = 1,
    ARGUMENT_MISSING = 2,
    ARGUMENT_EXTRA = 3,
    ARGUMENT_REQUIRES_ONE = 4,
    ARGUMENT_REQUIRES_AT_LEAST_ONE = 5,
    ARGUMENT_REQUIRES_TWO = 6,
    ARGUMENT_REQUIRES_AT_LEAST_TWO = 7,
    ARGUMENT_REQUIRES_ONE_OR_THREE = 8,
    ARGUMENT_UNNECESSARY = 9,
    DUPLICATE_BUILD_STAGE_NAME = 10,
    EMPTY_CONTINUATION_LINE = 11,
    INVALID_BUILD_STAGE_NAME = 12,
    FLAG_AT_LEAST_ONE = 13,
    FLAG_DUPLICATE = 14,
    FLAG_INVALID_DURATION = 15,
    FLAG_LESS_THAN_1MS = 16,
    FLAG_MISSING_DURATION = 17,
    FLAG_MISSING_VALUE = 18,
    FLAG_UNKNOWN_UNIT = 19,
    NO_SOURCE_IMAGE = 20,
    INVALID_ESCAPE_DIRECTIVE = 21,
    INVALID_AS = 22,
    INVALID_DESTINATION = 23,
    INVALID_PORT = 24,
    INVALID_PROTO = 25,
    /**
     * The error code used if the base image of a FROM instruction
     * has an invalid tag or digest specified.
     */
    INVALID_REFERENCE_FORMAT = 26,
    INVALID_SIGNAL = 27,
    INVALID_SYNTAX = 28,
    ONBUILD_CHAINING_DISALLOWED = 29,
    ONBUILD_TRIGGER_DISALLOWED = 30,
    SHELL_JSON_FORM = 31,
    SHELL_REQUIRES_ONE = 32,
    SYNTAX_MISSING_EQUALS = 33,
    SYNTAX_MISSING_NAMES = 34,
    SYNTAX_MISSING_SINGLE_QUOTE = 35,
    SYNTAX_MISSING_DOUBLE_QUOTE = 36,
    MULTIPLE_INSTRUCTIONS = 37,
    UNKNOWN_INSTRUCTION = 38,
    UNKNOWN_ADD_FLAG = 39,
    UNKNOWN_COPY_FLAG = 40,
    UNKNOWN_HEALTHCHECK_FLAG = 41,
    UNKNOWN_TYPE = 42,
    UNSUPPORTED_MODIFIER = 43,
    DEPRECATED_MAINTAINER = 44,
    HEALTHCHECK_CMD_ARGUMENT_MISSING = 45,
    FLAG_INVALID_FROM_VALUE = 46,
    /**
     * The error code used if an instruction has arguments written in
     * JSON form except that it is not actually valid JSON because
     * single quotes are used instead of double quotes.
     */
    JSON_IN_SINGLE_QUOTES = 47,
    /**
     * The error code used if a WORKDIR instrction does not point to
     * an absolute path.
     */
    WORKDIR_IS_NOT_ABSOLUTE = 48,
}
/**
 * The severity options that may be defined for the validator.
 */
export declare enum ValidationSeverity {
    /**
     * The value to set to ignore a problem that has been detected of
     * a certain type.
     */
    IGNORE = 0,
    /**
     * The value to set to classify a problem that has been detected
     * of a certain type as a warning.
     */
    WARNING = 1,
    /**
     * The value to set to classify a problem that has been detected
     * of a certain type as an error.
     */
    ERROR = 2,
}
/**
 * Settings for configuring if the validator should consider a
 * problem an error, a warning, or if the problem should be ignored
 * and not be reported.
 */
export interface ValidatorSettings {
    /**
     * The setting for flagging any deprecated MAINTAINER
     * instructions that are found in the Dockerfile.
     */
    deprecatedMaintainer?: ValidationSeverity;
    /**
     * The setting for flagging directives that are not all written
     * in lowercase.
     */
    directiveCasing?: ValidationSeverity;
    /**
     * The setting for flagging instructions that span multiple lines
     * and contain empty lines (or lines that only contain whitespace
     * characters).
     */
    emptyContinuationLine?: ValidationSeverity;
    /**
     * The setting for flagging instructions that are not all written
     * in uppercase.
     */
    instructionCasing?: ValidationSeverity;
    instructionCmdMultiple?: ValidationSeverity;
    instructionEntrypointMultiple?: ValidationSeverity;
    instructionHealthcheckMultiple?: ValidationSeverity;
    instructionJSONInSingleQuotes?: ValidationSeverity;
    /**
     * The setting for flagging WORKDIR instructions that do not use
     * an absolute path.
     */
    instructionWorkdirRelative?: ValidationSeverity;
}
/**
 * Analyzes the Dockerfile contained within the given string and
 * provides an array of TextEdits back for formatting the document.
 */
export declare function format(content: string, options: FormattingOptions): TextEdit[];
/**
 * Validates the Dockerfile that is contained in the given string.
 */
export declare function validate(content: string, settings?: ValidatorSettings): Diagnostic[];
