"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
// @ts-check
var chai_1 = require("chai");
require("mocha");
var _substituter_1 = require("../_substituter");
describe('substituter', function () {
    it('should replace property value with x', function () {
        chai_1.assert.deepEqual(performSubstitutions([
            'width: 1px;',
            "color: ${'red'};",
            'color: red;',
        ].join('\n')), [
            'width: 1px;',
            "color: xxxxxxxx;",
            'color: red;',
        ].join('\n'));
    });
    it('should insert whitespace when placeholder is used a entire property', function () {
        chai_1.assert.deepEqual(performSubstitutions([
            'width: 1px;',
            "${'color: red;'}",
            'color: red;',
        ].join('\n')), [
            'width: 1px;',
            "                ",
            'color: red;',
        ].join('\n'));
    });
    it('should insert a false property when placeholder is used a entire property with trailing semi-colon', function () {
        chai_1.assert.deepEqual(performSubstitutions([
            'width: 1px;',
            "${'color: red'};",
            'color: red;',
        ].join('\n')), [
            'width: 1px;',
            "$a:0           ;",
            'color: red;',
        ].join('\n'));
    });
    it('should add a zero for percent units', function () {
        chai_1.assert.deepEqual(performSubstitutions('width: ${10}%;'), 'width: 00000%;');
    });
    it('should replace property with fake proeprty when placeholder is used in name (#52)', function () {
        chai_1.assert.deepEqual(performSubstitutions([
            'width: 1px;',
            "${123}: 1px;",
            'color: red;',
        ].join('\n')), [
            'width: 1px;',
            "$axxxx: 1px;",
            'color: red;',
        ].join('\n'));
    });
    it('should insert x for placeholder used as rule', function () {
        chai_1.assert.deepEqual(performSubstitutions([
            '${"button"} {',
            'color: ${"red"};',
            '}',
        ].join('\n')), [
            'xxxxxxxxxxx {',
            'color: xxxxxxxx;',
            '}',
        ].join('\n'));
    });
    it('should insert x for placeholder used as part of a rule (#59)', function () {
        chai_1.assert.deepEqual(performSubstitutions([
            '${"button"}, ${"a"} {',
            'color: ${"red"};',
            '}',
        ].join('\n')), [
            'xxxxxxxxxxx, xxxxxx {',
            'color: xxxxxxxx;',
            '}',
        ].join('\n'));
    });
    it('should fake out property name when inside nested rule (#54)', function () {
        chai_1.assert.deepEqual(performSubstitutions([
            '&.buu-foo {',
            '  \${"baseShape"};',
            '  &.active {',
            '    font-size: 2rem;',
            '  }',
            '}',
        ].join('\n')), [
            '&.buu-foo {',
            '  $a:0          ;',
            '  &.active {',
            '    font-size: 2rem;',
            '  }',
            '}',
        ].join('\n'));
    });
    it('should add zeros for color units (#60)', function () {
        chai_1.assert.deepEqual(performSubstitutions('color: #${1};'), 'color: #000 ;');
    });
    it('should replace adjacent variables with x (#62)', function () {
        chai_1.assert.deepEqual(performSubstitutions([
            "margin: ${'1px'}${'1px'};",
            "padding: ${'1px'} ${'1px'};",
        ].join('\n')), [
            "margin: xxxxxxxxxxxxxxxx;",
            "padding: xxxxxxxx xxxxxxxx;",
        ].join('\n'));
    });
    it('should replace placeholder that spans multiple lines with x (#44)', function () {
        chai_1.assert.deepEqual(performSubstitutions([
            'background:',
            "  ${'transparent'};",
        ].join('\n')), [
            'background:',
            '  xxxxxxxxxxxxxxxx;',
        ].join('\n'));
    });
    it('should replace placeholder used in contextual selector (#71)', function () {
        chai_1.assert.deepEqual(performSubstitutions([
            'position: relative;',
            '',
            '${FlipContainer}:hover & {',
            '   transform: rotateY(180deg);',
            '}',
        ].join('\n')), [
            'position: relative;',
            '',
            '&               :hover & {',
            '   transform: rotateY(180deg);',
            '}',
        ].join('\n'));
    });
    it('should replace placeholder used in child selector (#75)', function () {
        chai_1.assert.deepEqual(performSubstitutions([
            'position: relative;',
            '> ${FlipContainer}:hover {',
            '   color: red;',
            '}',
        ].join('\n')), [
            'position: relative;',
            '> &               :hover {',
            '   color: red;',
            '}',
        ].join('\n'));
    });
});
function performSubstitutions(value) {
    return _substituter_1.getSubstitutions(value, getSpans(value));
}
function getSpans(value) {
    var spans = [];
    var re = /(\$\{[^}]*\})/g;
    var match = re.exec(value);
    while (match) {
        spans.push({ start: match.index, end: match.index + match[0].length });
        match = re.exec(value);
    }
    return spans;
}
