import { CustomSchemaProvider, SchemaAdditions, SchemaDeletions } from './services/yamlSchemaService';
import { TextDocument, Position, CompletionList, Diagnostic, Hover, SymbolInformation, DocumentSymbol, TextEdit, DocumentLink } from 'vscode-languageserver-types';
import { JSONSchema } from './jsonSchema';
import { JSONDocument, DefinitionLink } from 'vscode-json-languageservice';
export interface LanguageSettings {
    validate?: boolean;
    hover?: boolean;
    completion?: boolean;
    format?: boolean;
    isKubernetes?: boolean;
    schemas?: any[];
    customTags?: Array<string>;
    /**
     * Default indentation size
     */
    indentation?: string;
}
export interface WorkspaceContextService {
    resolveRelativePath(relativePath: string, resource: string): string;
}
/**
 * The schema request service is used to fetch schemas. The result should the schema file comment, or,
 * in case of an error, a displayable error string
 */
export interface SchemaRequestService {
    (uri: string): Promise<string>;
}
export interface SchemaConfiguration {
    /**
     * The URI of the schema, which is also the identifier of the schema.
     */
    uri: string;
    /**
     * A list of file names that are associated to the schema. The '*' wildcard can be used. For example '*.schema.json', 'package.json'
     */
    fileMatch?: string[];
    /**
     * The schema for the given URI.
     * If no schema is provided, the schema will be fetched with the schema request service (if available).
     */
    schema?: JSONSchema;
}
export interface CustomFormatterOptions {
    singleQuote?: boolean;
    bracketSpacing?: boolean;
    proseWrap?: string;
    printWidth?: number;
    enable?: boolean;
}
export interface LanguageService {
    configure(settings: LanguageSettings): void;
    registerCustomSchemaProvider(schemaProvider: CustomSchemaProvider): void;
    doComplete(document: TextDocument, position: Position, isKubernetes: boolean): Promise<CompletionList>;
    doValidation(document: TextDocument, isKubernetes: boolean): Promise<Diagnostic[]>;
    doHover(document: TextDocument, position: Position): Promise<Hover | null>;
    findDocumentSymbols(document: TextDocument): SymbolInformation[];
    findDocumentSymbols2(document: TextDocument): DocumentSymbol[];
    findDefinition(document: TextDocument, position: Position, doc: JSONDocument): Promise<DefinitionLink[]>;
    findLinks(document: TextDocument): Promise<DocumentLink[]>;
    resetSchema(uri: string): boolean;
    doFormat(document: TextDocument, options: CustomFormatterOptions): TextEdit[];
    addSchema(schemaID: string, schema: JSONSchema): void;
    deleteSchema(schemaID: string): void;
    modifySchemaContent(schemaAdditions: SchemaAdditions): void;
    deleteSchemaContent(schemaDeletions: SchemaDeletions): void;
}
export declare function getLanguageService(schemaRequestService: SchemaRequestService, workspaceContext: WorkspaceContextService): LanguageService;
