/* --------------------------------------------------------------------------------------------
 * Copyright (c) Remy Suen. All rights reserved.
 * Licensed under the MIT License. See License.txt in the project root for license information.
 * ------------------------------------------------------------------------------------------ */
'use strict';
Object.defineProperty(exports, "__esModule", { value: true });
const vscode_languageserver_textdocument_1 = require("vscode-languageserver-textdocument");
const dockerFormatter_1 = require("./dockerFormatter");
const dockerValidator_1 = require("./dockerValidator");
/**
 * Error codes that correspond to a given validation error. These
 * values are exposed for the purpose of allowing clients to identify
 * what kind of validation error has occurred and to then prompt
 * action that is appropriate for resolving the error to the user.
 *
 * Note that the names and values of these errors are not considered
 * to be API and may change between releases of dockerfiles-util.
 * This is because the Docker builder's errors and error messages are
 * itself not considered to be API. Thus, Dockerfiles that originally
 * would not build and throw an error may stop throwing an error
 * a future release of Docker due to newly added features. This would
 * then mean that an error (code and message) is no longer valid and
 * should thus be removed. Hence, this list of error codes is not
 * stable and as aforementioned may change between releases of
 * dockerfile-utils.
 */
var ValidationCode;
(function (ValidationCode) {
    ValidationCode[ValidationCode["CASING_INSTRUCTION"] = 0] = "CASING_INSTRUCTION";
    ValidationCode[ValidationCode["CASING_DIRECTIVE"] = 1] = "CASING_DIRECTIVE";
    ValidationCode[ValidationCode["ARGUMENT_MISSING"] = 2] = "ARGUMENT_MISSING";
    ValidationCode[ValidationCode["ARGUMENT_EXTRA"] = 3] = "ARGUMENT_EXTRA";
    ValidationCode[ValidationCode["ARGUMENT_REQUIRES_ONE"] = 4] = "ARGUMENT_REQUIRES_ONE";
    ValidationCode[ValidationCode["ARGUMENT_REQUIRES_AT_LEAST_ONE"] = 5] = "ARGUMENT_REQUIRES_AT_LEAST_ONE";
    ValidationCode[ValidationCode["ARGUMENT_REQUIRES_TWO"] = 6] = "ARGUMENT_REQUIRES_TWO";
    ValidationCode[ValidationCode["ARGUMENT_REQUIRES_AT_LEAST_TWO"] = 7] = "ARGUMENT_REQUIRES_AT_LEAST_TWO";
    ValidationCode[ValidationCode["ARGUMENT_REQUIRES_ONE_OR_THREE"] = 8] = "ARGUMENT_REQUIRES_ONE_OR_THREE";
    ValidationCode[ValidationCode["ARGUMENT_UNNECESSARY"] = 9] = "ARGUMENT_UNNECESSARY";
    ValidationCode[ValidationCode["DUPLICATE_BUILD_STAGE_NAME"] = 10] = "DUPLICATE_BUILD_STAGE_NAME";
    ValidationCode[ValidationCode["EMPTY_CONTINUATION_LINE"] = 11] = "EMPTY_CONTINUATION_LINE";
    ValidationCode[ValidationCode["INVALID_BUILD_STAGE_NAME"] = 12] = "INVALID_BUILD_STAGE_NAME";
    ValidationCode[ValidationCode["FLAG_AT_LEAST_ONE"] = 13] = "FLAG_AT_LEAST_ONE";
    ValidationCode[ValidationCode["FLAG_DUPLICATE"] = 14] = "FLAG_DUPLICATE";
    ValidationCode[ValidationCode["FLAG_INVALID_DURATION"] = 15] = "FLAG_INVALID_DURATION";
    ValidationCode[ValidationCode["FLAG_LESS_THAN_1MS"] = 16] = "FLAG_LESS_THAN_1MS";
    ValidationCode[ValidationCode["FLAG_MISSING_DURATION"] = 17] = "FLAG_MISSING_DURATION";
    ValidationCode[ValidationCode["FLAG_MISSING_VALUE"] = 18] = "FLAG_MISSING_VALUE";
    ValidationCode[ValidationCode["FLAG_UNKNOWN_UNIT"] = 19] = "FLAG_UNKNOWN_UNIT";
    ValidationCode[ValidationCode["NO_SOURCE_IMAGE"] = 20] = "NO_SOURCE_IMAGE";
    ValidationCode[ValidationCode["INVALID_ESCAPE_DIRECTIVE"] = 21] = "INVALID_ESCAPE_DIRECTIVE";
    ValidationCode[ValidationCode["INVALID_AS"] = 22] = "INVALID_AS";
    ValidationCode[ValidationCode["INVALID_DESTINATION"] = 23] = "INVALID_DESTINATION";
    ValidationCode[ValidationCode["INVALID_PORT"] = 24] = "INVALID_PORT";
    ValidationCode[ValidationCode["INVALID_PROTO"] = 25] = "INVALID_PROTO";
    /**
     * The error code used if the base image of a FROM instruction
     * has an invalid tag or digest specified.
     */
    ValidationCode[ValidationCode["INVALID_REFERENCE_FORMAT"] = 26] = "INVALID_REFERENCE_FORMAT";
    ValidationCode[ValidationCode["INVALID_SIGNAL"] = 27] = "INVALID_SIGNAL";
    ValidationCode[ValidationCode["INVALID_SYNTAX"] = 28] = "INVALID_SYNTAX";
    ValidationCode[ValidationCode["ONBUILD_CHAINING_DISALLOWED"] = 29] = "ONBUILD_CHAINING_DISALLOWED";
    ValidationCode[ValidationCode["ONBUILD_TRIGGER_DISALLOWED"] = 30] = "ONBUILD_TRIGGER_DISALLOWED";
    ValidationCode[ValidationCode["SHELL_JSON_FORM"] = 31] = "SHELL_JSON_FORM";
    ValidationCode[ValidationCode["SHELL_REQUIRES_ONE"] = 32] = "SHELL_REQUIRES_ONE";
    ValidationCode[ValidationCode["SYNTAX_MISSING_EQUALS"] = 33] = "SYNTAX_MISSING_EQUALS";
    ValidationCode[ValidationCode["SYNTAX_MISSING_NAMES"] = 34] = "SYNTAX_MISSING_NAMES";
    ValidationCode[ValidationCode["SYNTAX_MISSING_SINGLE_QUOTE"] = 35] = "SYNTAX_MISSING_SINGLE_QUOTE";
    ValidationCode[ValidationCode["SYNTAX_MISSING_DOUBLE_QUOTE"] = 36] = "SYNTAX_MISSING_DOUBLE_QUOTE";
    ValidationCode[ValidationCode["MULTIPLE_INSTRUCTIONS"] = 37] = "MULTIPLE_INSTRUCTIONS";
    ValidationCode[ValidationCode["UNKNOWN_INSTRUCTION"] = 38] = "UNKNOWN_INSTRUCTION";
    ValidationCode[ValidationCode["UNKNOWN_ADD_FLAG"] = 39] = "UNKNOWN_ADD_FLAG";
    ValidationCode[ValidationCode["UNKNOWN_COPY_FLAG"] = 40] = "UNKNOWN_COPY_FLAG";
    ValidationCode[ValidationCode["UNKNOWN_FROM_FLAG"] = 41] = "UNKNOWN_FROM_FLAG";
    ValidationCode[ValidationCode["UNKNOWN_HEALTHCHECK_FLAG"] = 42] = "UNKNOWN_HEALTHCHECK_FLAG";
    ValidationCode[ValidationCode["UNKNOWN_TYPE"] = 43] = "UNKNOWN_TYPE";
    ValidationCode[ValidationCode["UNSUPPORTED_MODIFIER"] = 44] = "UNSUPPORTED_MODIFIER";
    ValidationCode[ValidationCode["DEPRECATED_MAINTAINER"] = 45] = "DEPRECATED_MAINTAINER";
    ValidationCode[ValidationCode["HEALTHCHECK_CMD_ARGUMENT_MISSING"] = 46] = "HEALTHCHECK_CMD_ARGUMENT_MISSING";
    ValidationCode[ValidationCode["FLAG_INVALID_FROM_VALUE"] = 47] = "FLAG_INVALID_FROM_VALUE";
    /**
     * The error code used if an instruction has arguments written in
     * JSON form except that it is not actually valid JSON because
     * single quotes are used instead of double quotes.
     */
    ValidationCode[ValidationCode["JSON_IN_SINGLE_QUOTES"] = 48] = "JSON_IN_SINGLE_QUOTES";
    /**
     * The error code used if a WORKDIR instruction does not point to
     * an absolute path.
     */
    ValidationCode[ValidationCode["WORKDIR_IS_NOT_ABSOLUTE"] = 49] = "WORKDIR_IS_NOT_ABSOLUTE";
    /**
     * The error code used if a FROM instruction uses a variable to
     * reference its base image but the variable does not exist.
     */
    ValidationCode[ValidationCode["BASE_NAME_EMPTY"] = 50] = "BASE_NAME_EMPTY";
    /**
     * The error code used if more than one escape parser directive is
     * defined.
     */
    ValidationCode[ValidationCode["DUPLICATED_ESCAPE_DIRECTIVE"] = 51] = "DUPLICATED_ESCAPE_DIRECTIVE";
})(ValidationCode = exports.ValidationCode || (exports.ValidationCode = {}));
/**
 * The severity options that may be defined for the validator.
 */
var ValidationSeverity;
(function (ValidationSeverity) {
    /**
     * The value to set to ignore a problem that has been detected of
     * a certain type.
     */
    ValidationSeverity[ValidationSeverity["IGNORE"] = 0] = "IGNORE";
    /**
     * The value to set to classify a problem that has been detected
     * of a certain type as a warning.
     */
    ValidationSeverity[ValidationSeverity["WARNING"] = 1] = "WARNING";
    /**
     * The value to set to classify a problem that has been detected
     * of a certain type as an error.
     */
    ValidationSeverity[ValidationSeverity["ERROR"] = 2] = "ERROR";
})(ValidationSeverity = exports.ValidationSeverity || (exports.ValidationSeverity = {}));
/**
 * Analyzes the Dockerfile contained within the given string and
 * provides an array of TextEdits back for formatting the document.
 */
function format(content, options) {
    const document = vscode_languageserver_textdocument_1.TextDocument.create("", "", 0, content);
    let formatter = new dockerFormatter_1.DockerFormatter();
    return formatter.formatDocument(document, options);
}
exports.format = format;
function formatRange(content, range, settings) {
    const document = vscode_languageserver_textdocument_1.TextDocument.create("", "", 0, content);
    const formatter = new dockerFormatter_1.DockerFormatter();
    return formatter.formatRange(document, range, settings);
}
exports.formatRange = formatRange;
function formatOnType(content, position, character, settings) {
    const document = vscode_languageserver_textdocument_1.TextDocument.create("", "", 0, content);
    const formatter = new dockerFormatter_1.DockerFormatter();
    return formatter.formatOnType(document, position, character, settings);
}
exports.formatOnType = formatOnType;
/**
 * Validates the Dockerfile that is contained in the given string.
 */
function validate(content, settings) {
    const document = vscode_languageserver_textdocument_1.TextDocument.create("", "", 0, content);
    const validator = new dockerValidator_1.Validator(settings);
    return validator.validate(document);
}
exports.validate = validate;
