"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Red Hat. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
const vscode_languageserver_types_1 = require("vscode-languageserver-types");
const yamlSettings_1 = require("../src/yamlSettings");
const serviceSetup_1 = require("./utils/serviceSetup");
const testHelper_1 = require("./utils/testHelper");
const chai_1 = require("chai");
const verifyError_1 = require("./utils/verifyError");
describe('YAML Validation Tests', () => {
    let languageSettingsSetup;
    let validationHandler;
    let yamlSettings;
    before(() => {
        languageSettingsSetup = new serviceSetup_1.ServiceSetup().withValidate();
        const { validationHandler: valHandler, yamlSettings: settings } = (0, testHelper_1.setupLanguageService)(languageSettingsSetup.languageSettings);
        validationHandler = valHandler;
        yamlSettings = settings;
    });
    function parseSetup(content, customSchemaID) {
        const testTextDocument = (0, testHelper_1.setupSchemaIDTextDocument)(content, customSchemaID);
        yamlSettings.documents = new yamlSettings_1.TextDocumentTestManager();
        yamlSettings.documents.set(testTextDocument);
        return validationHandler.validateTextDocument(testTextDocument);
    }
    describe('TAB Character diagnostics', () => {
        it('Should report if TAB character present', () => __awaiter(void 0, void 0, void 0, function* () {
            const yaml = 'foo:\n\t- bar';
            const result = yield parseSetup(yaml);
            (0, chai_1.expect)(result).is.not.empty;
            (0, chai_1.expect)(result.length).to.be.equal(1);
            (0, chai_1.expect)(result[0]).deep.equal((0, verifyError_1.createExpectedError)('Tabs are not allowed as indentation', 1, 0, 1, 6));
        }));
        it('Should report one error for TAB character present in a row', () => __awaiter(void 0, void 0, void 0, function* () {
            const yaml = 'foo:\n\t\t- bar';
            const result = yield parseSetup(yaml);
            (0, chai_1.expect)(result).is.not.empty;
            (0, chai_1.expect)(result.length).to.be.equal(1);
            (0, chai_1.expect)(result[0]).deep.equal((0, verifyError_1.createExpectedError)('Tabs are not allowed as indentation', 1, 0, 1, 7));
        }));
        it('Should report one error for TAB`s characters present in the middle of indentation', () => __awaiter(void 0, void 0, void 0, function* () {
            const yaml = 'foo:\n \t\t\t - bar';
            const result = yield parseSetup(yaml);
            (0, chai_1.expect)(result).is.not.empty;
            (0, chai_1.expect)(result.length).to.be.equal(1);
            (0, chai_1.expect)(result[0]).deep.equal((0, verifyError_1.createExpectedError)('Tabs are not allowed as indentation', 1, 1, 1, 10));
        }));
    });
    describe('Unused anchors diagnostics', () => {
        it('should report unused anchor', () => __awaiter(void 0, void 0, void 0, function* () {
            const yaml = 'foo: &bar bar\n';
            const result = yield parseSetup(yaml);
            (0, chai_1.expect)(result).is.not.empty;
            (0, chai_1.expect)(result.length).to.be.equal(1);
            (0, chai_1.expect)(result[0]).deep.equal((0, verifyError_1.createUnusedAnchorDiagnostic)('Unused anchor "&bar"', 0, 5, 0, 9));
        }));
        it('should not report used anchor', () => __awaiter(void 0, void 0, void 0, function* () {
            const yaml = 'foo: &bar bar\nfff: *bar';
            const result = yield parseSetup(yaml);
            (0, chai_1.expect)(result).is.empty;
        }));
        it('should report unused anchors in array ', () => __awaiter(void 0, void 0, void 0, function* () {
            const yaml = `foo: &bar   doe
aaa: some
dd: *ba
some: 
  &a ss: ss
&aa ff: 
  - s
  - o
  - &e m
  - e`;
            const result = yield parseSetup(yaml);
            (0, chai_1.expect)(result).is.not.empty;
            (0, chai_1.expect)(result.length).to.be.equal(4);
            (0, chai_1.expect)(result).to.include.deep.members([
                (0, verifyError_1.createUnusedAnchorDiagnostic)('Unused anchor "&bar"', 0, 5, 0, 9),
                (0, verifyError_1.createUnusedAnchorDiagnostic)('Unused anchor "&a"', 4, 2, 4, 4),
                (0, verifyError_1.createUnusedAnchorDiagnostic)('Unused anchor "&aa"', 5, 0, 5, 3),
                (0, verifyError_1.createUnusedAnchorDiagnostic)('Unused anchor "&e"', 8, 4, 8, 6),
            ]);
        }));
    });
    describe(`YAML styles test`, () => {
        it('should not report flow style', () => __awaiter(void 0, void 0, void 0, function* () {
            const yaml = `host: phl-42  
datacenter:   
  location: canada  
  cab: 15  
animals:  
  - dog  
  - cat  
  - mouse`;
            const result = yield parseSetup(yaml);
            (0, chai_1.expect)(result).to.be.empty;
        }));
        it('should report flow style', () => __awaiter(void 0, void 0, void 0, function* () {
            const yaml = `host: phl-42  
datacenter: {location: canada , cab: 15}  
animals: [dog , cat , mouse]  `;
            yamlSettings.style = {
                flowMapping: 'forbid',
                flowSequence: 'forbid',
            };
            languageSettingsSetup = new serviceSetup_1.ServiceSetup().withValidate().withFlowMapping('forbid').withFlowSequence('forbid');
            const { validationHandler: valHandler, yamlSettings: settings } = (0, testHelper_1.setupLanguageService)(languageSettingsSetup.languageSettings);
            validationHandler = valHandler;
            yamlSettings = settings;
            const result = yield parseSetup(yaml);
            (0, chai_1.expect)(result).not.to.be.empty;
            (0, chai_1.expect)(result.length).to.be.equal(2);
            (0, chai_1.expect)(result).to.include.deep.members([
                (0, verifyError_1.createExpectedError)('Flow style mapping is forbidden', 1, 12, 1, 42, vscode_languageserver_types_1.DiagnosticSeverity.Error, 'YAML', 'flowMap'),
                (0, verifyError_1.createExpectedError)('Flow style sequence is forbidden', 2, 9, 2, 28, vscode_languageserver_types_1.DiagnosticSeverity.Error, 'YAML', 'flowSeq'),
            ]);
        }));
        it('should report only sequence when flow mapping is allow', () => __awaiter(void 0, void 0, void 0, function* () {
            const yaml = `host: phl-42  
datacenter: {location: canada , cab: 15}  
animals: [dog , cat , mouse]  `;
            yamlSettings.style = {
                flowMapping: 'forbid',
                flowSequence: 'forbid',
            };
            languageSettingsSetup = new serviceSetup_1.ServiceSetup().withValidate().withFlowMapping('allow').withFlowSequence('forbid');
            const { validationHandler: valHandler, yamlSettings: settings } = (0, testHelper_1.setupLanguageService)(languageSettingsSetup.languageSettings);
            validationHandler = valHandler;
            yamlSettings = settings;
            const result = yield parseSetup(yaml);
            (0, chai_1.expect)(result).not.to.be.empty;
            (0, chai_1.expect)(result.length).to.be.equal(1);
            (0, chai_1.expect)(result).to.include.deep.members([
                (0, verifyError_1.createExpectedError)('Flow style sequence is forbidden', 2, 9, 2, 28, vscode_languageserver_types_1.DiagnosticSeverity.Error, 'YAML', 'flowSeq'),
            ]);
        }));
        it('should report flow error for empty map & sequence', () => __awaiter(void 0, void 0, void 0, function* () {
            const yaml = 'object: {} \nobject2: []';
            yamlSettings.style = {
                flowMapping: 'forbid',
                flowSequence: 'forbid',
            };
            languageSettingsSetup = new serviceSetup_1.ServiceSetup().withValidate().withFlowMapping('forbid').withFlowSequence('forbid');
            const { validationHandler: valHandler, yamlSettings: settings } = (0, testHelper_1.setupLanguageService)(languageSettingsSetup.languageSettings);
            validationHandler = valHandler;
            yamlSettings = settings;
            const result = yield parseSetup(yaml);
            (0, chai_1.expect)(result).not.to.be.empty;
            (0, chai_1.expect)(result.length).to.be.equal(2);
            (0, chai_1.expect)(result).to.include.deep.members([
                (0, verifyError_1.createExpectedError)('Flow style mapping is forbidden', 0, 8, 0, 11, vscode_languageserver_types_1.DiagnosticSeverity.Error, 'YAML', 'flowMap'),
                (0, verifyError_1.createExpectedError)('Flow style sequence is forbidden', 1, 9, 1, 10, vscode_languageserver_types_1.DiagnosticSeverity.Error, 'YAML', 'flowSeq'),
            ]);
        }));
    });
});
//# sourceMappingURL=yamlValidation.test.js.map