package org.mandas.docker.client.messages;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link NetworkConfig}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableNetworkConfig.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableNetworkConfig implements NetworkConfig {
  private final String name;
  private final @Nullable String driver;
  private final @Nullable Ipam ipam;
  private final Map<String, String> options;
  private final @Nullable Boolean checkDuplicate;
  private final @Nullable Boolean internal;
  private final @Nullable Boolean enableIPv6;
  private final @Nullable Boolean attachable;
  private final @Nullable Map<String, String> labels;

  private ImmutableNetworkConfig(
      String name,
      @Nullable String driver,
      @Nullable Ipam ipam,
      Map<String, String> options,
      @Nullable Boolean checkDuplicate,
      @Nullable Boolean internal,
      @Nullable Boolean enableIPv6,
      @Nullable Boolean attachable,
      @Nullable Map<String, String> labels) {
    this.name = name;
    this.driver = driver;
    this.ipam = ipam;
    this.options = options;
    this.checkDuplicate = checkDuplicate;
    this.internal = internal;
    this.enableIPv6 = enableIPv6;
    this.attachable = attachable;
    this.labels = labels;
  }

  /**
   * @return The value of the {@code name} attribute
   */
  @JsonProperty("Name")
  @Override
  public String name() {
    return name;
  }

  /**
   * @return The value of the {@code driver} attribute
   */
  @JsonProperty("Driver")
  @Override
  public @Nullable String driver() {
    return driver;
  }

  /**
   * @return The value of the {@code ipam} attribute
   */
  @JsonProperty("IPAM")
  @Override
  public @Nullable Ipam ipam() {
    return ipam;
  }

  /**
   * @return The value of the {@code options} attribute
   */
  @JsonProperty("Options")
  @Override
  public Map<String, String> options() {
    return options;
  }

  /**
   * @return The value of the {@code checkDuplicate} attribute
   */
  @JsonProperty("CheckDuplicate")
  @Override
  public @Nullable Boolean checkDuplicate() {
    return checkDuplicate;
  }

  /**
   * @return The value of the {@code internal} attribute
   */
  @JsonProperty("Internal")
  @Override
  public @Nullable Boolean internal() {
    return internal;
  }

  /**
   * @return The value of the {@code enableIPv6} attribute
   */
  @JsonProperty("EnableIPv6")
  @Override
  public @Nullable Boolean enableIPv6() {
    return enableIPv6;
  }

  /**
   * @return The value of the {@code attachable} attribute
   */
  @JsonProperty("Attachable")
  @Override
  public @Nullable Boolean attachable() {
    return attachable;
  }

  /**
   * @return The value of the {@code labels} attribute
   */
  @JsonProperty("Labels")
  @Override
  public @Nullable Map<String, String> labels() {
    return labels;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link NetworkConfig#name() name} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for name
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableNetworkConfig withName(String value) {
    String newValue = Objects.requireNonNull(value, "name");
    if (this.name.equals(newValue)) return this;
    return new ImmutableNetworkConfig(
        newValue,
        this.driver,
        this.ipam,
        this.options,
        this.checkDuplicate,
        this.internal,
        this.enableIPv6,
        this.attachable,
        this.labels);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link NetworkConfig#driver() driver} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for driver (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableNetworkConfig withDriver(@Nullable String value) {
    if (Objects.equals(this.driver, value)) return this;
    return new ImmutableNetworkConfig(
        this.name,
        value,
        this.ipam,
        this.options,
        this.checkDuplicate,
        this.internal,
        this.enableIPv6,
        this.attachable,
        this.labels);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link NetworkConfig#ipam() ipam} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for ipam (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableNetworkConfig withIpam(@Nullable Ipam value) {
    if (this.ipam == value) return this;
    return new ImmutableNetworkConfig(
        this.name,
        this.driver,
        value,
        this.options,
        this.checkDuplicate,
        this.internal,
        this.enableIPv6,
        this.attachable,
        this.labels);
  }

  /**
   * Copy the current immutable object by replacing the {@link NetworkConfig#options() options} map with the specified map.
   * Nulls are not permitted as keys or values.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param entries The entries to replace the options map
   * @return A modified copy or {@code this} if not changed
   */
  public final ImmutableNetworkConfig withOptions(Map<String, ? extends String> entries) {
    if (this.options == entries) return this;
    Map<String, String> newValue = createUnmodifiableMap(true, false, entries);
    return new ImmutableNetworkConfig(
        this.name,
        this.driver,
        this.ipam,
        newValue,
        this.checkDuplicate,
        this.internal,
        this.enableIPv6,
        this.attachable,
        this.labels);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link NetworkConfig#checkDuplicate() checkDuplicate} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for checkDuplicate (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableNetworkConfig withCheckDuplicate(@Nullable Boolean value) {
    if (Objects.equals(this.checkDuplicate, value)) return this;
    return new ImmutableNetworkConfig(
        this.name,
        this.driver,
        this.ipam,
        this.options,
        value,
        this.internal,
        this.enableIPv6,
        this.attachable,
        this.labels);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link NetworkConfig#internal() internal} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for internal (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableNetworkConfig withInternal(@Nullable Boolean value) {
    if (Objects.equals(this.internal, value)) return this;
    return new ImmutableNetworkConfig(
        this.name,
        this.driver,
        this.ipam,
        this.options,
        this.checkDuplicate,
        value,
        this.enableIPv6,
        this.attachable,
        this.labels);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link NetworkConfig#enableIPv6() enableIPv6} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for enableIPv6 (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableNetworkConfig withEnableIPv6(@Nullable Boolean value) {
    if (Objects.equals(this.enableIPv6, value)) return this;
    return new ImmutableNetworkConfig(
        this.name,
        this.driver,
        this.ipam,
        this.options,
        this.checkDuplicate,
        this.internal,
        value,
        this.attachable,
        this.labels);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link NetworkConfig#attachable() attachable} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for attachable (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableNetworkConfig withAttachable(@Nullable Boolean value) {
    if (Objects.equals(this.attachable, value)) return this;
    return new ImmutableNetworkConfig(
        this.name,
        this.driver,
        this.ipam,
        this.options,
        this.checkDuplicate,
        this.internal,
        this.enableIPv6,
        value,
        this.labels);
  }

  /**
   * Copy the current immutable object by replacing the {@link NetworkConfig#labels() labels} map with the specified map.
   * Nulls are not permitted as keys or values.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param entries The entries to replace the labels map
   * @return A modified copy or {@code this} if not changed
   */
  public final ImmutableNetworkConfig withLabels(@Nullable Map<String, ? extends String> entries) {
    if (this.labels == entries) return this;
    @Nullable Map<String, String> newValue = entries == null ? null : createUnmodifiableMap(true, false, entries);
    return new ImmutableNetworkConfig(
        this.name,
        this.driver,
        this.ipam,
        this.options,
        this.checkDuplicate,
        this.internal,
        this.enableIPv6,
        this.attachable,
        newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableNetworkConfig} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableNetworkConfig
        && equalsByValue((ImmutableNetworkConfig) another);
  }

  private boolean equalsByValue(ImmutableNetworkConfig another) {
    return name.equals(another.name)
        && Objects.equals(driver, another.driver)
        && Objects.equals(ipam, another.ipam)
        && options.equals(another.options)
        && Objects.equals(checkDuplicate, another.checkDuplicate)
        && Objects.equals(internal, another.internal)
        && Objects.equals(enableIPv6, another.enableIPv6)
        && Objects.equals(attachable, another.attachable)
        && Objects.equals(labels, another.labels);
  }

  /**
   * Computes a hash code from attributes: {@code name}, {@code driver}, {@code ipam}, {@code options}, {@code checkDuplicate}, {@code internal}, {@code enableIPv6}, {@code attachable}, {@code labels}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + name.hashCode();
    h += (h << 5) + Objects.hashCode(driver);
    h += (h << 5) + Objects.hashCode(ipam);
    h += (h << 5) + options.hashCode();
    h += (h << 5) + Objects.hashCode(checkDuplicate);
    h += (h << 5) + Objects.hashCode(internal);
    h += (h << 5) + Objects.hashCode(enableIPv6);
    h += (h << 5) + Objects.hashCode(attachable);
    h += (h << 5) + Objects.hashCode(labels);
    return h;
  }

  /**
   * Prints the immutable value {@code NetworkConfig} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "NetworkConfig{"
        + "name=" + name
        + ", driver=" + driver
        + ", ipam=" + ipam
        + ", options=" + options
        + ", checkDuplicate=" + checkDuplicate
        + ", internal=" + internal
        + ", enableIPv6=" + enableIPv6
        + ", attachable=" + attachable
        + ", labels=" + labels
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link NetworkConfig} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable NetworkConfig instance
   */
  public static ImmutableNetworkConfig copyOf(NetworkConfig instance) {
    if (instance instanceof ImmutableNetworkConfig) {
      return (ImmutableNetworkConfig) instance;
    }
    return ImmutableNetworkConfig.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableNetworkConfig ImmutableNetworkConfig}.
   * <pre>
   * ImmutableNetworkConfig.builder()
   *    .name(String) // required {@link NetworkConfig#name() name}
   *    .driver(String | null) // nullable {@link NetworkConfig#driver() driver}
   *    .ipam(org.mandas.docker.client.messages.Ipam | null) // nullable {@link NetworkConfig#ipam() ipam}
   *    .addOption|putAllOptions(String =&gt; String) // {@link NetworkConfig#options() options} mappings
   *    .checkDuplicate(Boolean | null) // nullable {@link NetworkConfig#checkDuplicate() checkDuplicate}
   *    .internal(Boolean | null) // nullable {@link NetworkConfig#internal() internal}
   *    .enableIPv6(Boolean | null) // nullable {@link NetworkConfig#enableIPv6() enableIPv6}
   *    .attachable(Boolean | null) // nullable {@link NetworkConfig#attachable() attachable}
   *    .labels(Map&lt;String, String&gt; | null) // nullable {@link NetworkConfig#labels() labels}
   *    .build();
   * </pre>
   * @return A new ImmutableNetworkConfig builder
   */
  public static ImmutableNetworkConfig.Builder builder() {
    return new ImmutableNetworkConfig.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableNetworkConfig ImmutableNetworkConfig}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder implements NetworkConfig.Builder {
    private static final long INIT_BIT_NAME = 0x1L;
    private long initBits = 0x1L;

    private String name;
    private String driver;
    private Ipam ipam;
    private Map<String, String> options = new LinkedHashMap<String, String>();
    private Boolean checkDuplicate;
    private Boolean internal;
    private Boolean enableIPv6;
    private Boolean attachable;
    private Map<String, String> labels = null;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code NetworkConfig} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * Collection elements and entries will be added, not replaced.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(NetworkConfig instance) {
      Objects.requireNonNull(instance, "instance");
      this.name(instance.name());
      @Nullable String driverValue = instance.driver();
      if (driverValue != null) {
        driver(driverValue);
      }
      @Nullable Ipam ipamValue = instance.ipam();
      if (ipamValue != null) {
        ipam(ipamValue);
      }
      putAllOptions(instance.options());
      @Nullable Boolean checkDuplicateValue = instance.checkDuplicate();
      if (checkDuplicateValue != null) {
        checkDuplicate(checkDuplicateValue);
      }
      @Nullable Boolean internalValue = instance.internal();
      if (internalValue != null) {
        internal(internalValue);
      }
      @Nullable Boolean enableIPv6Value = instance.enableIPv6();
      if (enableIPv6Value != null) {
        enableIPv6(enableIPv6Value);
      }
      @Nullable Boolean attachableValue = instance.attachable();
      if (attachableValue != null) {
        attachable(attachableValue);
      }
      @Nullable Map<String, String> labelsValue = instance.labels();
      if (labelsValue != null) {
        putAllLabels(labelsValue);
      }
      return this;
    }

    /**
     * Initializes the value for the {@link NetworkConfig#name() name} attribute.
     * @param name The value for name 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Name")
    public final Builder name(String name) {
      this.name = Objects.requireNonNull(name, "name");
      initBits &= ~INIT_BIT_NAME;
      return this;
    }

    /**
     * Initializes the value for the {@link NetworkConfig#driver() driver} attribute.
     * @param driver The value for driver (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Driver")
    public final Builder driver(@Nullable String driver) {
      this.driver = driver;
      return this;
    }

    /**
     * Initializes the value for the {@link NetworkConfig#ipam() ipam} attribute.
     * @param ipam The value for ipam (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("IPAM")
    public final Builder ipam(@Nullable Ipam ipam) {
      this.ipam = ipam;
      return this;
    }

    /**
     * Put one entry to the {@link NetworkConfig#options() options} map.
     * @param key The key in the options map
     * @param value The associated value in the options map
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addOption(String key, String value) {
      this.options.put(
          Objects.requireNonNull(key, "options key"),
          Objects.requireNonNull(value, value == null ? "options value for key: " + key : null));
      return this;
    }

    /**
     * Put one entry to the {@link NetworkConfig#options() options} map. Nulls are not permitted
     * @param entry The key and value entry
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addOption(Map.Entry<String, ? extends String> entry) {
      String k = entry.getKey();
      String v = entry.getValue();
      this.options.put(
          Objects.requireNonNull(k, "options key"),
          Objects.requireNonNull(v, v == null ? "options value for key: " + k : null));
      return this;
    }

    /**
     * Sets or replaces all mappings from the specified map as entries for the {@link NetworkConfig#options() options} map. Nulls are not permitted
     * @param entries The entries that will be added to the options map
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Options")
    public final Builder options(Map<String, ? extends String> entries) {
      this.options.clear();
      return putAllOptions(entries);
    }

    /**
     * Put all mappings from the specified map as entries to {@link NetworkConfig#options() options} map. Nulls are not permitted
     * @param entries The entries that will be added to the options map
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder putAllOptions(Map<String, ? extends String> entries) {
      for (Map.Entry<String, ? extends String> e : entries.entrySet()) {
        String k = e.getKey();
        String v = e.getValue();
        this.options.put(
            Objects.requireNonNull(k, "options key"),
            Objects.requireNonNull(v, v == null ? "options value for key: " + k : null));
      }
      return this;
    }

    /**
     * Initializes the value for the {@link NetworkConfig#checkDuplicate() checkDuplicate} attribute.
     * @param checkDuplicate The value for checkDuplicate (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("CheckDuplicate")
    public final Builder checkDuplicate(@Nullable Boolean checkDuplicate) {
      this.checkDuplicate = checkDuplicate;
      return this;
    }

    /**
     * Initializes the value for the {@link NetworkConfig#internal() internal} attribute.
     * @param internal The value for internal (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Internal")
    public final Builder internal(@Nullable Boolean internal) {
      this.internal = internal;
      return this;
    }

    /**
     * Initializes the value for the {@link NetworkConfig#enableIPv6() enableIPv6} attribute.
     * @param enableIPv6 The value for enableIPv6 (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("EnableIPv6")
    public final Builder enableIPv6(@Nullable Boolean enableIPv6) {
      this.enableIPv6 = enableIPv6;
      return this;
    }

    /**
     * Initializes the value for the {@link NetworkConfig#attachable() attachable} attribute.
     * @param attachable The value for attachable (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Attachable")
    public final Builder attachable(@Nullable Boolean attachable) {
      this.attachable = attachable;
      return this;
    }

    /**
     * Put one entry to the {@link NetworkConfig#labels() labels} map.
     * @param key The key in the labels map
     * @param value The associated value in the labels map
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addLabel(String key, String value) {
      if (this.labels == null) {
        this.labels = new LinkedHashMap<String, String>();
      }
      this.labels.put(
          Objects.requireNonNull(key, "labels key"),
          Objects.requireNonNull(value, value == null ? "labels value for key: " + key : null));
      return this;
    }

    /**
     * Put one entry to the {@link NetworkConfig#labels() labels} map. Nulls are not permitted
     * @param entry The key and value entry
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addLabel(Map.Entry<String, ? extends String> entry) {
      if (this.labels == null) {
        this.labels = new LinkedHashMap<String, String>();
      }
      String k = entry.getKey();
      String v = entry.getValue();
      this.labels.put(
          Objects.requireNonNull(k, "labels key"),
          Objects.requireNonNull(v, v == null ? "labels value for key: " + k : null));
      return this;
    }

    /**
     * Sets or replaces all mappings from the specified map as entries for the {@link NetworkConfig#labels() labels} map. Nulls are not permitted as keys or values, but parameter itself can be null
     * @param entries The entries that will be added to the labels map
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Labels")
    public final Builder labels(@Nullable Map<String, ? extends String> entries) {
      if (entries == null) {
        this.labels = null;
        return this;
      }
      this.labels = new LinkedHashMap<String, String>();
      return putAllLabels(entries);
    }

    /**
     * Put all mappings from the specified map as entries to {@link NetworkConfig#labels() labels} map. Nulls are not permitted
     * @param entries The entries that will be added to the labels map
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder putAllLabels(Map<String, ? extends String> entries) {
      if (this.labels == null) {
        this.labels = new LinkedHashMap<String, String>();
      }
      for (Map.Entry<String, ? extends String> e : entries.entrySet()) {
        String k = e.getKey();
        String v = e.getValue();
        this.labels.put(
            Objects.requireNonNull(k, "labels key"),
            Objects.requireNonNull(v, v == null ? "labels value for key: " + k : null));
      }
      return this;
    }

    /**
     * Builds a new {@link ImmutableNetworkConfig ImmutableNetworkConfig}.
     * @return An immutable instance of NetworkConfig
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableNetworkConfig build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableNetworkConfig(
          name,
          driver,
          ipam,
          createUnmodifiableMap(false, false, options),
          checkDuplicate,
          internal,
          enableIPv6,
          attachable,
          labels == null ? null : createUnmodifiableMap(false, false, labels));
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_NAME) != 0) attributes.add("name");
      return "Cannot build NetworkConfig, some of required attributes are not set " + attributes;
    }
  }

  private static <K, V> Map<K, V> createUnmodifiableMap(boolean checkNulls, boolean skipNulls, Map<? extends K, ? extends V> map) {
    switch (map.size()) {
    case 0: return Collections.emptyMap();
    case 1: {
      Map.Entry<? extends K, ? extends V> e = map.entrySet().iterator().next();
      K k = e.getKey();
      V v = e.getValue();
      if (checkNulls) {
        Objects.requireNonNull(k, "key");
        Objects.requireNonNull(v, v == null ? "value for key: " + k : null);
      }
      if (skipNulls && (k == null || v == null)) {
        return Collections.emptyMap();
      }
      return Collections.singletonMap(k, v);
    }
    default: {
      Map<K, V> linkedMap = new LinkedHashMap<>(map.size() * 4 / 3 + 1);
      if (skipNulls || checkNulls) {
        for (Map.Entry<? extends K, ? extends V> e : map.entrySet()) {
          K k = e.getKey();
          V v = e.getValue();
          if (skipNulls) {
            if (k == null || v == null) continue;
          } else if (checkNulls) {
            Objects.requireNonNull(k, "key");
            Objects.requireNonNull(v, v == null ? "value for key: " + k : null);
          }
          linkedMap.put(k, v);
        }
      } else {
        linkedMap.putAll(map);
      }
      return Collections.unmodifiableMap(linkedMap);
    }
    }
  }
}
